#include "location.h"

#include "block.h"
#include "ctx.h"

#include <rtline/library/json/cast.h>

namespace {
    const TString LocationClickHouseTable = "drive_locations";
    const TString LocationHistoryClickHouseTable = "drive_location_history";

    template <class TCtx, class TLoc>
    void Append(TCtx& ctx, const TLoc& location, const TString& imei, const TString& objectId) {
        ctx.ObjectIdColumn->Append(objectId);
        ctx.ImeiColumn->Append(imei);
        ctx.TimestampColumn->Append(location.Timestamp);
        ctx.LatitudeColumn->Append(location.Latitude);
        ctx.LongitudeColumn->Append(location.Longitude);
        ctx.CourseColumn->Append(location.Course);
    }

    template <class TCtx, class TRecord>
    void Extract(TRecord& record, const TCtx& ctx, size_t i) {
        record.ObjectId = ctx.ObjectIdColumn->At(i);
        record.Imei = ctx.ImeiColumn->At(i);
        record.Timestamp = TInstant::Seconds(ctx.TimestampColumn->At(i));
        record.Latitude = ctx.LatitudeColumn->At(i);
        record.Longitude = ctx.LongitudeColumn->At(i);
        record.Course = ctx.CourseColumn->At(i);
    }
}

const TString& NDrive::GetLocationClickHouseTable() {
    return LocationClickHouseTable;
}

const TString& NDrive::GetLocationHistoryClickHouseTable() {
    return LocationHistoryClickHouseTable;
}

void Append(NClickHouse::TSimpleBlock& block, const NDrive::TLocationClickHouseRecord& record) {
    NDrive::TLocationClickHouseCtx ctx;
    Append(block, record, ctx);
}

void Append(NClickHouse::TSimpleBlock& block, const NDrive::TLocationClickHouseRecord& record, NDrive::TLocationClickHouseCtx& ctx) {
    NDrive::Initialize(ctx, block);
    Append(ctx, record, record.Imei, record.ObjectId);
    ctx.NameColumn->Append(record.Name);
    ctx.SinceColumn->Append(record.Since);
}

void Append(NClickHouse::TSimpleBlock& block, const NDrive::TLocationHistoryClickHouseRecord& record) {
    Append(block, record.Imei, record.ObjectId, record, record.Received);
}

void Append(NClickHouse::TSimpleBlock& block, const NDrive::TLocationHistoryClickHouseRecord& record, NDrive::TLocationHistoryClickHouseCtx& ctx) {
    Append(block, record.Imei, record.ObjectId, record, record.Received, ctx);
}

void Append(NClickHouse::TSimpleBlock& block, const TString& imei, const TString& objectId, const NDrive::TGpsLocation& location, TInstant received) {
    NDrive::TLocationHistoryClickHouseCtx ctx;
    Append(block, imei, objectId, location, received, ctx);
}

void Append(NClickHouse::TSimpleBlock& block, const TString& imei, const TString& objectId, const NDrive::TGpsLocation& location, TInstant received, NDrive::TLocationHistoryClickHouseCtx& ctx) {
    NDrive::Initialize(ctx, block);
    Append(ctx, location, imei, objectId);
    ctx.SpeedColumn->Append(location.Speed);
    ctx.ReceivedColumn->Append(received);
}

void Extract(const NClickHouse::TBlock& block, NDrive::TLocationClickHouseRecords& records) {
    NDrive::TLocationClickHouseCtx ctx;
    NDrive::FillColumns(ctx, block);
    auto size = ctx.ObjectIdColumn->Size();
    ForEach(ctx.Columns(), [size] (auto&& column) {
        Y_ENSURE_BT(size == column->Size(), column.GetName());
    });
    for (size_t i = 0; i < size; ++i) {
        NDrive::TLocationClickHouseRecord record;
        Extract(record, ctx, i);
        record.Name = ctx.NameColumn->At(i);
        record.Since = TInstant::Seconds(ctx.SinceColumn->At(i));
        records.push_back(std::move(record));
    }
}

void Extract(const NClickHouse::TBlock& block, NDrive::TLocationHistoryClickHouseRecords& records) {
    NDrive::TLocationHistoryClickHouseCtx ctx;
    NDrive::FillColumns(ctx, block);
    auto size = ctx.ObjectIdColumn->Size();
    ForEach(ctx.Columns(), [size] (auto&& column) {
        Y_ENSURE_BT(size == column->Size(), column.GetName());
    });
    for (size_t i = 0; i < size; ++i) {
        NDrive::TLocationHistoryClickHouseRecord record;
        Extract(record, ctx, i);
        record.Speed = ctx.SpeedColumn->At(i);
        record.Received = TInstant::Seconds(ctx.ReceivedColumn->At(i));
        record.Since = record.Timestamp;
        records.push_back(std::move(record));
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TLocationClickHouseRecord& object) {
    auto result = ToJson<NDrive::TLocation>(object);
    if (object.Imei) {
        result["imei"] = object.Imei;
    }
    if (object.ObjectId) {
        result["object_id"] = object.ObjectId;
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TLocationHistoryClickHouseRecord& object) {
    auto result = ToJson<NDrive::TGpsLocation>(object);
    if (object.Imei) {
        result["imei"] = object.Imei;
    }
    if (object.ObjectId) {
        result["object_id"] = object.ObjectId;
    }
    if (object.Received) {
        result["received"] = ToJson(object.Received);
    }
    return result;
}
