#include "sensor.h"

#include "block.h"
#include "ctx.h"

#include <rtline/library/json/cast.h>

namespace {
    const TString SensorClickHouseTable = "drive_sensors";
    const TString SensorHistoryClickHouseTable = "drive_sensor_history";

    template <class TCtx, class TRecord>
    void Append(TCtx& ctx, const TRecord& record) {
        ui64 intValue = 0;
        double floatValue = 0;
        TString stringValue = {};
        if (std::holds_alternative<ui64>(record.Value)) {
            intValue = std::get<ui64>(record.Value);
        } else if (std::holds_alternative<double>(record.Value)) {
            floatValue = std::get<double>(record.Value);
        } else {
            stringValue = record.template ConvertTo<TString>();
        }
        ctx.ObjectIdColumn->Append(record.ObjectId);
        ctx.ImeiColumn->Append(record.Imei);
        ctx.IdColumn->Append(record.Id);
        ctx.SubIdColumn->Append(record.SubId);
        ctx.TimestampColumn->Append(record.Timestamp);
        ctx.ValueIntColumn->Append(intValue);
        ctx.ValueFloatColumn->Append(floatValue);
        ctx.ValueStringColumn->Append(stringValue);
    }

    template <class TCtx, class TRecord>
    void Extract(TRecord& record, const TCtx& ctx, size_t i) {
        record.ObjectId = ctx.ObjectIdColumn->At(i);
        record.Imei = ctx.ImeiColumn->At(i);
        record.Id = ctx.IdColumn->At(i);
        record.SubId = ctx.SubIdColumn->At(i);
        record.Timestamp = TInstant::Seconds(ctx.TimestampColumn->At(i));
        record.Since = record.Timestamp;
        if (const auto& valueString = ctx.ValueStringColumn->At(i)) {
            record.Value = NDrive::SensorValueFromString(valueString, record);
        } else if (const auto valueFloat = ctx.ValueFloatColumn->At(i); std::abs(valueFloat) > 0.00001) {
            record.Value = valueFloat;
        } else {
            auto valueInt = ctx.ValueIntColumn->At(i);
            record.Value = valueInt;
        }
    }
}

const TString& NDrive::GetSensorClickHouseTable() {
    return SensorClickHouseTable;
}

const TString& NDrive::GetSensorHistoryClickHouseTable() {
    return SensorHistoryClickHouseTable;
}

void Append(NClickHouse::TSimpleBlock& block, const NDrive::TSensorClickHouseRecord& record) {
    NDrive::TSensorClickHouseCtx ctx;
    Append(block, record, ctx);
}

void Append(NClickHouse::TSimpleBlock& block, const NDrive::TSensorClickHouseRecord& record, NDrive::TSensorClickHouseCtx& ctx) {
    NDrive::Initialize(ctx, block);
    Append(ctx, record);
    ctx.SinceColumn->Append(record.Since);
}

void Append(NClickHouse::TSimpleBlock& block, const NDrive::TSensorHistoryClickHouseRecord& record) {
    NDrive::TSensorHistoryClickHouseCtx ctx;
    Append(block, record, ctx);
}

void Append(NClickHouse::TSimpleBlock& block, const NDrive::TSensorHistoryClickHouseRecord& record, NDrive::TSensorHistoryClickHouseCtx& ctx) {
    const auto& sensor = record;
    NDrive::Initialize(ctx, block);
    Append(ctx, record);
    ctx.ReceivedColumn->Append(sensor.Received);
    ctx.GroupstampColumn->Append(sensor.Groupstamp);
}

void Extract(const NClickHouse::TBlock& block, NDrive::TSensorClickHouseRecords& records) {
    NDrive::TSensorClickHouseCtx ctx;
    NDrive::FillColumns(ctx, block);
    auto size = ctx.ObjectIdColumn->Size();
    Y_ENSURE_BT(size == ctx.ImeiColumn->Size());
    Y_ENSURE_BT(size == ctx.IdColumn->Size());
    Y_ENSURE_BT(size == ctx.SubIdColumn->Size());
    Y_ENSURE_BT(size == ctx.SinceColumn->Size());
    Y_ENSURE_BT(size == ctx.TimestampColumn->Size());
    Y_ENSURE_BT(size == ctx.ValueIntColumn->Size());
    Y_ENSURE_BT(size == ctx.ValueFloatColumn->Size());
    Y_ENSURE_BT(size == ctx.ValueStringColumn->Size());
    for (size_t i = 0; i < size; ++i) {
        NDrive::TSensorClickHouseRecord record;
        Extract(record, ctx, i);
        record.Since = TInstant::Seconds(ctx.SinceColumn->At(i));
        records.push_back(std::move(record));
    }
}

void Extract(const NClickHouse::TBlock& block, NDrive::TSensorHistoryClickHouseRecords& records) {
    NDrive::TSensorHistoryClickHouseCtx ctx;
    NDrive::FillColumns(ctx, block);
    auto size = ctx.ObjectIdColumn->Size();
    Y_ENSURE_BT(size == ctx.ImeiColumn->Size());
    Y_ENSURE_BT(size == ctx.IdColumn->Size());
    Y_ENSURE_BT(size == ctx.SubIdColumn->Size());
    Y_ENSURE_BT(size == ctx.TimestampColumn->Size());
    Y_ENSURE_BT(size == ctx.ValueIntColumn->Size());
    Y_ENSURE_BT(size == ctx.ValueFloatColumn->Size());
    Y_ENSURE_BT(size == ctx.ValueStringColumn->Size());
    if (ctx.ReceivedColumn) {
        Y_ENSURE_BT(size == ctx.ReceivedColumn->Size());
    }
    if (ctx.GroupstampColumn) {
        Y_ENSURE_BT(size == ctx.GroupstampColumn->Size());
    }
    for (size_t i = 0; i < size; ++i) {
        NDrive::TSensorHistoryClickHouseRecord record;
        Extract(record, ctx, i);
        if (ctx.ReceivedColumn) {
            record.Received = TInstant::Seconds(ctx.ReceivedColumn->At(i));
        }
        if (ctx.GroupstampColumn) {
            record.Groupstamp = TInstant::Seconds(ctx.GroupstampColumn->At(i));
        }
        records.push_back(std::move(record));
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TSensorHistoryClickHouseRecord& object) {
    auto result = ToJson<NDrive::TSensor>(object);
    if (object.Imei) {
        result["imei"] = object.Imei;
    }
    if (object.ObjectId) {
        result["object_id"] = object.ObjectId;
    }
    if (object.Received) {
        result["received"] = ToJson(object.Received);
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TSensorClickHouseRecord& object) {
    auto result = ToJson<NDrive::TSensor>(object);
    if (object.Imei) {
        result["imei"] = object.Imei;
    }
    if (object.ObjectId) {
        result["object_id"] = object.ObjectId;
    }
    return result;
}
