#include "block.h"
#include "ctx.h"
#include "sensor.h"

#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(ClickHouseSensor) {
    TString SampleImei = "19643932284730";
    TString SampleObjectId = "75abe596-ff27-4c12-a3bd-5cdad1eed104";

    ui64 IntValue = 42;
    double FloatValue = 142.42;
    TString StringValue = "VEGA MT";

    Y_UNIT_TEST(Columns) {
        NClickHouse::TBlock block;
        NDrive::TSensorClickHouseCtx ctx;
        NDrive::Initialize(ctx, block);

        UNIT_ASSERT_VALUES_EQUAL(block.GetRowCount(), 0);
        UNIT_ASSERT_VALUES_EQUAL(block.GetColumnCount(), 9);

        NClickHouse::TBlock::TIterator iterator(block);
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "imei");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "id");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "subid");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "value_int");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "value_float");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "value_string");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "since");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "timestamp");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "object_id");
        iterator.Next();
        UNIT_ASSERT(!iterator.IsValid());
    }

    Y_UNIT_TEST(HistoryColumns) {
        NClickHouse::TBlock block;
        NDrive::TSensorHistoryClickHouseCtx ctx;
        NDrive::Initialize(ctx, block);

        UNIT_ASSERT_VALUES_EQUAL(block.GetRowCount(), 0);
        UNIT_ASSERT_VALUES_EQUAL(block.GetColumnCount(), 10);

        NClickHouse::TBlock::TIterator iterator(block);
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "object_id");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "imei");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "id");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "subid");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "timestamp");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "value_int");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "value_float");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "value_string");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "groupstamp");
        iterator.Next();
        UNIT_ASSERT(iterator.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(iterator.Name(), "received");
        iterator.Next();
        UNIT_ASSERT(!iterator.IsValid());
    }

    template <class TCtx, class TRecord>
    void TestInputOutput() {
        NClickHouse::TSimpleBlock simpleBlock;
        TCtx ctx;
        TRecord record;
        record.Imei = SampleImei;
        record.ObjectId = SampleObjectId;
        record.Id = 41;
        record.SubId = 42;
        record.Timestamp = Now();
        {
            record.Value = IntValue;
            Append(simpleBlock, record);
        }
        {
            record.Value = FloatValue;
            Append(simpleBlock, record);
        }
        {
            record.Value = StringValue;
            Append(simpleBlock, record);
        }
        NClickHouse::TBlock block = simpleBlock;
        UNIT_ASSERT_VALUES_EQUAL(block.GetRowCount(), 3);

        size_t ints = 0;
        size_t floats = 0;
        size_t strings = 0;
        TVector<TRecord> records;
        Extract(block, records);
        for (auto&& r : records) {
            UNIT_ASSERT_VALUES_EQUAL(r.Imei, record.Imei);
            UNIT_ASSERT_VALUES_EQUAL(r.ObjectId, record.ObjectId);
            UNIT_ASSERT_VALUES_EQUAL(r.Id, record.Id);
            UNIT_ASSERT_VALUES_EQUAL(r.SubId, record.SubId);
            UNIT_ASSERT_VALUES_EQUAL(r.Timestamp, record.Timestamp);
            if (std::holds_alternative<ui64>(r.Value)) {
                UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(r.Value), IntValue);
                ints += 1;
            } else if (std::holds_alternative<double>(r.Value)) {
                UNIT_ASSERT_DOUBLES_EQUAL(std::get<double>(r.Value), FloatValue, 0.00001);
                floats += 1;
            } else {
                UNIT_ASSERT_VALUES_EQUAL(r.template ConvertTo<TString>(), StringValue);
                strings += 1;
            }
        }
        UNIT_ASSERT(ints);
        UNIT_ASSERT(floats);
        UNIT_ASSERT(strings);
    }

    Y_UNIT_TEST(InputOutput) {
        TestInputOutput<NDrive::TSensorClickHouseCtx, NDrive::TSensorClickHouseRecord>();
    }

    Y_UNIT_TEST(HistoryInputOutput) {
        TestInputOutput<NDrive::TSensorHistoryClickHouseCtx, NDrive::TSensorHistoryClickHouseRecord>();
    }
}
