#include "factors.h"

#include "blackbox.h"

#include <drive/telematics/protocol/vega.h>

NDrive::TThresholdDurationCounterResult NDrive::Calc(
    const TThresholdDurationCounterCalcer& calcer,
    const TTelematicsHistory& history,
    const TThresholdDurationCounterOptions& options,
    TFeaturesCalculationContext* context
) {
    TFeaturesCalculationContext local;
    if (!context) {
        context = &local;
    }

    Y_ASSERT(context);
    if (!context->Locations) {
        context->Locations = history.GetLocations().GetValues();
    }

    Y_ASSERT(context->Locations);
    const auto& locations = *context->Locations;
    Y_ASSERT(std::is_sorted(locations.begin(), locations.end(), [](const TGpsLocation& left, const TGpsLocation& right) {
        return left.Timestamp < right.Timestamp;
    }));

    TMaybe<std::pair<TInstant, TInstant>> currentRange;
    TThresholdDurationCounterResult result;
    for (auto&& location : locations) {
        auto optionalValue = calcer(location, history, *context);
        if (!optionalValue) {
            context->PreviousLocation = location;
            continue;
        }
        auto value = *optionalValue;
        if (options.Brake) {
            value *= -1;
        }

        auto timestamp = location.Timestamp.Get();
        if (options.Debug) {
            result.Values.emplace_back(timestamp, value);
        }

        bool speedOk =
            currentRange ||
            !options.SpeedRange ||
            !context->PreviousLocation ||
            (options.SpeedRange->first <= context->PreviousLocation->Speed && context->PreviousLocation->Speed < options.SpeedRange->second);
        if (speedOk && value > options.MinValue) {
            if (!currentRange) {
                currentRange = std::make_pair(context->PreviousLocation.GetOrElse(location).Timestamp, timestamp);
            }
            Y_ASSERT(currentRange);
            Y_ASSERT(currentRange->second <= timestamp);
            Y_ASSERT(currentRange->second >= currentRange->first);
            currentRange->second = timestamp;
        } else {
            if (currentRange && (currentRange->second - currentRange->first > options.Threshold)) {
                result.Ranges.push_back(*currentRange);
            }
            currentRange.Clear();
        }

        context->PreviousLocation = location;
    }
    if (currentRange && (currentRange->second - currentRange->first > options.Threshold)) {
        result.Ranges.push_back(*currentRange);
    }

    return result;
}

namespace {
    NDrive::TThresholdDurationCounterFactors DefaultFactors = {
        { NDrive::MakeSensorValueCalcer(CAN_ENGINE_RPM), { TDuration::Seconds(5), Nothing(), 3900, false, false } },
        { NDrive::MakeGpsAccelerationCalcer(),           { TDuration::Seconds(2), Nothing(), 12, false, true} },
        { NDrive::MakeGpsAccelerationCalcer(),           { TDuration::Seconds(6), Nothing(), 5, false, false} },
    };
}

const NDrive::TThresholdDurationCounterFactors& NDrive::GetDefaultThresholdDurationCounterFactors() {
    return DefaultFactors;
}

NDrive::TThresholdDurationCounterCalcer NDrive::MakeGpsAccelerationCalcer() {
    return [] (const TGpsLocation& location, const TTelematicsHistory& history, TFeaturesCalculationContext& context) -> TMaybe<double> {
        Y_UNUSED(history);
        if (!context.PreviousLocation) {
            return {};
        }
        const auto& previousLocation = *context.PreviousLocation;
        if (previousLocation.Timestamp == location.Timestamp) {
            return {};
        }
        Y_ASSERT(previousLocation.Timestamp < location.Timestamp);
        auto speedDiff = 1.0 * location.Speed - 1.0 * previousLocation.Speed;
        auto timestampDiff = location.Timestamp - previousLocation.Timestamp;
        return speedDiff / timestampDiff.Seconds();
    };
}

NDrive::TThresholdDurationCounterCalcer NDrive::MakeSensorValueCalcer(TSensorId id) {
    return [id] (const TGpsLocation& location, const TTelematicsHistory& history, TFeaturesCalculationContext& context) -> TMaybe<double> {
        auto& sensor = context.Sensor;
        auto& sensorValue = context.SensorValue;
        auto timestamp = location.Timestamp.Get();
        if (sensor) {
            Y_ASSERT(sensor->Since <= timestamp);
        }
        if (!sensor || *sensor != id || sensor->Timestamp < timestamp) {
            sensor = history.GetSensors().Get(id.Id, id.SubId, timestamp);
            sensorValue = {};
        }
        if (!sensor) {
            return {};
        }

        Y_ASSERT(sensor);
        Y_ASSERT(*sensor == id);
        Y_ASSERT(sensor->Since <= timestamp);

        if (!sensorValue) {
            sensorValue = sensor->TryConvertTo<double>();
        }
        if (!sensorValue) {
            context.BadSensors.push_back(*sensor);
            return {};
        }
        return sensorValue;
    };
}
