#include "blackbox.h"
#include "factors.h"

#include <drive/library/cpp/yt/node/cast.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/json/cast.h>

#include <util/stream/file.h>
#include <util/string/vector.h>
#include <util/system/env.h>

TAtomicSharedPtr<IOutputStream> GetOutputStream() {
    auto filename = GetEnv("DumpPath");
    if (!filename) {
        return nullptr;
    }
    return MakeAtomicShared<TOFStream>(filename);
}

Y_UNIT_TEST_SUITE(TelematicsFeatures) {
    Y_UNIT_TEST(Simple) {
        NDrive::TTelematicsHistory history;

        TVector<ui64> factors;
        TVector<TInstant> timestamps;
        TSet<TInstant> starts;

        auto output = GetOutputStream();
        auto previousTimestamp = TTimestamp();
        auto dumpPath = TFsPath(GetWorkPath()) / "ytdump" / "867962042708449.json";
        TFileInput input(dumpPath);
        TString line;
        while (input.ReadLine(line)) {
            auto json = NJson::ReadJsonFastTree(line);
            auto yson = NYT::ToNode(json);
            auto records = NYT::FromNode<NDrive::TBlackboxRecords>(yson["data"]);
            if (records.Values.empty()) {
                continue;
            }

            auto timestamp = records.Values.back().Location->Timestamp;
            //UNIT_ASSERT(previousTimestamp <= timestamp);
            history.Add(std::move(records));

            factors.clear();
            timestamps.clear();
            for (auto&& [calcer, options] : NDrive::GetDefaultThresholdDurationCounterFactors()) {
                auto value = NDrive::Calc(calcer, history, options);
                factors.push_back(value.Ranges.size());
                UNIT_ASSERT(std::is_sorted(value.Ranges.begin(), value.Ranges.end()));
                for (auto&& range : value.Ranges) {
                    UNIT_ASSERT(range.second - range.first >= options.Threshold);
                }
                if (!value.Ranges.empty()) {
                    auto start = value.Ranges.back().first;
                    auto location = history.GetLocations().GetValueAt(start);
                    UNIT_ASSERT(location);
                    timestamps.push_back(value.Ranges.back().first);
                }
                if (options.Debug) {
                    Cerr << timestamp << ": " << NJson::ToJson(value.Values).GetStringRobust() << Endl;
                }
            }
            for (auto&& ts : timestamps) {
                if (!starts.insert(ts).second) {
                    continue;
                }
                if (output) {
                    *output << timestamp << "@" << ts.Seconds() << ": " << JoinVectorIntoString(factors, "\n") << Endl;
                }
            }
            previousTimestamp = timestamp;
        }
    }
}
