#include "ble_session_key_watcher.h"

#include "logging.h"
#include "scenarios.h"
#include "server.h"

#include <drive/telematics/protocol/vega.h>

#include <rtline/library/json/builder.h>

NDrive::TBleSessionKeyWatcher::TBleSessionKeyWatcher(const NDrive::TTelematicsServer& server, const TString& imei, TInstant now, TDuration interval)
    : TGlobalScheduler::TScheduledItem<TBleSessionKeyWatcher>(server.Name(),
                                    TStringBuilder() << "ble_session_watcher:" << imei << ":" << static_cast<const void*>(this), now)
    , Server(server)
    , IMEI(imei)
    , Interval(interval)
{
}

void NDrive::TBleSessionKeyWatcher::Process(void* /*threadSpecificResource*/) {
    auto cleanup = Hold(this);
    auto connection = Server.GetConnection(IMEI);
    if (!connection) {
        return;
    }
    const auto& sensors = connection->GetSensorsCache();
    auto sessionKey = sensors.Get(NVega::BleSessionKey.Id);
    if (!sessionKey) {
        return;
    }
    const auto& key = std::get<TBuffer>(sessionKey->Value);
    if (static_cast<size_t>(std::count(key.Begin(), key.End(), '\0')) == key.Size()) {
        TTelematicsLog::Log(TTelematicsLog::EMiscEvent::Monitoring, connection.Get(), "ble_session_key_watcher", NJson::JSON_MAP);
        TCommandOptions options;
        auto id = NProtocol::GenerateId(IMEI);
        auto handler = CreateCommand(id, NVega::ECommandCode::SCENARIO_BLE_RESET, {}, options);
        connection->AddHandler(std::move(handler));
    }
}

THolder<TGlobalScheduler::IScheduledItem> NDrive::TBleSessionKeyWatcher::GetNextScheduledItem(TInstant now) const {
    auto connection = Server.GetConnection(IMEI);
    if (!connection || !connection->Alive()) {
        return nullptr;
    }
    return MakeHolder<TBleSessionKeyWatcher>(Server, IMEI, now + Interval, Interval);
}
