#include "can_scanner.h"

#include "handlers.h"
#include "server.h"

#include <drive/telematics/server/common/settings.h>

NDrive::TCanScannerScheduler::TCanScannerScheduler(const NDrive::TTelematicsServer& server, const TString& imei, TInstant now, TDuration interval)
    : TGlobalScheduler::TScheduledItem<TCanScannerScheduler>(server.Name(), "can_scanner_scheduler:" + imei, now)
    , Server(server)
    , IMEI(imei)
    , Interval(interval)
{
}

void NDrive::TCanScannerScheduler::Process(void* /*threadSpecificResource*/) {
    auto cleanup = Hold(this);
    auto storage = Server.GetStorage();
    if (!storage) {
        ERROR_LOG << GetId() << ": storage is missing" << Endl;
        return;
    }

    auto connection = Server.GetConnection(IMEI);
    if (!connection || !connection->Alive()) {
        return;
    }

    const auto& settings = Server.GetDynamicSettings();
    const auto imeis = settings.Get<TString>("can_scanner:imeis").GetOrElse("");
    if (!imeis.Contains(IMEI)) {
        DEBUG_LOG << GetId() << ": CanScanner is not enabled" << Endl;
        return;
    }

    auto duration = settings.Get<TDuration>("can_scanner:duration").GetOrElse(TDuration::Seconds(150));
    auto value = settings.Get<ui32>("can_scanner:value").GetOrElse(0);
    auto mask = settings.Get<ui32>("can_scanner:mask").GetOrElse(0);
    auto now = Now();

    NDrive::NVega::TCommandRequest::TCustomObdForwardConfig config;
    config.Duration = duration.Seconds();
    config.Value = value;
    config.Mask = mask;
    NDrive::NVega::TArgument argument;
    argument.Set(config);

    auto handler = MakeIntrusive<TCanSetupTask>(
        ToString(now.MicroSeconds()) + '-' + GetId(),
        argument
    );
    INFO_LOG << GetId() << ": scheduling CanScanner using " << handler->GetId() << Endl;
    connection->AddHandler(handler);
}

THolder<TGlobalScheduler::IScheduledItem> NDrive::TCanScannerScheduler::GetNextScheduledItem(TInstant now) const {
    auto connection = Server.GetConnection(IMEI);
    if (!connection || !connection->Alive()) {
        return nullptr;
    }
    return MakeHolder<TCanScannerScheduler>(Server, IMEI, now + Interval, Interval);
}
