#include "client.h"

#include "handlers.h"
#include "logging.h"
#include "scenarios.h"
#include "server.h"

#include <drive/telematics/server/common/signals.h>

#include <drive/telematics/common/file.h>
#include <drive/telematics/common/handler.h>
#include <drive/telematics/protocol/actions.h>
#include <drive/telematics/protocol/vega.h>

#include <drive/library/cpp/network/data/data.h>
#include <drive/library/cpp/searchserver/replier.h>
#include <drive/library/cpp/threading/container.h>
#include <drive/library/cpp/threading/future.h>

#include <kernel/reqid/reqid.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>

#include <util/generic/adaptor.h>
#include <util/random/random.h>
#include <util/stream/buffer.h>
#include <util/system/hostname.h>

namespace NDrive {
    class TReplier: public IHttpReplier {
    private:
        struct TCtx {
            TString IMEI;
            TString ReqId;
            NTvmAuth::TTvmId TvmClientId = 0;
        };

    public:
        TReplier(TTelematicsServer* server, IReplyContext::TPtr context)
            : IHttpReplier(context, nullptr)
            , Cgi(Context->GetCgiParameters())
            , Server(server)
        {
            Ctx.IMEI = Cgi.Get("imei");
            Ctx.ReqId = NUtil::GetReqId(Context->GetRequestData(), Context->GetCgiParameters());
            CHECK_WITH_LOG(Server);
            const auto& post = Context->GetBuf();
            Post = NJson::ToJson(NJson::JsonString({ post.AsCharPtr(), post.Size() }));
            if (!Ctx.ReqId) {
                Ctx.ReqId = ReqIdGenerate("TELE");
            }
            NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::Access, Context.Get(), Ctx.IMEI, NJson::TMapBuilder
                ("cgi", Cgi.Print())
                ("reqid", Ctx.ReqId)
                ("post", Post)
            );
        }

    protected:
        void DoSearchAndReply() override {
            Y_ASSERT(Context);
            const TStringBuf handle = Context->GetUri();
            if (handle == "/connection/command/"sv) {
                ConnectionCommand();
                return;
            }
            if (handle == "/connection/drop/"sv) {
                ConnectionDrop();
                return;
            }
            if (handle == "/connection/list/"sv) {
                ConnectionList();
                return;
            }
            if (handle == "/connection/log/"sv) {
                ConnectionLog();
                return;
            }
            if (handle == "/connection/ping/"sv) {
                ConnectionPing();
                return;
            }
            if (handle == "/connection/locations/"sv) {
                ConnectionLocations();
                return;
            }
            if (handle == "/connection/sensors/"sv) {
                ConnectionSensors();
                return;
            }
            if (handle == "/connection/download/"sv) {
                ConnectionDownload();
                return;
            }
            if (handle == "/connection/upload/"sv) {
                ConnectionUpload();
                return;
            }
            if (handle == "/connection/view/"sv) {
                ConnectionView();
                return;
            }
            if (handle == "/task/view/"sv) {
                TaskView();
                return;
            }
            if (handle == "/task/wait/"sv) {
                TaskWait();
                return;
            }
            throw TCodedException(HTTP_BAD_REQUEST) << "unknown handle: " << handle;
        }
        IThreadPool* DoSelectHandler() override {
            return nullptr;
        }
        TDuration GetDefaultTimeout() const override {
            return TDuration::Seconds(100);
        }
        void MakeErrorPage(ui32 code, const TString& error) override {
            NJson::TJsonValue result;
            result["errors"].AppendValue(error);
            SendReply(std::move(result), code);
        }
        void OnRequestExpired() override {
        }

    private:
        void ConnectionCommand() {
            CheckAuthorization();
            const TString& imei = GetIMEI();
            const auto post = GetPost();
            const auto externalId = post["external_id"].GetString();
            const auto retries = post["retries"];
            const auto commandTimeout = NJson::FromJson<TMaybe<TDuration>>(post["timeout"]).GetOrElse(GetDefaultTimeout());
            const auto waitTimeout = Cgi.Has("wait_timeout") ? TDuration::MicroSeconds(FromString<ui64>(Cgi.Get("wait_timeout"))) : TDuration::Zero();

            auto expectedCommand = NDrive::ParseCommand(post);
            if (!expectedCommand) {
                throw TCodedException(HTTP_BAD_REQUEST) << expectedCommand.GetError().what();
            }

            TCommandOptions options;
            options.Retries = NJson::FromJson<TMaybe<ui32>>(retries).GetOrElse(3);
            options.Timeout = commandTimeout;
            auto id = externalId ? externalId : NProtocol::GenerateId(imei);
            auto deadline = Context->GetRequestStartTime() + commandTimeout;
            auto code = expectedCommand->Code;
            auto argument = expectedCommand->Argument;
            auto createHandler = [
                id,
                deadline,
                code,
                argument,
                options
            ](NDrive::TTelematicsConnectionPtr connection) {
                auto handler = connection->CreateCommand(id, code, argument, options);
                Y_ASSERT(handler->GetId() == id);
                handler->SetDeadline(deadline);
                return handler;
            };
            AddTask(std::move(createHandler), imei, waitTimeout);
        }
        void ConnectionDrop() {
            CheckAuthorization();
            const TString& imei = GetIMEI();
            const auto connection = GetConnection(imei);
            CHECK_WITH_LOG(connection);
            connection->Drop();

            NJson::TJsonValue result;
            result["connections"] = NJson::JSON_ARRAY;
            SendReply(std::move(result));
        }
        void ConnectionList() {
            const auto connections = GetConnections();
            NJson::TJsonValue result;
            NJson::TJsonValue& cs = result.InsertValue("connections", NJson::JSON_ARRAY);
            for (auto&& connection : connections) {
                cs.AppendValue(PrintConnection(connection));
            }
            result.InsertValue("type", "ConnectionList");
            SendReply(std::move(result));
        }
        void ConnectionLog() {
            const TString& imei = GetIMEI();
            const auto count = FromStringWithDefault<ui32>(Cgi.Get("count"), 64);
            const auto connection = GetConnection(imei);
            CHECK_WITH_LOG(connection);

            auto records = connection->GetLogRecords();
            std::sort(records.begin(), records.end());

            NJson::TJsonValue result;
            NJson::TJsonValue& rs = result.InsertValue("records", NJson::JSON_ARRAY);
            NDrive::NVega::TLogRecord previous;
            for (auto&& record : Reversed(records)) {
                if (record == previous) {
                    continue;
                }
                if (rs.GetArraySafe().size() >= count) {
                    break;
                }

                auto& report = rs.AppendValue(NJson::JSON_MAP);
                report["time"] = record.Timestamp.ToString();
                report["message"] = record.Message;
                previous = record;
            }
            SendReply(std::move(result));
        }
        void ConnectionPing() {
            CheckAuthorization();
            const TString& imei = GetIMEI();
            const auto connection = GetConnection(imei);
            CHECK_WITH_LOG(connection);

            auto handler = MakeIntrusive<NDrive::TPingTask>(GenerateTaskId(imei, "ping"));
            connection->AddHandler(handler);
            handler->Wait(Context->GetRequestDeadline());

            SendReply(handler->Serialize());
        }
        void ConnectionLocations() {
            const TString& imei = GetIMEI();
            const auto connection = GetConnection(imei);
            CHECK_WITH_LOG(connection);

            NJson::TJsonValue result(NJson::JSON_ARRAY);
            if (auto cache = connection->GetLocations()) {
                auto locations = cache->GetLocations();
                for (auto&& location : locations) {
                    result.AppendValue(location.ToJson());
                }
            }
            SendReply(std::move(result));
        }
        void ConnectionSensors() {
            const TString& imei = GetIMEI();
            const auto connection = GetConnection(imei);
            CHECK_WITH_LOG(connection);

            NJson::TJsonValue result(NJson::JSON_ARRAY);
            connection->GetSensorsCache().Fill(result);
            SendReply(std::move(result));
        }
        void ConnectionView() {
            const TString& imei = GetIMEI();
            const auto connection = GetConnection(imei);
            CHECK_WITH_LOG(connection);
            const auto printed = PrintConnection(connection);

            NJson::TJsonValue result;
            result["connections"].AppendValue(printed);
            SendReply(std::move(result));
        }
        void TaskView() {
            const TString& id = GetTaskId();
            const auto task = GetTask(id);
            CHECK_WITH_LOG(task);

            SendReply(task->Serialize());
        }
        void TaskWait() {
            const TString& id = GetTaskId();
            const auto waitTimeout = Cgi.Has("wait_timeout") ? TDuration::MicroSeconds(FromString<ui64>(Cgi.Get("wait_timeout"))) : TDuration::Zero();

            auto contextContainer = MakeThreadSafeContainer(MakeCopy(Context));
            auto callback = [contextContainer, ctx = Ctx](const TCommonTask& task) {
                auto context = contextContainer->Release().GetOrElse(nullptr);
                if (context) {
                    SendReply(context, ctx, task.Serialize());
                }
            };
            auto deadline = Context->GetRequestDeadline();

            if (waitTimeout) {
                auto task = Yensured(Server)->WaitTask(id);
                auto waitDeadline = Context->GetRequestStartTime() + waitTimeout;
                auto waitCallback = [callback = std::move(callback), context = Context, ctx = Ctx, deadline](const NDrive::TTelematicsServer::TTaskFuture& t) {
                    try {
                        if (t.HasException()) {
                            t.GetValue();
                        }
                        if (t.HasValue()) {
                            auto task = t.GetValue();
                            Yensured(task)->AddCallback(std::move(callback), /*time=*/TInstant::Zero(), deadline);
                        } else {
                            SendReply(context, ctx, "task is still absent", HTTP_NOT_FOUND);
                        }
                    } catch (const std::exception& e) {
                        SendReply(context, ctx, FormatExc(e), HTTP_INTERNAL_SERVER_ERROR);
                    }
                };
                NThreading::Subscribe(task, std::move(waitCallback), waitDeadline);
            } else {
                auto task = GetTask(id);
                Yensured(task)->AddCallback(std::move(callback), /*time=*/TInstant::Zero(), deadline);
            }
        }
        void ConnectionDownload() {
            CheckAuthorization();
            const auto& imei = GetIMEI();
            const auto connection = GetConnection(imei);
            CHECK_WITH_LOG(connection);
            const auto filename = Cgi.Get("filename");
            const auto& externalId = Cgi.Get("external_id");
            const auto downloadTimeout = Cgi.Has("timeout") ? TDuration::MicroSeconds(FromString<ui64>(Cgi.Get("timeout"))) : GetDefaultTimeout();
            const auto waitTimeout = Cgi.Has("wait_timeout") ? TDuration::MicroSeconds(FromString<ui64>(Cgi.Get("wait_timeout"))) : TDuration::Zero();
            auto deadline = Context->GetRequestStartTime() + downloadTimeout;
            auto createHandler = [
                externalId,
                deadline,
                filename
            ](NDrive::TTelematicsConnectionPtr connection) {
                Y_UNUSED(connection);
                auto handler = MakeAtomicShared<NDrive::NVega::TGetFileHandler>(filename);
                auto wrapper = MakeIntrusive<NDrive::TDownloadFileHandler>(externalId, handler);
                wrapper->SetDeadline(deadline);
                return wrapper;
            };
            AddTask(std::move(createHandler), imei, waitTimeout);
        }
        void ConnectionUpload() {
            CheckAuthorization();
            const auto& imei = GetIMEI();
            const auto& filename = Cgi.Get("filename");
            const auto& externalId = Cgi.Get("external_id");
            const TDuration uploadTimeout = Cgi.Has("timeout") ? TDuration::MicroSeconds(FromString<ui64>(Cgi.Get("timeout"))) : GetDefaultTimeout();
            const auto waitTimeout = Cgi.Has("wait_timeout") ? TDuration::MicroSeconds(FromString<ui64>(Cgi.Get("wait_timeout"))) : TDuration::Zero();
            auto deadline = Context->GetRequestStartTime() + uploadTimeout;
            auto createHandler = [
                externalId,
                deadline,
                fileContents = GetPostRaw(),
                filename
            ](NDrive::TTelematicsConnectionPtr connection) {
                auto fileContentsCopy = fileContents;
                auto handler = MakeAtomicShared<NDrive::NVega::TChunkedFileHandler>(
                    filename,
                    std::move(fileContentsCopy),
                    std::min<ui16>(connection->GetInputBufferSize() - 24, Max<ui16>())
                );

                auto wrapper = MakeIntrusive<NDrive::TUploadFileHandler>(externalId, handler);
                wrapper->SetDeadline(deadline);
                return wrapper;
            };
            AddTask(std::move(createHandler), imei, waitTimeout);
        }
        const TString& GetIMEI() const {
            const TString& imei = Ctx.IMEI;
            if (!imei) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter 'imei' is required";
            }
            return imei;
        }
        const TString& GetTaskId() const {
            const TString& id = Cgi.Get("id");
            if (!id) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter 'id' is required";
            }
            return id;
        }
        NJson::TJsonValue GetPost() const {
            const TBlob& buf = Context->GetBuf();
            const TStringBuf str(buf.AsCharPtr(), buf.Size());
            NJson::TJsonValue result;
            if (!NJson::ReadJsonFastTree(str, &result)) {
                throw TCodedException(HTTP_BAD_REQUEST) << "cannot parse POST data as json";
            }
            return result;
        }
        TBuffer GetPostRaw() const {
            const TBlob& buf = Context->GetBuf();
            return {buf.AsCharPtr(), buf.Size()};
        }
        TTelematicsConnectionPtr GetConnection(const TString& imei) const {
            Y_ASSERT(Server);
            auto connection = Server->GetConnection(imei);
            if (!connection) {
                throw TCodedException(HTTP_NOT_FOUND) << "IMEI " << imei << " is not found";
            }
            if (!connection->Alive()) {
                throw TCodedException(HTTP_GONE) << "IMEI " << imei << " is dead";
            }
            return connection;
        }
        TTelematicsServer::TConnectionPtrs GetConnections() const {
            Y_ASSERT(Server);
            return Server->GetConnections();
        }
        TTaskPtr GetTask(const TString& id) {
            Y_ASSERT(Server);
            auto task = Server->GetTask(id);
            if (!task) {
                throw TCodedException(HTTP_NOT_FOUND) << "task " << id << " is not found";
            }
            return task;
        }
        TString GenerateTaskId(const TString& imei, const TString& type) const {
            Y_ENSURE(type, "type cannot be empty");
            TStringStream ss;
            ss  << Context->GetRequestStartTime().Seconds() << '-'
                << RandomNumber<ui32>() << '-'
                << imei << '-'
                << HostName() << '-'
                << type;
            return ss.Str();
        }

        NJson::TJsonValue PrintConnection(TTelematicsConnectionPtr connection) const {
            CHECK_WITH_LOG(connection);
            CHECK_WITH_LOG(Server);
            NJson::TJsonValue result;
            result["alive"] = connection->Alive();
            result["created"] = NJson::ToJson(connection->GetCreatedTime());
            result["blackbox"] = NJson::ToJson(connection->GetBlackboxTime());
            result["heartbeat"] = NJson::ToJson(connection->GetHeartbeatTime());
            result["imei"] = connection->GetIMEI();
            result["client"] = connection->GetRemoteAddr();
            result["server"] = connection->GetServerAddr();
            return result;
        }

        void CheckAuthorization() {
            const NTvmAuth::TTvmClient* tvm = Server->GetTvmClient();
            if (tvm) {
                TStringBuf serviceTicketHeader = Context->GetRequestData().HeaderInOrEmpty("X-Ya-Service-Ticket");
                if (serviceTicketHeader) {
                    NTvmAuth::TCheckedServiceTicket ticket = tvm->CheckServiceTicket(serviceTicketHeader);
                    Y_ENSURE_EX(ticket, TCodedException(HTTP_UNAUTHORIZED) << "TVM service ticket is invalid");
                    NTvmAuth::ETicketStatus status = ticket.GetStatus();
                    Y_ENSURE_EX(status == NTvmAuth::ETicketStatus::Ok, TCodedException(HTTP_UNAUTHORIZED) << "TVM service ticket status is " << NTvmAuth::StatusToString(status));
                    NTvmAuth::TTvmId clientId = ticket.GetSrc();
                    Y_ENSURE_EX(Server->GetConfig().GetTvmOptions().AcceptedClientIds.contains(clientId), TCodedException(HTTP_FORBIDDEN) << "TVM service " << clientId << " is not permitted");
                    Ctx.TvmClientId = clientId;
                } else {
                    Y_ENSURE_EX(Context->IsLocal(), TCodedException(HTTP_UNAUTHORIZED) << "either TVM service ticket or localhost is required");
                }
            } else {
                Y_ENSURE_EX(Context->IsLocal(), TCodedException(HTTP_UNAUTHORIZED) << "localhost is required");
            }
        }

        template <typename F>
        void AddTask(F&& createHandler, const TString& imei, TDuration waitTimeout) {
            if (waitTimeout) {
                auto connection = Yensured(Server)->WaitConnection(imei);
                auto waitDeadline = Context->GetRequestStartTime() + waitTimeout;
                auto callback = [context = Context, ctx = Ctx, server = Server, createHandler = std::forward<F>(createHandler)](const NThreading::TFuture<TTelematicsConnectionPtr>& c) {
                    try {
                        if (c.HasException()) {
                            c.GetValue();
                        }
                        if (c.HasValue()) {
                            const auto& connection = c.GetValue();
                            Y_ENSURE(connection->Alive(), "connection is dead");
                            auto handler = createHandler(connection);
                            auto inserted = Yensured(server)->AddTask(handler);
                            connection->AddHandler(handler);
                            SendReply(context, ctx, handler->Serialize());
                        } else {
                            SendReply(context, ctx, "connection is still absent", HTTP_NOT_FOUND);
                        }
                    } catch (const std::exception& e) {
                        SendReply(context, ctx, FormatExc(e), HTTP_INTERNAL_SERVER_ERROR);
                    }
                };
                NThreading::Subscribe(connection, std::move(callback), waitDeadline);
            } else {
                auto connection = GetConnection(imei);
                Y_ENSURE(connection->Alive(), "connection is dead");
                auto handler = createHandler(connection);
                auto inserted = Yensured(Server)->AddTask(handler);
                connection->AddHandler(handler);
                SendReply(handler->Serialize());
            }
        }
        void SendReply(NJson::TJsonValue&& reply, ui32 code = HTTP_OK) {
            SendReply(Context, Ctx, std::move(reply), code);
        }
        static void SendReply(IReplyContext::TPtr context, const TCtx& ctx, NJson::TJsonValue&& reply, ui32 code = HTTP_OK) try {
            TBufferOutput output;
            output << reply.GetStringRobust() << Endl;
            context->MakeSimpleReply(output.Buffer(), code);
            NJson::TJsonValue data;
            data["code"] = code;
            data["reqid"] = ctx.ReqId;
            data["response"] = std::move(reply);
            data["tvm_client_id"] = ctx.TvmClientId;
            NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::Response, context.Get(), ctx.IMEI, std::move(data));
            TDuration duration = Now() - context->GetRequestStartTime();
            TTelematicsUnistatSignals::Get().ClientTimes.Signal(TString{context->GetUri()}, duration.MilliSeconds());
        } catch (const std::exception& e) {
            TString error = FormatExc(e);
            ERROR_LOG << "SendReplyException: " << ctx.ReqId << ' ' << code << ' ' << reply.GetStringRobust() << ": " << error << Endl;
            NJson::TJsonValue data;
            data["code"] = code;
            data["exception"] = error;
            data["reqid"] = ctx.ReqId;
            data["response"] = std::move(reply);
            data["tvm_client_id"] = ctx.TvmClientId;
            NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::ResponseError, context.Get(), ctx.IMEI, std::move(data));
            TTelematicsUnistatSignals::Get().ClientErrors.Signal(TString{context->GetUri()}, 1);
        }

    private:
        const TCgiParameters& Cgi;
        TTelematicsServer* Server;

        NJson::TJsonValue Post;
        TCtx Ctx;
    };
}

IHttpReplier::TPtr NDrive::TTelematicsServerHttpClient::DoSelectHandler(IReplyContext::TPtr context) {
    return new NDrive::TReplier(Server, context);
}
