#include "config.h"

#include <drive/telematics/server/back/pusher.h>
#include <drive/telematics/server/pusher/local.h>
#include <drive/telematics/server/pusher/meta.h>
#include <drive/telematics/server/pusher/pusher.h>
#include <drive/telematics/server/saas/pusher.h>
#include <drive/telematics/server/telematics_cache_api/pusher.h>

#include <library/cpp/mediator/global_notifications/system_status.h>

#include <util/generic/iterator_range.h>
#include <util/string/split.h>
#include <util/string/vector.h>

namespace {
    const TString DefaultConfig = "<Server></Server>";
}

NDrive::TSensorPushOptions NDrive::TSensorPushOptions::Realtime() {
    NDrive::TSensorPushOptions result;
    result.Policy = EPolicy::Realtime;
    return result;
}

NDrive::TSensorPushOptions NDrive::TSensorPushOptions::Periodical(TDuration period) {
    NDrive::TSensorPushOptions result;
    result.Period = period;
    return result;
}

NDrive::TSensorPushOptions NDrive::TSensorPushOptions::Ignore() {
    NDrive::TSensorPushOptions result;
    result.Policy = EPolicy::Ignore;
    return result;
}

NDrive::TTelematicsConfig::TTelematicsConfig()
    : TSectionParser<TAnyYandexConfig>(DefaultConfig.c_str(), "Server")
{
}

NDrive::TTelematicsConfig::TTelematicsConfig(const TServerConfigConstructorParams& params)
    : TSectionParser<TAnyYandexConfig>(params.GetText().c_str(), "Server")
    , DaemonConfig(*params.GetDaemonConfig())
    , UnifiedAgentGrpcMaxMessageSize()
    , UnifiedAgentMaxInflightBytes()
{
    Parse();
}

void NDrive::TTelematicsConfig::AddPusher(THolder<IPusherOptions>&& options) {
    if (!PusherOptions) {
        PusherOptions = std::move(options);
        return;
    }
    if (auto meta = dynamic_cast<TMetaPusherOptions*>(PusherOptions.Get())) {
        meta->MutablePushersOptions().push_back(std::move(options));
        return;
    }

    auto meta = MakeHolder<TMetaPusherOptions>();
    meta->MutablePushersOptions().push_back(std::move(PusherOptions));
    meta->MutablePushersOptions().push_back(std::move(options));
    PusherOptions = std::move(meta);
}

void NDrive::TTelematicsConfig::Parse() {
    CHECK_WITH_LOG(ServerSection);
    const auto& directives = ServerSection->GetDirectives();
    EventLog = directives.Value("EventLog", EventLog);
    UnifiedAgentEventLog = directives.Value("UnifiedAgentEventLog", UnifiedAgentEventLog);
    UnifiedAgentGrpcMaxMessageSize = directives.Value("UnifiedAgentGrpcMaxMessageSize", UnifiedAgentGrpcMaxMessageSize);
    UnifiedAgentGrpcReconnectDelay = directives.Value("UnifiedAgentGrpcReconnectDelay", UnifiedAgentGrpcReconnectDelay);
    UnifiedAgentGrpcSendDelay = directives.Value("UnifiedAgentGrpcSendDelay", UnifiedAgentGrpcSendDelay);
    UnifiedAgentMaxInflightBytes = directives.Value("UnifiedAgentMaxInflightBytes", UnifiedAgentMaxInflightBytes);
    RigorMortisThreshold = TDuration::Seconds(directives.Value("RigorMortisThreshold", RigorMortisThreshold.Seconds()));
    CanScannerInterval = TDuration::Seconds(directives.Value("CanScannerInterval", CanScannerInterval.Seconds()));
    PingerInterval = TDuration::Seconds(directives.Value("PingerInterval", PingerInterval.Seconds()));
    SensorInterval = TDuration::Seconds(directives.Value("SensorInterval", SensorInterval.Seconds()));
    FastDataInterval = TDuration::Seconds(directives.Value("FastDataInterval", FastDataInterval.Seconds()));
    LogLoaderInterval = TDuration::Seconds(directives.Value("LogLoaderInterval", LogLoaderInterval.Seconds()));
    SimSwitcherInterval = TDuration::Seconds(directives.Value("SimSwitcherInterval", SimSwitcherInterval.Seconds()));
    BeaconsRefreshInterval = TDuration::Seconds(directives.Value("BeaconsRefreshInterval", BeaconsRefreshInterval.Seconds()));
    BleSessionKeyWatchInterval = TDuration::Seconds(directives.Value("BleSessionKeyWatchInterval", BleSessionKeyWatchInterval.Seconds()));
    PinResetterInterval = TDuration::Seconds(directives.Value("PinResetterInterval", PinResetterInterval.Seconds()));
    KnownBeaconsSynchronizationInterval = TDuration::Seconds(directives.Value("KnownBeaconsSynchronizationInterval", KnownBeaconsSynchronizationInterval.Seconds()));
    DropErrorCount = directives.Value("DropErrorCount", DropErrorCount);
    SchedulerThreads = directives.Value("SchedulerThreads", SchedulerThreads);
    SensorValuesCount = directives.Value("SensorValuesCount", SensorValuesCount);
    TraceInputTraffic = directives.Value("TraceInputTraffic", TraceInputTraffic);
    ProtocolType = directives.Value("ProtocolType", ProtocolType);

    const auto sections = ServerSection->GetAllChildren();
    {
        auto p = sections.find("ClientServer");
        AssertCorrectConfig(p != sections.end(), "cannot find ClientServer section");
        ClientServerOptions.Init(p->second->GetDirectives());
    }
    {
        auto p = sections.find("TelematicsServer");
        AssertCorrectConfig(p != sections.end(), "cannot find TelematicsServer section");
        TelematicsServerOptions.Init(p->second->GetDirectives());
    }
    {
        auto p = sections.find("Executor");
        if (p != sections.end()) {
            AssertCorrectConfig(TaskExecutorConfig.Init(p->second), "cannot read Executor section");
        }
    }
    {
        auto p = sections.find("Sensors");
        if (p != sections.end()) {
            // sensor directives format: "SENSOR_NAME: SENSOR_ID [UPDATE_PERIOD]"
            for (auto&& d : p->second->GetDirectives()) {
                TVector<TString> parts;
                StringSplitter(d.second).Split(' ').SkipEmpty().Collect(&parts);
                Sensors.emplace(d.first, FromString<TSensorId>(parts[0]));

                TDuration interval = SensorInterval;
                if (parts.size() > 1) {
                    interval = FromString<TDuration>(parts[1]);
                }
                Y_ENSURE(interval >= TDuration::Seconds(1));
                auto it = SensorIntervals.find(interval);
                if (it == SensorIntervals.end()) {
                    TSensorsDict sensors;
                    SensorIntervals.emplace(interval, sensors);
                }
                SensorIntervals[interval][d.first] = Sensors[d.first];
            }
        }
    }
    {
        auto createPusherOptions = [&](const TYandexConfig::Section& section) mutable -> THolder<NDrive::IPusherOptions> {
            auto options = IPusherOptions::ConstructPusherOptions(section);
            auto destinationIds = options->GetDestinationClientIds();
            TvmOptions.DestinationClientIds.insert(destinationIds.begin(), destinationIds.end());
            return options;
        };

        auto range = sections.equal_range("Pusher");
        if (std::distance(range.first, range.second) == 0) {
            PusherOptions = MakeHolder<TLocalPusherOptions>();
        } else if (std::distance(range.first, range.second) == 1) {
            PusherOptions = createPusherOptions(*range.first->second);
        } else {
            auto pusherOptions = MakeHolder<TMetaPusherOptions>();
            for(auto it = range.first; it != range.second; ++it) {
                pusherOptions->MutablePushersOptions().push_back(createPusherOptions(*it->second));
            }
            PusherOptions = std::move(pusherOptions);
        }
    }
    {
        auto p = sections.find("Locator");
        if (p != sections.end()) {
            const auto& d = p->second->GetDirectives();
            LocatorOptions.LBSToken = d.Value("LBSToken", LocatorOptions.LBSToken);
            LocatorOptions.LinkerHost = d.Value("LinkerHost", LocatorOptions.LinkerHost);
            LocatorOptions.SensorHost = d.Value("SensorHost", LocatorOptions.SensorHost);
            LocatorOptions.SensorPort = d.Value("SensorPort", LocatorOptions.SensorPort);
            LocatorOptions.SensorSpMetaSearch = d.Value("SensorSpMetaSearch", LocatorOptions.SensorSpMetaSearch);
            LocatorOptions.SensorSaasTvmId = d.Value("SensorSaasTvmId", LocatorOptions.SensorSaasTvmId);
            LocatorOptions.SensorService = d.Value("SensorService", LocatorOptions.SensorService);
            LocatorOptions.SensorAPIName = d.Value("SensorAPIName", LocatorOptions.SensorAPIName);
            LocatorOptions.TracksHost = d.Value("TracksHost", LocatorOptions.TracksHost);
            LocatorOptions.GeocoderHost = d.Value("GeocoderHost", LocatorOptions.GeocoderHost);
            LocatorOptions.GeocoderPort = d.Value("GeocoderPort", LocatorOptions.GeocoderPort);
            LocatorOptions.GeocoderPath = d.Value("GeocoderPath", LocatorOptions.GeocoderPath);
            LocatorOptions.GeocoderClientId = d.Value("GeocoderClientId", LocatorOptions.GeocoderClientId);
            LocatorOptions.EnableClusterization = d.Value("EnableClusterization", LocatorOptions.EnableClusterization);
            LocatorOptions.LinkDeviationLengthThreshold = d.Value("LinkDeviationLengthThreshold", LocatorOptions.LinkDeviationLengthThreshold);
            LocatorOptions.LinkEnableProjection = d.Value("LinkEnableProjection", LocatorOptions.LinkEnableProjection);
            LocatorOptions.LinkFilterByEngine = d.Value("LinkFilterByEngine", LocatorOptions.LinkFilterByEngine);
            LocatorOptions.LinkFilterBySpeed = d.Value("LinkFilterBySpeed", LocatorOptions.LinkFilterBySpeed);
            LocatorOptions.LinkFilterByRandomFraction = d.Value("LinkFilterByRandomFraction", LocatorOptions.LinkFilterByRandomFraction);
            LocatorOptions.LinkTailLength = d.Value("LinkTailLength", LocatorOptions.LinkTailLength);
            const auto sections = p->second->GetAllChildren();
            const auto restricted = sections.equal_range("RestrictedAreas");
            for (auto&&[sectioname, section] : MakeIteratorRange(restricted)) {
                AssertCorrectConfig(section, "bad section %s", sectioname.c_str());
                const auto& areas = section->GetDirectives();
                for (auto&&[name, line] : areas) {
                    TStringBuf str(line);
                    TVector<TGeoCoord> coordinates;
                    AssertCorrectConfig(TGeoCoord::DeserializeVector(str, coordinates), "cannot deserialize %s", line);
                    LocatorOptions.RestrictedAreas.emplace_back(std::move(coordinates));
                }
            }
            if (LocatorOptions.GeocoderClientId) {
                TvmOptions.DestinationClientIds.insert(LocatorOptions.GeocoderClientId);
            }
            if (LocatorOptions.SensorSaasTvmId) {
                TvmOptions.DestinationClientIds.insert(LocatorOptions.SensorSaasTvmId);
            }
        }
    }
    {
        auto p = sections.find("Storage");
        if (p == sections.end()) {
            p = sections.find("OffsetStorageOptions");
        }
        if (LogLoaderInterval) {
            AssertCorrectConfig(p != sections.end(), "cannot find Storage section");
        }
        if (p != sections.end()) {
            AssertCorrectConfig(StorageOptions.Init(p->second), "cannot read Storage options");
        }
    }
    {
        auto p = sections.find("Tvm");
        if (p != sections.end()) {
            const auto& d = p->second->GetDirectives();
            TvmOptions.Cache = d.Value("Cache", TvmOptions.Cache);
            TvmOptions.Token = d.Value("Token", TvmOptions.Token);
            StringSplitter(
                d.Value("AcceptedClientIds", JoinStrings(TvmOptions.AcceptedClientIds.begin(), TvmOptions.AcceptedClientIds.end(), TStringBuf(",")))
            ).Split(',').SkipEmpty().ParseInto(&TvmOptions.AcceptedClientIds);
            TvmOptions.SelfClientId = d.Value("SelfClientId", TvmOptions.SelfClientId);
        }
    }
    {
        auto p = sections.find("Validation");
        if (p != sections.end()) {
            const auto& d = p->second->GetDirectives();
            DataValidationOptions.FilterOdometerEnabled = d.Value("FilterOdometerEnabled", DataValidationOptions.FilterOdometerEnabled);
            DataValidationOptions.FilterSkipZeroEnabled = d.Value("FilterSkipZeroEnabled", DataValidationOptions.FilterSkipZeroEnabled);
        }
    }
}
