#pragma once

#include <drive/telematics/server/pusher/interface.h>
#include <drive/telematics/server/sensors/validation.h>

#include <drive/telematics/protocol/protocol.h>
#include <drive/telematics/protocol/sensor.h>
#include <drive/telematics/protocol/vega.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/module/module.h>

#include <library/cpp/tvmauth/type.h>

#include <rtline/library/executor/executor.h>
#include <rtline/library/geometry/polyline.h>

namespace NDrive {
    struct TConnectionCategory {
        TString Name;
        TVector<TSensorId> Sensors;
        TSet<TString> Values;
    };

    struct TLocatorOptions {
        TString LBSToken;
        TString LinkerHost;
        TString SensorHost;
        ui16 SensorPort = 0;
        TString SensorService;
        ui32 SensorSaasTvmId = 0;
        TString SensorAPIName;
        TString SensorSpMetaSearch;
        TString TracksHost;
        TString GeocoderHost;
        ui16 GeocoderPort = 0;
        TString GeocoderPath;
        ui32 GeocoderClientId = 0;
        TVector<TGeoPolyLine> RestrictedAreas;
        double LinkDeviationLengthThreshold = 50;
        double LinkFilterByRandomFraction = 0;
        double LinkTailLength = 250;
        double SpeedThreshold = 200.0 / 3600 * 1000;
        bool EnableClusterization = true;
        bool LinkEnableProjection = false;
        bool LinkFilterByEngine = true;
        bool LinkFilterBySpeed = true;
    };

    struct TTvmOptions {
        TString Cache;
        TString Token;
        TSet<NTvmAuth::TTvmId> AcceptedClientIds;
        TSet<NTvmAuth::TTvmId> DestinationClientIds;
        NTvmAuth::TTvmId SelfClientId = 0;
    };

    struct TSensorPushOptions {
    public:
        enum class EPolicy {
            Realtime,
            Deferred,
            Ignore,
        };

    public:
        TDuration Period = TDuration::Seconds(60);
        EPolicy Policy = EPolicy::Deferred;

    public:
        static TSensorPushOptions Realtime();
        static TSensorPushOptions Periodical(TDuration period);
        static TSensorPushOptions Ignore();
    };

    using TSensorsDict = TMap<TString, TSensorId>;

    class TTelematicsConfig
        : public TSectionParser<TAnyYandexConfig>
        , public IServerConfig
    {
    public:
        TTelematicsConfig();
        TTelematicsConfig(const TServerConfigConstructorParams& params);

        void AddPusher(THolder<IPusherOptions>&& options);

        virtual const TDaemonConfig& GetDaemonConfig() const override {
            return DaemonConfig;
        }
        virtual TSet<TString> GetModulesSet() const override {
            return {};
        }

        const THttpServerOptions& GetClientServerOptions() const {
            return ClientServerOptions;
        }
        const THttpServerOptions& GetTelematicsServerOptions() const {
            return TelematicsServerOptions;
        }
        const TTaskExecutorConfig& GetTaskExecutorConfig() const {
            return TaskExecutorConfig;
        }
        const TLocatorOptions& GetLocatorOptions() const {
            return LocatorOptions;
        }
        const NDrive::IPusherOptions& GetPusherOptions() const {
            return *PusherOptions;
        }
        const TTvmOptions& GetTvmOptions() const {
            return TvmOptions;
        }
        const NRTLine::TStorageOptions& GetStorageOptions() const {
            return StorageOptions;
        }
        const TSensorsDict& GetSensors() const {
            return Sensors;
        }
        const TString& GetEventLog() const {
            return EventLog;
        }
        const TString& GetUnifiedAgentEventLog() const {
            return UnifiedAgentEventLog;
        }
        size_t GetUnifiedAgentGrpcMaxMessageSize() const {
            return UnifiedAgentGrpcMaxMessageSize;
        }
        TDuration GetUnifiedAgentGrpcReconnectDelay() const {
            return UnifiedAgentGrpcReconnectDelay;
        }
        TDuration GetUnifiedAgentGrpcSendDelay() const {
            return UnifiedAgentGrpcSendDelay;
        }
        size_t GetUnifiedAgentMaxInflightBytes() const {
            return UnifiedAgentMaxInflightBytes;
        }
        TDuration GetRigorMortisThreshold() const {
            return RigorMortisThreshold;
        }
        TDuration GetCanScannerInterval() const {
            return CanScannerInterval;
        }
        TDuration GetPingerInterval() const {
            return PingerInterval;
        }
        TDuration GetSensorInterval() const {
            return SensorInterval;
        }
        TDuration GetTickInterval() const {
            return TickInterval;
        }
        TDuration GetFastDataInterval() const {
            return FastDataInterval;
        }
        TDuration GetLogLoaderInterval() const {
            return LogLoaderInterval;
        }
        TDuration GetSimSwitcherInterval() const {
            return SimSwitcherInterval;
        }
        TDuration GetBeaconsRefreshInterval() const {
            return BeaconsRefreshInterval;
        }
        TDuration GetBleSessionKeyWatchInterval() const {
            return BleSessionKeyWatchInterval;
        }
        TDuration GetKnownBeaconsSynchronizationInterval() const {
            return KnownBeaconsSynchronizationInterval;
        }
        TDuration GetTaskGarbageCollectionInterval() const {
            return TaskGarbageCollectionInterval;
        }
        TDuration GetSuccessfulTaskLifetime() const {
            return SuccessfulTaskLifetime;
        }
        TDuration GetFailedTaskLifetime() const {
            return FailedTaskLifetime;
        }
        TDuration GetPinResetterInterval() const {
            return PinResetterInterval;
        }
        const TVector<TConnectionCategory>& GetConnectionCategories() const {
            return ConnectionCategories;
        }
        const TDataValidationOptions& GetDataValidationOptions() const {
            return DataValidationOptions;
        }
        const TMap<TDuration, TSensorsDict>& GetSensorIntervals() const {
            return SensorIntervals;
        }
        const TMap<ui16, TSensorPushOptions>& GetSensorPushOptions() const {
            return SensorPushOptions;
        }
        ui32 GetDropErrorsCount() const {
            return DropErrorCount;
        }
        ui32 GetSchedulerThreads() const {
            return SchedulerThreads;
        }
        ui32 GetSensorValuesCount() const {
            return SensorValuesCount;
        }
        bool GetTraceInputTraffic() const {
            return TraceInputTraffic;
        }
        NProtocol::EProtocolType GetProtocolType() const {
            return ProtocolType;
        }

        void SetClientServerPort(ui16 port) {
            ClientServerOptions.SetPort(port);
        }
        void SetTelematicsServerPort(ui16 port) {
            TelematicsServerOptions.SetPort(port);
        }
        void SetTaskExecutorConfig(const TTaskExecutorConfig& value) {
            TaskExecutorConfig = value;
        }
        void SetEventLog(const TString& path) {
            EventLog = path;
        }
        void SetPingerInterval(TDuration interval) {
            PingerInterval = interval;
        }
        void SetSensorInterval(TDuration interval) {
            SensorInterval = interval;
        }
        void SetFastDataInterval(TDuration interval) {
            FastDataInterval = interval;
        }
        void SetLogLoaderInterval(TDuration interval) {
            LogLoaderInterval = interval;
        }
        void SetSimSwitcherInterval(TDuration interval) {
            SimSwitcherInterval = interval;
        }
        void SetBeaconsRefreshInterval(TDuration interval) {
            BeaconsRefreshInterval = interval;
        }
        void SetBleSessionKeyWatchInterval(TDuration interval) {
            BleSessionKeyWatchInterval = interval;
        }
        void SetKnownBeaconsSynchronizationInterval(TDuration interval) {
            KnownBeaconsSynchronizationInterval = interval;
        }
        void SetPinResetterInterval(TDuration interval) {
            PinResetterInterval = interval;
        }
        void SetLocatorOptions(const TLocatorOptions& options) {
            LocatorOptions = options;
        }
        void SetOffsetStorageOptions(const NRTLine::TStorageOptions& options) {
            StorageOptions = options;
        }
        void SetPusherOptions(THolder<IPusherOptions>&& options) {
            PusherOptions = std::move(options);
        }
        void SetSensorPushOptions(const TMap<ui16, TSensorPushOptions>& options) {
            SensorPushOptions = options;
        }
        void SetTraceInputTraffic(bool state) {
            TraceInputTraffic = state;
        }
        void SetProtocolType(NProtocol::EProtocolType protocolType) {
            ProtocolType = protocolType;
        }

    private:
        void Parse();

    private:
        TDaemonConfig DaemonConfig;

        TDaemonConfig::THttpOptions ClientServerOptions;
        TDaemonConfig::THttpOptions TelematicsServerOptions;
        TTaskExecutorConfig TaskExecutorConfig;
        TLocatorOptions LocatorOptions;
        THolder<IPusherOptions> PusherOptions;
        TTvmOptions TvmOptions;
        NRTLine::TStorageOptions StorageOptions;

        TVector<TConnectionCategory> ConnectionCategories = {
            { "hni", { VEGA_MCC, VEGA_MNC }, { "250-2", "250-99" } },
        };

        TDataValidationOptions DataValidationOptions;

        TSensorsDict Sensors;
        TMap<TDuration, TSensorsDict> SensorIntervals;
        TMap<ui16, TSensorPushOptions> SensorPushOptions = {
            { CAN_ENGINE_IS_ON, TSensorPushOptions::Realtime() },
            { CAN_FUEL_LEVEL_P, TSensorPushOptions::Realtime() },
            { CAN_FUEL_DISTANCE_KM, TSensorPushOptions::Realtime() },
            { VEGA_SPEED,       TSensorPushOptions::Realtime() },
            { CAN_DRIVER_DOOR,  TSensorPushOptions::Realtime() },
            { CAN_PASS_DOOR,    TSensorPushOptions::Realtime() },
            { CAN_L_REAR_DOOR,  TSensorPushOptions::Realtime() },
            { CAN_R_REAR_DOOR,  TSensorPushOptions::Realtime() },
            { NVega::DigitalOutput<3>(), TSensorPushOptions::Realtime() },
            { VEGA_SETTING_TRANSLATE_SENSORS, TSensorPushOptions::Ignore() },
            { VEGA_MCU_FIRMWARE_VERSION, TSensorPushOptions::Realtime() },
        };

        TString EventLog;
        TString UnifiedAgentEventLog;
        size_t UnifiedAgentGrpcMaxMessageSize;
        TDuration UnifiedAgentGrpcReconnectDelay;
        TDuration UnifiedAgentGrpcSendDelay;
        size_t UnifiedAgentMaxInflightBytes;
        TDuration RigorMortisThreshold;
        TDuration CanScannerInterval;
        TDuration PingerInterval;
        TDuration SensorInterval = TDuration::Hours(1);
        TDuration TickInterval = TDuration::Seconds(5);
        TDuration FastDataInterval;
        TDuration LogLoaderInterval;
        TDuration SimSwitcherInterval;
        TDuration BeaconsRefreshInterval = TDuration::Minutes(1);
        TDuration BleSessionKeyWatchInterval = TDuration::Minutes(5);
        TDuration PinResetterInterval = TDuration::Zero();
        TDuration KnownBeaconsSynchronizationInterval = TDuration::Minutes(10);
        TDuration TaskGarbageCollectionInterval = TDuration::Seconds(10);
        TDuration SuccessfulTaskLifetime = TDuration::Minutes(1);
        TDuration FailedTaskLifetime = TDuration::Hours(3);
        ui32 DropErrorCount = 128;
        ui32 SchedulerThreads = 0;
        ui32 SensorValuesCount = 8;
        bool TraceInputTraffic = false;
        NProtocol::EProtocolType ProtocolType = NProtocol::EProtocolType::PT_VEGA;
    };
}
