#include "connection.h"

#include "handlers.h"
#include "logging.h"
#include "navtelecom.h"
#include "protocol.h"
#include "scenarios.h"
#include "sensors.h"
#include "server.h"
#include "wialon.h"

#include <drive/telematics/server/common/signals.h>
#include <drive/telematics/server/location/beacon_recognizer.h>
#include <drive/telematics/server/location/locator.h>
#include <drive/telematics/server/metadata/client.h>
#include <drive/telematics/server/pusher/interface.h>
#include <drive/telematics/server/sensors/calculator.h>
#include <drive/telematics/server/sensors/validation.h>

#include <drive/telematics/api/client.h>
#include <drive/telematics/common/beacon.h>
#include <drive/telematics/protocol/crc.h>
#include <drive/telematics/protocol/mio.h>
#include <drive/telematics/protocol/vega.h>

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/logger/global/global.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/scheduler/global.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>

#include <util/generic/buffer.h>
#include <util/random/random.h>
#include <util/stream/buffer.h>
#include <util/stream/mem.h>
#include <util/string/hex.h>

namespace {
    THolder<NDrive::NProtocol::IProtocol> CreateProtocol(
        NDrive::NProtocol::EProtocolType type,
        NDrive::NProtocol::IProtocol::TSendHandler sender,
        NDrive::TTelematicsServer* server
    ) {
        switch (type) {
            case NDrive::NProtocol::PT_WIALON_IPS:
                return NDrive::NProtocol::IProtocol::Create<NDrive::NWialon::TWialonProtocol>(std::move(sender), server);
            case NDrive::NProtocol::PT_NAVTELECOM:
                return NDrive::NProtocol::IProtocol::Create<NDrive::NNavTelecom::TNavTelecomProtocol>(std::move(sender));
            default:
                return nullptr;
        }
    }
}

namespace NDrive {
    class TTelematicsConnection::TScheduledOnAfterRegister: public TGlobalScheduler::TScheduledItem<TScheduledOnAfterRegister> {
    private:
        using TBase = TGlobalScheduler::TScheduledItem<TScheduledOnAfterRegister>;

    public:
        TScheduledOnAfterRegister(
            const NDrive::TTelematicsServer* server,
            const TString& imei,
            NDrive::TTelematicsHandlerPtr handler,
            TInstant time
        )
            : TBase(server ? server->Name() : "NullServerName", "OnAfterRegister:" + imei + ":" + ToString<const void*>(handler.Get()), time)
            , Server(server)
            , IMEI(imei)
            , Handler(handler)
        {
        }

        void Process(void* /*threadSpecificResource*/) override {
            THolder<TScheduledOnAfterRegister> cleanup(this);
            Y_ASSERT(Server);
            auto connection = Server ? Server->GetConnection(IMEI) : nullptr;
            if (connection && connection->Alive()) {
                connection->InitializeHandler(Handler);
            }
        }

    private:
        const NDrive::TTelematicsServer* Server;
        const TString IMEI;
        const NDrive::TTelematicsHandlerPtr Handler;
    };

    class TTelematicsConnection::TScheduledHandlerTick: public TGlobalScheduler::TScheduledItem<TTelematicsConnection::TScheduledHandlerTick> {
    private:
        using TBase = TGlobalScheduler::TScheduledItem<TTelematicsConnection::TScheduledHandlerTick>;

    public:
        TScheduledHandlerTick(const NDrive::TTelematicsServer* server, const TString& imei, TInstant now)
            : TBase(server ? server->Name() : "NullServerName", "HandlerTick:" + imei, now + server->GetConfig().GetTickInterval())
            , Server(server)
            , IMEI(imei)
        {
        }

        void Process(void* /*threadSpecificResource*/) override {
            THolder<TScheduledHandlerTick> cleanup(this);
            Y_ASSERT(Server);
            auto connection = Server ? Server->GetConnection(IMEI) : nullptr;
            if (connection && connection->Alive()) {
                INFO_LOG << connection->DebugString() << ": commencing tick" << Endl;
                connection->Handle(NVega::TMessage());
                connection->PushDeferred();
            }
        }

        THolder<TGlobalScheduler::IScheduledItem> GetNextScheduledItem(TInstant now) const override {
            Y_ASSERT(Server);
            auto connection = Server ? Server->GetConnection(IMEI) : nullptr;
            if (connection && connection->Alive()) {
                return MakeHolder<TScheduledHandlerTick>(Server, IMEI, now);
            } else {
                return nullptr;
            }
        }

    private:
        const NDrive::TTelematicsServer* Server;
        const TString IMEI;
    };

    class TTelematicsConnection::TDeferredSensorState {
    public:
        inline TDeferredSensorState(const NDrive::TSensor& sensor, TDuration period)
            : Sensor(sensor)
            , Period(period)
            , Committed(TInstant::Zero())
        {
        }

        NDrive::TSensorId Id() const {
            return Sensor;
        }

        TMaybe<NDrive::TSensor> Get(TInstant now) const {
            if (ShouldPush(now)) {
                auto guard = Guard(Lock);
                return Sensor;
            } else {
                return {};
            }
        }

        TMaybe<NDrive::TSensor> Set(NDrive::TSensor&& sensor) {
            if (sensor.Timestamp > Sensor.Timestamp) {
                auto guard = Guard(Lock);
                CHECK_WITH_LOG(Sensor.Id == sensor.Id);
                CHECK_WITH_LOG(Sensor.SubId == sensor.SubId);
                if (sensor.Timestamp > Sensor.Timestamp) {
                    Sensor = std::move(sensor);
                    if (ShouldPush(Sensor.Timestamp)) {
                        return Sensor;
                    }
                }
            }
            return {};
        }

        void Commit(TInstant timestamp) {
            if (timestamp > Committed) {
                auto guard = Guard(Lock);
                Committed = std::max(Committed, timestamp);
            }
        }

    private:
        bool ShouldPush(TInstant now) const {
            return Sensor.Timestamp > Committed && now > Committed + Period;
        }

    private:
        NDrive::TSensor Sensor;
        TDuration Period;
        TInstant Committed;

        TAdaptiveLock Lock;
    };
}

NDrive::TTelematicsConnectionPtr NDrive::TTelematicsConnection::Create(TTelematicsServer* server, NUtil::TTcpServer::TConnectionPtr connection) {
    NDrive::TTelematicsConnectionPtr result = new TTelematicsConnection(server, connection);
    result->Launch();
    return result;
}

NDrive::TTelematicsConnection::TTelematicsConnection(TTelematicsServer* server, NUtil::TTcpServer::TConnectionPtr connection)
    : Server(server)
    , Connection(connection)
    , CreatedTime(Now())
    , Locations(MakeAtomicShared<TLocationCache>())
    , SensorsCache(server ? server->GetConfig().GetSensorValuesCount() : 42)
    , MessageId(0)
    , InputBufferSize(1 << 13)
    , ErrorsCount(0)
    , ExpectedNumber(0)
    , Registered(false)
    , CurrentProtocolType(NDrive::NProtocol::PT_INVALID)
{
    RegisterSensorsSignals();
    RegisterSensorsFallback();
    RemoteAddr = Connection ? Connection->GetRemoteAddr() : "NullConnectionRemoteAddr";
    DEBUG_LOG << DebugString() << ": created connection" << Endl;
    TTelematicsUnistatSignals::Get().ConnectionEstablished.Signal(1);
}

NDrive::TTelematicsConnection::~TTelematicsConnection() {
    DEBUG_LOG << DebugString() << ": destroying " << Alive() << Endl;
    if (auto guard = Guard(Lock)) {
        if (IMEI && Registered.load()) {
            Deregister();
        }
        for (auto&& handler : Handlers) {
            CHECK_WITH_LOG(handler);
            handler->OnTermination();
        }
        TTelematicsLog::Log(TTelematicsLog::EEvent::Closed, this);
    }
    TTelematicsUnistatSignals::Get().ConnectionDestroyed.Signal(1);
    DEBUG_LOG << DebugString() << ": destroyed " << Endl;
}

template <class T>
NThreading::TFuture<void> Push(NDrive::IPusher& pusher, const TString& imei, const TConstArrayRef<T> dataArray) {
    if (dataArray.empty()) {
        return NThreading::MakeFuture();
    }
    if constexpr (std::is_same_v<T, NDrive::TSensor>) {
        TVector<NDrive::TSensorId> sensorIds;
        sensorIds.reserve(dataArray.size());
        for (const auto& sensor : dataArray) {
            sensorIds.push_back(sensor.Id);
        }
        return pusher.BulkPush(imei, dataArray).Subscribe([sensorIds = std::move(sensorIds)](const NThreading::TFuture<NDrive::IPusher::TBulkPushResult>& result) {
            if (result.HasValue() && result.GetValue().Written) {
                for (const auto sensorId : sensorIds) {
                    NDrive::TTelematicsConnection::SensorsPushSignals.Signal(sensorId, 1);
                }
                NDrive::TTelematicsUnistatSignals::Get().PusherSuccess.Signal(1);
            } else {
                NDrive::TTelematicsUnistatSignals::Get().PusherFailure.Signal(1);
            }
        }).IgnoreResult();
    }
    return pusher.BulkPush(imei, dataArray).Subscribe([](const NThreading::TFuture<NDrive::IPusher::TBulkPushResult>& result) {
        if (result.HasValue() && result.GetValue().Written) {
            NDrive::TTelematicsUnistatSignals::Get().PusherSuccess.Signal(1);
        } else {
            NDrive::TTelematicsUnistatSignals::Get().PusherFailure.Signal(1);
        }
    }).IgnoreResult();
}

template <class T>
NThreading::TFuture<void> NDrive::TTelematicsConnection::Push(TArrayRef<T> dataArray) const {
    auto pusher = Server ? Server->GetPusher() : nullptr;
    if (pusher && IMEI) {
        TVector<T> dataToPush;
        dataToPush.reserve(dataArray.size());
        for (auto&& data : dataArray) {
            auto&& [d, status] = Validate(std::move(data), SensorsCache, Server->GetConfig().GetDataValidationOptions());
            if (status == EDataValidationStatus::Ok) {
                dataToPush.push_back(std::move(d));
            } else {
                auto serialized = NJson::ToJson(d);
                const auto debugString = serialized.GetStringRobust();
                TTelematicsLog::Log(TTelematicsLog::ValidationError, this, TypeName<T>(), NJson::TMapBuilder
                    ("object", std::move(serialized))
                    ("status", ToString(status))
                );
                ERROR_LOG << DebugString() << ": " << debugString << " validation error " << status << Endl;
            }
        }
        return ::Push<T>(*pusher, IMEI, dataToPush);
    }
    return {};
}

NThreading::TFuture<void> NDrive::TTelematicsConnection::Push(TArrayRef<TTelematicsHandlerPtr> handlers) const {
    auto pusher = Server ? Server->GetPusher() : nullptr;
    if (pusher && IMEI) {
        TVector<TTelematicsHandlerPtr> handlersToPush;
        handlersToPush.reserve(handlers.size());
        for (auto&& handler : handlers) {
            if (handler && handler->Serializable()) {
                handlersToPush.push_back(std::move(handler));
            }
        }
        ::Push<TTelematicsHandlerPtr>(*pusher, IMEI, handlersToPush);
    }
    return {};
}

NThreading::TFuture<void> NDrive::TTelematicsConnection::PushOrDefer(TArrayRef<TSensor> sensors) {
    const auto& pushOptions = Server ? Server->GetConfig().GetSensorPushOptions() : Default<TMap<ui16, TSensorPushOptions>>();

    TMultiSensor sensorsToPush;
    TMultiSensor sensorsToPushDeferred;
    TVector<std::pair<TDeferredSensorStatePtr, TInstant>> statesAndTimestamps;
    for (auto&& sensor : sensors) {
        const auto pushOptionsIt = pushOptions.find(sensor.Id);
        const auto& options = pushOptionsIt != pushOptions.end() ? pushOptionsIt->second : Default<TSensorPushOptions>();
        switch (options.Policy) {
        case TSensorPushOptions::EPolicy::Deferred: {
            TDeferredSensorStatePtr state;
            TDeferredSensors::iterator deferredSensorsIt;
            {
                auto guard = Guard(DeferredSensorsLock);
                deferredSensorsIt = DeferredSensors.find(sensor);
                if (deferredSensorsIt == DeferredSensors.end()) {
                    deferredSensorsIt = DeferredSensors.emplace(sensor, MakeAtomicShared<TDeferredSensorState>(sensor, options.Period)).first;
                }
            }
            state = deferredSensorsIt->second;
            auto pushable = Checked(state)->Set(std::move(sensor));
            if (pushable) {
                statesAndTimestamps.emplace_back(state, pushable->Timestamp);
                sensorsToPushDeferred.push_back(std::move(*pushable));
            }
            break;
        }
        case TSensorPushOptions::EPolicy::Realtime:
            sensorsToPush.push_back(std::move(sensor));
            break;
        case TSensorPushOptions::EPolicy::Ignore:
            break;
        }
    }
    const auto future1 = Push<TSensor>(sensorsToPush);
    const auto future2 = PushDeferred(sensorsToPushDeferred, std::move(statesAndTimestamps));
    return NThreading::WaitAll(future1, future2);
}

NDrive::TMultiSensor NDrive::TTelematicsConnection::Derive(const NDrive::TMultiSensor& sensors, const NDrive::TSensorCalculator& calculator) const {
    NDrive::TMultiSensor result;
    for (const auto& sensor : sensors) {
        auto derived = calculator.Derive(sensor, SensorsCache);
        result.insert(
            result.end(),
            std::make_move_iterator(derived.begin()),
            std::make_move_iterator(derived.end())
        );
    }
    return result;
}

NThreading::TFuture<void> NDrive::TTelematicsConnection::PushDeferred(TArrayRef<TSensor> sensors, TVector<std::pair<TDeferredSensorStatePtr, TInstant>>&& statesAndTimestamps) const {
    auto future = Push(sensors);
    future.Subscribe([statesAndTimestamps = std::move(statesAndTimestamps)](const NThreading::TFuture<void>& result) {
        if (result.HasValue()) {
            for (auto& [state, timestamp] : statesAndTimestamps) {
                Checked(state)->Commit(timestamp);
            }
        }
    });
    return future;
}

void NDrive::TTelematicsConnection::PushDeferred() {
    TDeferredSensors states;
    {
        auto guard = Guard(DeferredSensorsLock);
        states = DeferredSensors;
    }

    auto now = TInstant::Now();
    TMultiSensor sensors;
    TVector<std::pair<TDeferredSensorStatePtr, TInstant>> statesAndTimestamps;
    for (const auto& [id, s] : states) {
        auto sensor = Checked(s)->Get(now);
        if (sensor) {
            statesAndTimestamps.emplace_back(s, sensor->Timestamp);
            sensors.push_back(std::move(*sensor));
        }
    }
    PushDeferred(sensors, std::move(statesAndTimestamps));
}

bool NDrive::TTelematicsConnection::Alive() const {
    if (Connection) {
        return Connection->Alive();
    } else {
        WARNING_LOG << DebugString() << ": connection is dead" << Endl;
        TTelematicsLog::Log(TTelematicsLog::Error, this, "DeadConnection", NJson::TMapBuilder
            ("created", CreatedTime.Seconds())
            ("heartbeat", HeartbeatTime.Seconds())
        );
        return false;
    }
}

void NDrive::TTelematicsConnection::Drop() {
    if (Connection) {
        Connection->Drop();
    }
}

void NDrive::TTelematicsConnection::AddHandler(TTelematicsHandlerPtr handler, TInstant time) {
    Y_ENSURE(handler);
    auto guard = Guard(Lock);
    Handlers.push_back(handler);
    TTelematicsUnistatSignals::Get().HandlersRegistered.Signal(1);
    TTelematicsLog::Log(TTelematicsLog::Registered, this, handler.Get());
    if (time) {
        ScheduleHandler(handler, time);
    } else {
        InitializeHandler(handler);
    }
}

void NDrive::TTelematicsConnection::ScheduleHandler(TTelematicsHandlerPtr handler, TInstant time) {
    CHECK_WITH_LOG(handler);
    CHECK_WITH_LOG(Server);
    Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<TScheduledOnAfterRegister>(Server, IMEI, handler, time)));
}

void NDrive::TTelematicsConnection::InitializeHandler(TTelematicsHandlerPtr handler) {
    CHECK_WITH_LOG(handler);
    if (handler->OnAfterRegister(this)) {
        Handle(NVega::TMessage()); // tick if the handler says it's ready
    } else {
        Push(NContainer::Scalar(handler));
    }
}

const TString& NDrive::TTelematicsConnection::GetId() const {
    return GetIMEI();
}

void NDrive::TTelematicsConnection::AddMessageHandler(NProtocol::THandlerPtr handler) {
    AddHandler(MakeIntrusive<NDrive::TWrapperHandler>(NProtocol::GenerateId(IMEI), handler));
}

void NDrive::TTelematicsConnection::SendMessage(THolder<NProtocol::IMessage>&& message) {
    Y_ENSURE(message->IsValid());

    const auto protocolType = GetProtocolType();
    Y_ENSURE(protocolType != NProtocol::PT_INVALID);

    if (protocolType == NProtocol::PT_VEGA) {
        Send(std::move(message));
    } else {
        Y_ENSURE(Protocol);
        Protocol->Command(std::move(message));
    }
}

void NDrive::TTelematicsConnection::OnSensor(ui16 id, ui16 subid, NDrive::TSensorRef value) {
    auto calculator = Server ? Server->GetSensorCalculator() : nullptr;
    auto sensor = SensorsCache.Add(id, subid, NDrive::SensorValueFromRef(value), Now());
    TMultiSensor derivedSensors;
    if (calculator && sensor) {
        derivedSensors = Derive({ *sensor }, *calculator);
    }
    // intentionally by value
    for (auto derived : derivedSensors) {
        SensorsCache.Add(std::move(derived));
    }
    auto pusher = Server ? Server->GetPusher() : nullptr;
    if (pusher && sensor) {
        PushOrDefer(NContainer::Scalar(*sensor));
        PushOrDefer(derivedSensors);
    }
    if (sensor) {
        TTelematicsLog::Log(TTelematicsLog::Sensors, this, "OnSensor", NJson::ToJson(NContainer::Scalar(*sensor)));
    }
    if (!derivedSensors.empty()) {
        TTelematicsLog::Log(TTelematicsLog::Sensors, this, "OnSensorDerived", NJson::ToJson(derivedSensors));
    }
    switch (id) {
    case VEGA_INPUT_BUFFER_SIZE:
        SetInputBufferSize(std::get<ui64>(value));
        break;
    default:
        break;
    }
    if (BLE_EXT_BOARD_BEACONS_INFO1 <= id && id <= BLE_EXT_BOARD_BEACONS_INFO10) {
        if (pusher && IMEI) {
            const auto& locator = Server->GetLocator();
            auto locationF = locator.Beacon(IMEI, SensorsCache);
            if (locationF.Initialized()) {
                auto self = NDrive::TTelematicsConnectionPtr(this);
                locationF.Subscribe([imei = IMEI, pusher, self = std::move(self)](const auto& f) {
                    if (f.HasValue()) {
                        const auto location = f.GetValue();
                        NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::Location, self.Get(), ToString(location.Type), location.ToJson());
                        ::Push(*pusher, imei, NContainer::Scalar(location));
                    }
                });
            }
        }
    }
}

TString NDrive::TTelematicsConnection::DebugString() const {
    return "Loop " + RemoteAddr + " " + IMEI;
}

TString NDrive::TTelematicsConnection::GetServerAddr() const {
    return Server ? Server->GetClientEndpoint() : "NullServerClientEndpoint";
}

THolder<NDrive::NProtocol::IMessage> NDrive::TTelematicsConnection::CreateAuthorizationRequest() const {
    auto result = MakeHolder<NVega::TMessage>(NDrive::NVega::AUTH_REQUEST);
    auto& payload = result->As<NDrive::NVega::TAuthorizationRequest>();
    payload.EnableBlackbox = NDrive::NVega::TAuthorizationRequest::BlackboxEnabled;

    for (ui8& item : payload.Password) {
        item = RandomNumber<ui8>();
    }
    auto expected = NDrive::VegaCrc32(payload.Password.cbegin(), payload.Password.cend());
    ExpectedNumber = expected;
    INFO_LOG << DebugString() << ": expect number " << expected << Endl;
    return result;
}

THolder<NDrive::NProtocol::IMessage> NDrive::TTelematicsConnection::CreateBlackboxAck(ui16 id) const {
    auto result = MakeHolder<NVega::TMessage>(NDrive::NVega::BLACKBOX_RECORDS_ACK);
    auto& payload = result->As<NDrive::NVega::TBlackboxRecordsAck>();
    payload.Id = id;
    payload.Result = payload.Received;
    return result;
}

THolder<NDrive::NProtocol::IMessage> NDrive::TTelematicsConnection::CreateCommandRequest(NDrive::NVega::TCommandRequest&& request, ui32 sequenceId/* = 0*/) {
    auto result = MakeHolder<NVega::TMessage>(NDrive::NVega::COMMAND_REQUEST);
    auto& payload = result->As<NDrive::NVega::TCommandRequest>();
    payload.Id = sequenceId ? sequenceId : ++MessageId;
    payload.Code = request.Code;
    payload.Argument = std::move(request.Argument);

    auto guard = Guard(SequenceIdsMutex);
    ExpectedSequenceIds.insert(payload.Id);
    return result;
}

THolder<NDrive::NProtocol::IMessage> NDrive::TTelematicsConnection::CreatePingRequest() const {
    return MakeHolder<NVega::TMessage>(NDrive::NVega::PING_REQUEST);
}

THolder<NDrive::NProtocol::IMessage> NDrive::TTelematicsConnection::CreateListResponse() const {
    THolder<NDrive::NProtocol::IMessage> result = MakeHolder<NDrive::NVega::TMessage>(NDrive::NVega::LIST_RESPONSE);
    auto& payload = result->As<NDrive::NVega::TListResponse>();
    NDrive::NVega::TListResponse::TDevice device;
    device.IMEI.Set("000000000000000");
    device.Name.Set("fake");
    payload.Devices.push_back(std::move(device));
    return result;
}

void NDrive::TTelematicsConnection::Process(const NVega::TAuthorizationResponse& response) {
    if (response.Status == response.Failure) {
        WARNING_LOG << DebugString() << ": authorization failed" << Endl;
        Drop();
        return;
    }

    if (response.Status == response.Encrypt) {
        auto expected = ExpectedNumber.load();
        if (expected != response.EncryptedData) {
            WARNING_LOG << DebugString() << ": encrypted check failed " << expected << " " << response.EncryptedData << Endl;
            Drop();
            return;
        }
    }

    if (RegisterIMEI()) {
        AddHandler(MakeIntrusive<TGetParameterTask>("get_input_buffer_size", ui16(VEGA_INPUT_BUFFER_SIZE)));
    }
}

void NDrive::TTelematicsConnection::Process(const NVega::TPingResponse& response) {
    const auto& imei = response.IMEI.Get();
    SetIMEI(imei);

    auto pusher = Server ? Server->GetPusher() : nullptr;
    if (pusher) {
        THeartbeat heartbeat(HeartbeatTime, GetServerAddr());
        heartbeat.Created = CreatedTime;
        Push(NContainer::Scalar(heartbeat));
    }
}

void NDrive::TTelematicsConnection::Process(const NVega::TBlackboxRecords& response) {
    auto start = Now();
    auto sensors = SensorsCache.Add(response);
    if (response.Records.size() > 1) {
        Prepare(sensors);
    }

    ProcessCache(std::move(sensors));

    for (auto&& record : response.Records) {
        TInstant timestamp = TInstant::Seconds(record.Timestamp);
        UpdateAge(start, timestamp);
    }
}

void NDrive::TTelematicsConnection::UpdateAge(TInstant start, TInstant timestamp) {
    TDuration age = start - timestamp;
    BlackboxTime = timestamp;
    TTelematicsUnistatSignals::Get().BlackboxAges.Signal(age.MilliSeconds());
}

void NDrive::TTelematicsConnection::Prepare(TMultiSensor& sensors) {
    std::sort(sensors.begin(), sensors.end(), [](const NDrive::TSensor& left, const NDrive::TSensor& right) {
        return std::tie(left.Id, left.SubId, left.Timestamp) > std::tie(right.Id, right.SubId, right.Timestamp);
    });
    sensors.erase(
        std::unique(sensors.begin(), sensors.end()),
        sensors.end()
    );
}

void NDrive::TTelematicsConnection::ProcessCache(TMultiSensor&& sensors) {
    auto pusher = Server ? Server->GetPusher() : nullptr;
    auto calculator = Server ? Server->GetSensorCalculator() : nullptr;

    TMultiSensor derivedSensors;
    if (calculator) {
        derivedSensors = Derive(sensors, *calculator);
    }
    derivedSensors = SensorsCache.Add(std::move(derivedSensors));
    if (IMEI) {
        auto self = NDrive::TTelematicsConnectionPtr(this);
        auto callback = [pusher, locations = Locations, imei = IMEI, self = std::move(self)](const NThreading::TFuture<TLocation>& l) {
          try {
            const auto [location, status] = Validate(MakeCopy(l.GetValue()), self->GetSensorsCache());
            NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::Location, self.Get(), ToString(location.Type), location.ToJson());
            if (status != EDataValidationStatus::Ok) {
                auto serialized = location.ToJson();
                auto debugString = serialized.GetStringRobust();
                TTelematicsLog::Log(TTelematicsLog::ValidationError, self.Get(), TypeName(location), NJson::TMapBuilder
                    ("object", std::move(serialized))
                    ("status", ToString(status))
                );
                return;
            }
            if (pusher) {
                ::Push(*pusher, imei, NContainer::Scalar(location));
            } else {
                INFO_LOG << imei << " location: " << location.ToJson().GetStringRobust() << Endl;
            }
            if (locations) {
                locations->AddLocation(location);
            }
          } catch (const std::exception& e) {
            WARNING_LOG << imei << ": an exception has occurred during location processing " << FormatExc(e) << Endl;
          }
        };
        auto locations = NDrive::TLocator::TAsyncLocations();
        if (Server) {
            const auto& locator = Server->GetLocator();
            locations = locator.LocateAll(IMEI, SensorsCache, Locations.Get());
        }
        for (auto&& location : locations) {
            location.Subscribe(MakeCopy(callback));
        }
        if (pusher) {
            {
                THeartbeat heartbeat(HeartbeatTime, GetServerAddr());
                heartbeat.Created = CreatedTime;
                Push(NContainer::Scalar(heartbeat));
            }
            PushOrDefer(sensors);
            PushOrDefer(derivedSensors);
        }
    }
    if (!sensors.empty()) {
        TTelematicsLog::Log(TTelematicsLog::Sensors, this, "OnCache", NJson::ToJson(sensors));
    }
    if (!derivedSensors.empty()) {
        TTelematicsLog::Log(TTelematicsLog::Sensors, this, "OnCacheDerived", NJson::ToJson(derivedSensors));
    }
}

void NDrive::TTelematicsConnection::Process(const NVega::TCommandResponse& response) {
    const auto sequenceId = response.GetSequenceId();
    auto guard = Guard(SequenceIdsMutex);
    if (!ExpectedSequenceIds.contains(sequenceId)) {
        WARNING_LOG << DebugString() << ": got unexpected sequenceId " << sequenceId << Endl;
        TTelematicsUnistatSignals::Get().MessagesOrphans.Signal(1);
    }
}

void NDrive::TTelematicsConnection::Process(TInstant timestamp, TInstant start, const NDrive::NWialon::TShortData& data) {
    auto sensors = SensorsCache.Add(data);
    ProcessCache(std::move(sensors));
    UpdateAge(start, timestamp);
}

void NDrive::TTelematicsConnection::Process(TInstant timestamp, TInstant start, const NDrive::NWialon::TAdditionalData& data) {
    auto sensors = SensorsCache.Add(timestamp, data);
    ProcessCache(std::move(sensors));
    UpdateAge(start, timestamp);
}

void NDrive::TTelematicsConnection::Process(TInstant start, const TVector<NDrive::NNavTelecom::TRecord>& records) {
    for (auto&& record : records) {
        Process(start, record);
    }
}

void NDrive::TTelematicsConnection::Process(TInstant start, const TVector<NDrive::NNavTelecom::TAdditional>& records) {
    for (auto&& record: records) {
        Process(start, record);
    }
}

void NDrive::TTelematicsConnection::Process(const NDrive::NNavTelecom::TICCIDAnswer& data) {
    auto sensors = SensorsCache.Add(data);
    ProcessCache(std::move(sensors));
}

void NDrive::TTelematicsConnection::Process(TInstant start, const NDrive::NNavTelecom::TRecord& record) {
    auto sensors = SensorsCache.Add(record);

    ProcessCache(std::move(sensors));

    auto timestampValue = record.Get(NNavTelecom::EParameterId::PI_TIMESTAMP);

    if (std::holds_alternative<ui64>(timestampValue)) {
        TInstant timestamp = TInstant::Seconds(std::get<ui64>(timestampValue));
        UpdateAge(start, timestamp);
    }
}

void NDrive::TTelematicsConnection::Process(TInstant start, const NDrive::NNavTelecom::TAdditional& record) {
    auto sensors = SensorsCache.Add(record);

    ProcessCache(std::move(sensors));

    auto timestamp = TInstant::Seconds(record.Timestamp);
    UpdateAge(start, timestamp);
}

void NDrive::TTelematicsConnection::Process(THolder<NDrive::NProtocol::IMessage>&& message) {
    if (!message) {
        ERROR_LOG << DebugString() << ": message mismatch" << Endl;
        return;
    }

    if (!message->IsValid()) {
        ERROR_LOG << DebugString() << ": try process empty message" << Endl;
        return;
    }

    auto start = Now();

    TTelematicsLog::Log(TTelematicsLog::Incoming, this, *message);
    if (message->GetProtocolType() == NDrive::NProtocol::PT_WIALON_IPS) {
        auto type = static_cast<NWialon::EMessageType>(message->GetMessageType());

        if (type == NWialon::MT_LOGIN_REQUEST) {
            auto& payload = message->As<NWialon::TLoginRequest>();
            SetIMEI(payload.IMEI.Get());
            if (!RegisterIMEI()) {
                ERROR_LOG << DebugString() << ": error register imei " << IMEI << Endl;
                Drop();
            }
        } else if (type == NWialon::MT_SHORT_DATA_REQUEST) {
            auto& payload = message->As<NWialon::TShortDataRequest>();
            auto timestamp = payload.ShortData.DateTime.Get().GetOrElse(Now());
            Process(timestamp, start, payload.ShortData);
        } else if (type == NWialon::MT_DATA_REQUEST) {
            auto& payload = message->As<NWialon::TDataRequest>();
            auto timestamp = payload.ShortData.DateTime.Get().GetOrElse(Now());
            Process(timestamp, start, payload.ShortData);
            Process(timestamp, start, payload.AdditionalData);
        } else if (type == NWialon::MT_BLACK_BOX_REQUEST) {
            auto& payload = message->As<NWialon::TBlackBoxRequest>();
            for (auto&& item : payload.Data.Get()) {
                auto timestamp = item.ShortData.DateTime.Get().GetOrElse(Now());
                Process(timestamp, start, item.ShortData);
                Process(timestamp, start, item.AdditionalData);
            }
        }
    } else if (message->GetProtocolType() == NDrive::NProtocol::PT_NAVTELECOM) {
        auto type = static_cast<NNavTelecom::EMessageType>(message->GetMessageType());

        if (type == NNavTelecom::MT_HANDSHAKE_REQUEST) {
            auto& payload = message->As<NNavTelecom::THandShakeRequest>();
            SetIMEI(ToString(payload.IMEI.Get()));
            if (!RegisterIMEI()) {
                ERROR_LOG << DebugString() << ": error register imei " << IMEI << Endl;
                Drop();
            }
        } else if (type == NNavTelecom::MT_BLACKBOX_REQUEST) {
            auto& payload = message->As<NNavTelecom::TBlackBoxRequest>();
            Process(start, payload.Records);
        } else if (type == NNavTelecom::MT_ADDITIONAL_BLACKBOX_REQUEST) {
            auto& payload = message->As<NNavTelecom::TAdditionalBlackBoxRequest>();
            Process(start, payload.Records);
        } else if (type == NNavTelecom::MT_SINGLE_BLACKBOX_REQUEST) {
            auto& payload = message->As<NNavTelecom::TSingleBlackBoxRequest>();
            Process(start, payload.Record);
        } else if (type == NNavTelecom::MT_ADDITIONAL_SINGLE_BLACKBOX_REQUEST) {
            auto& payload = message->As<NNavTelecom::TAdditionalSingleBlackBoxRequest>();
            Process(start, payload.Record);
        } else if (type == NNavTelecom::MT_ONLINE_BLACKBOX_REQUEST) {
            auto& payload = message->As<NNavTelecom::TOnlineBlackBoxRequest>();
            Process(start, payload.Record);
        } else if (type == NNavTelecom::MT_ICCID_ANSWER) {
            auto& payload = message->As<NNavTelecom::TICCIDAnswer>();
            Process(payload);
        }
    } else if (message->GetProtocolType() == NDrive::NProtocol::PT_VEGA) {
        using namespace NDrive::NVega;

        TTelematicsUnistatSignals::Get().IncomingMessages.Signal(message->GetMessageTypeAs<EMessageType>(), 1);

        if (message->GetProtocolType() == NProtocol::PT_VEGA) {
        switch (message->GetMessageType()) {
            case AUTH_RESPONSE: {
                Process(message->As<TAuthorizationResponse>());
                break;
            }
            case PING_RESPONSE: {
                Process(message->As<TPingResponse>());
                break;
            }
            case BLACKBOX_RECORDS: {
                const auto sequenceId = message->GetSequenceId();
                Process(message->As<TBlackboxRecords>());
                Send(CreateBlackboxAck(sequenceId));
                break;
            }
            case LIST_REQUEST: {
                Send(CreateListResponse());
                break;
            }
            case COMMAND_RESPONSE: {
                Process(message->As<TCommandResponse>());
                break;
            }
            default:
                break;
            }
        }
    }

    Handle(*message);

    if (message) {
        DEBUG_LOG << DebugString() << ": processed " << message->DebugString() << Endl;
    }

    if (message->GetProtocolType() != NDrive::NProtocol::PT_VEGA) {
        Protocol->Process(*message);
    }
}

void NDrive::TTelematicsConnection::Launch() {
    CurrentProtocolType = Server->GetConfig().GetProtocolType();

    if (!Connection) {
        ERROR_LOG << DebugString() << ": null connection" << Endl;
        return;
    }

    INFO_LOG << DebugString() << ": launching Loop" << Endl;
    TIntrusivePtr<NDrive::TTelematicsConnection> self(this);
    Connection->Read([self](const TBuffer& buffer) {
        self->Loop(buffer);
    });

    auto protocolType = GetProtocolType();
    if (protocolType != NProtocol::PT_VEGA) {
        if (protocolType != NProtocol::PT_AUTODETECT) {
            CreateProtocol(protocolType);
        } else {
            CurrentProtocolType = NProtocol::PT_AUTODETECT;
        }
    } else {
        CurrentProtocolType = NProtocol::PT_VEGA;
        self->Send(CreatePingRequest());
        self->Send(CreateAuthorizationRequest());
    }
    DEBUG_LOG << DebugString() << ": launched Loop" << Endl;
}

void NDrive::TTelematicsConnection::Fuzz(const ui8* data, size_t size) {
    Loop(reinterpret_cast<const char*>(data), size);
}

void NDrive::TTelematicsConnection::Loop(const TBuffer& buffer) noexcept {
    Loop(buffer.data(), buffer.size());
}

void NDrive::TTelematicsConnection::Loop(const char* data, size_t length) noexcept {
    TTelematicsUnistatSignals::Get().BytesReceived.Signal(length);

    const char* begin = nullptr;
    size_t size = 0;
    bool bufferIsEmpty = false;
    if (Buffer.Empty()) {
        begin = data;
        size = length;
        bufferIsEmpty = true;
    } else {
        Buffer.Append(data, length);
        begin = Buffer.Data();
        size = Buffer.Size();
    }
    const char* end = begin + size;

    if (Server->GetConfig().GetTraceInputTraffic()) {
        Trace(ETraceDirection::Input, begin, size);
    }

    if (GetProtocolType() == NProtocol::PT_AUTODETECT && !Protocol) {
        TBuffer buffer(begin, size);
        auto protocolType = NProtocol::TProtocolMeta::TryDetect(buffer);
        CurrentProtocolType = protocolType;

        if (protocolType == NProtocol::PT_INVALID) {
            ERROR_LOG << DebugString() << ": unexpected data to detect protocol " << HexEncode(data, size) << Endl;
            Drop();
            return;
        }

        CreateProtocol(protocolType);
        INFO_LOG << DebugString() << ": detect protocol " << ToString(protocolType) << Endl;
    }

    TMemoryInput input(begin, size);
    NProtocol::TMessageInput messages(input);
    while (!input.Exhausted()) {
        const auto start = input.Buf();
        try {
            HeartbeatTime = Now();

            THolder<NProtocol::IMessage> message;
            if (GetProtocolType() == NProtocol::PT_VEGA) {
                message = MakeHolder<NVega::TMessage>();
            } else if (GetProtocolType() == NProtocol::PT_NAVTELECOM) {
                const auto& protocol = Protocol->As<NNavTelecom::TNavTelecomProtocol>();
                message = MakeHolder<NNavTelecom::TMessage>(NNavTelecom::MT_INCORRECT, NNavTelecom::TMessage::ESource::Server, protocol.GetBitField());
            } else if (GetProtocolType() == NProtocol::PT_WIALON_IPS) {
                message = MakeHolder<NWialon::TMessage>();
            } else if (!Protocol) {
                ERROR_LOG << DebugString() << ": protocol invalid" << Endl;
                Drop();
                return;
            }
            messages.Get(*message);
            if (!message->IsValid()) {
                WARNING_LOG << DebugString() << ": invalid message" << Endl;
                break;
            }
            Process(std::move(message));

        } catch (const NDrive::TIncompleteDataException& e) {
            WARNING_LOG << DebugString() << ": incomplete data " << e.GetExpected() << " expected " << e.GetReceived() << " received" << Endl;
            CHECK_WITH_LOG(start <= end);
            Buffer.Assign(start, end);
            return;
        } catch (const std::exception& e) {
            const TString& bytes = HexEncode(begin, size);
            const TString& error = FormatExc(e);
            ERROR_LOG << DebugString() << ": " << bytes << " buffered " << bufferIsEmpty << " " << error << Endl;
            {
                NJson::TJsonValue data;
                data["bytes"] = bytes;
                data["error"] = error;
                TTelematicsLog::Log(TTelematicsLog::Error, this, "ParsingError", std::move(data));
            }
            auto dropErrorsCount = Server ? Server->GetConfig().GetDropErrorsCount() : 42;
            if (++ErrorsCount > dropErrorsCount) {
                WARNING_LOG << DebugString() << ": dropping due to excess errors count " << ErrorsCount.load() << Endl;
                Drop();
            }
            TTelematicsUnistatSignals::Get().MessagesErrors.Signal(1);
        }
    }
    Buffer.Clear();
}

void NDrive::TTelematicsConnection::Handle(const NProtocol::IMessage& message) {
    if (auto guard = Guard(Lock)) {
        TTelematicsHandlers survivors;
        survivors.reserve(Handlers.size());
        TTelematicsHandlers handlersToPush;
        handlersToPush.reserve(Handlers.size());
        for (auto&& handler : Handlers) {
            bool finished = false;
            try {
                CHECK_WITH_LOG(handler);
                finished = handler->OnMessage(this, message);
            } catch (const std::exception& e) {
                ERROR_LOG << DebugString() << ": handler exception " << FormatExc(e) << Endl;
            }
            if (handler->Serializable()) {
                handlersToPush.push_back(handler);
            }
            if (!finished) {
                survivors.push_back(std::move(handler));
            } else {
                TTelematicsUnistatSignals::Get().HandlersDeregistered.Signal(1);
                TTelematicsLog::Log(TTelematicsLog::Deregistered, this, handler.Get());
            }
        }
        Push(handlersToPush);
        Handlers = std::move(survivors);
    }
}

void NDrive::TTelematicsConnection::Send(THolder<NProtocol::IMessage>&& message) {
    Y_ENSURE(message);
    TTelematicsUnistatSignals::Get().OutgoingMessages.Signal(message->GetMessageTypeAs<NVega::EMessageType>(), 1);
    Send(*message);
}

void NDrive::TTelematicsConnection::Send(const NProtocol::IMessage& message) {
    auto guard = Guard(SendMutex);

    if (!message.IsValid()) {
        ERROR_LOG << DebugString() << ": try sent empty message" << Endl;
        return;
    }

    DEBUG_LOG << DebugString() << ": sending IMessage " << message.DebugString() << Endl;

    TTelematicsLog::Log(TTelematicsLog::Outcoming, this, message);

    NUtil::TTcpServer::TConnectionOutput co(Connection);
    TFlushBufferedOutputStream output(co, InputBufferSize.load(std::memory_order_relaxed));
    NProtocol::TMessageOutput messages(output);
    messages.Send(message);

    if (Server->GetConfig().GetTraceInputTraffic()) {
        TString dataTrace;
        TStringOutput outputTrace(dataTrace);
        message.Save(outputTrace);

        Trace(ETraceDirection::Output, dataTrace.Data(), dataTrace.Size());
    }

    TTelematicsUnistatSignals::Get().BytesSent.Signal(co.GetBytesSent());
    DEBUG_LOG << DebugString() << ": sent " << message.DebugString() << Endl;
}

void NDrive::TTelematicsConnection::Register() noexcept {
    if (Server) {
        Server->Register(this);
    }
}

void NDrive::TTelematicsConnection::Deregister() noexcept {
    if (Server) {
        Server->Deregister(this);
    }
}

static void LogRecord(const NDrive::TTelematicsConnection* self, const NDrive::NVega::TLogRecord& record) {
    NJson::TJsonValue data;
    data["timestamp"] = record.Timestamp.Seconds();
    data["message"] = record.Message;
    NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::LogRecord, self, "AddLogRecord", std::move(data));
}

void NDrive::TTelematicsConnection::AddLogRecord(NVega::TLogRecord&& record) {
    {
        LogRecord(this, record);
    }
    auto guard = Guard(Lock);
    if (LogRecords.empty()) {
        LogRecords.push_back(std::move(record));
        return;
    }
    if (LogRecords.back() < record) {
        LogRecords.push_back(std::move(record));
    } else if (record < LogRecords.back()) {
        LogRecords.push_back(std::move(record));
        std::sort(LogRecords.begin(), LogRecords.end());
        LogRecords.erase(
            std::unique(LogRecords.begin(), LogRecords.end()),
            LogRecords.end()
        );
    }
}

void NDrive::TTelematicsConnection::AddLogRecords(NDrive::NVega::TLogRecords&& records) {
    for (auto&& record : records) {
        LogRecord(this, record);
    }
    auto guard = Guard(Lock);
    if (LogRecords.empty()) {
        LogRecords = std::move(records);
        return;
    }
    for (auto&& record : records) {
        LogRecords.push_back(std::move(record));
    }
    std::sort(LogRecords.begin(), LogRecords.end());
    LogRecords.erase(
        std::unique(LogRecords.begin(), LogRecords.end()),
        LogRecords.end()
    );
}

void NDrive::TTelematicsConnection::SetIMEI(const TString& imei) {
    auto guard = Guard(Lock);
    if (!IMEI && imei) {
        IMEI = imei;
    } else if (IMEI != imei) {
        ERROR_LOG << DebugString() << ": IMEI mismatch " << IMEI << " " << imei << Endl;
    }
}

bool NDrive::TTelematicsConnection::RegisterIMEI() {
    auto guard = Guard(Lock);

    if (IMEI && !Registered.load()) {
        Register();
        Registered = true;

        RestoreState();
        if (Server && Server->GetConfig().GetTickInterval()) {
            bool scheduled = TGlobalScheduler::Schedule(MakeHolder<TScheduledHandlerTick>(Server, IMEI, Now()));
            if (!scheduled) {
                ERROR_LOG << DebugString() << ": cannot schedule ticker" << Endl;
            }
        }
        TTelematicsLog::Log(TTelematicsLog::Created, this);
        return true;
    } else if (!IMEI) {
        ERROR_LOG << DebugString() << ": try register empty IMEI" << Endl;
    }
    return false;
}

void NDrive::TTelematicsConnection::RestoreState() {
    if (auto api = Server ? Server->GetSensors() : nullptr) {
        auto deadline = TInstant::Now() + TDuration::Seconds(1);
        auto metadataClient = Server->GetMetadataClient();
        auto asyncHandlers = metadataClient ? metadataClient->GetActiveHandlers(IMEI) : NThreading::MakeFuture<TTelematicsMetadataClient::THandlers>();
        auto asyncMultiSensors = api->GetSensors(IMEI);
        auto asyncLocation = api->GetLocation(IMEI);

        if (!asyncMultiSensors.Wait(deadline)) {
            ERROR_LOG << DebugString() << ": could not wait for sensors request" << Endl;
        }
        if (asyncMultiSensors.HasValue()) {
            SensorsCache.Add(asyncMultiSensors.ExtractValue());
        }
        if (asyncMultiSensors.HasException()) {
            ERROR_LOG << DebugString() << ": an exception occurred in sensors request " << NThreading::GetExceptionMessage(asyncMultiSensors) << Endl;
        }

        if (!asyncLocation.Wait(deadline)) {
            ERROR_LOG << DebugString() << ": could not wait for location request" << Endl;
        }
        if (asyncLocation.HasValue()) {
            auto location = asyncLocation.ExtractValue();
            if (location) {
                NDrive::TSensor base;
                base.Timestamp = location->Timestamp;
                base.Since = location->Since;

                auto latitude = base;
                latitude.Value = location->Latitude;
                SensorsCache.Add(std::move(latitude));

                auto longitude = base;
                longitude.Value = location->Longitude;
                SensorsCache.Add(std::move(longitude));

                auto course = base;
                course.Value = static_cast<double>(location->Course);
                SensorsCache.Add(std::move(course));
            }
        }
        if (asyncLocation.HasException()) {
            ERROR_LOG << DebugString() << ": an exception occurred in location request " << NThreading::GetExceptionMessage(asyncLocation) << Endl;
        }

        if (!asyncHandlers.Wait(deadline)) {
            ERROR_LOG << DebugString() << ": could not wait for handlers request" << Endl;
        }
        if (asyncHandlers.HasValue()) {
            for (auto&& i : asyncHandlers.GetValue()) {
                try {
                    auto handler = TCommonTask::Restore(i.Data);
                    Y_ENSURE(handler, "nullptr CommonTask");
                    Handlers.push_back(handler);
                    Server->AddTask(handler);

                    auto expectedSequenceIds = handler->GetExpectedSequenceIds();
                    ExpectedSequenceIds.insert(expectedSequenceIds.begin(), expectedSequenceIds.end());
                    INFO_LOG << DebugString() << ": restored handler " << handler->GetId() << Endl;
                } catch (const std::exception& e) {
                    ERROR_LOG << DebugString() << ": cannot restore handler " << i.Id << " " << FormatExc(e) << Endl;
                }
            }
        }
        if (asyncHandlers.HasException()) {
            ERROR_LOG << DebugString() << ": an exception occurred during handlers request " << NThreading::GetExceptionMessage(asyncHandlers);
        }
    }
}

void NDrive::TTelematicsConnection::SetInputBufferSize(size_t size) {
    if (size) {
        INFO_LOG << DebugString() << ": set InputBuffer size " << size << Endl;
        InputBufferSize = size;
    } else {
        WARNING_LOG << DebugString() << ": trying to set zero InputBuffer" << Endl;
    }
}

void NDrive::TTelematicsConnection::Trace(ETraceDirection direction, const char* data, size_t size) {
    if (!data || size == 0) {
        return;
    }

    TString message = ": trace ";

    if (direction == ETraceDirection::Input) {
        message += "input ";
    } else if (direction == ETraceDirection::Output) {
        message += "output ";
    }

    INFO_LOG << DebugString() << message << HexEncode(data, size) << Endl;
    TTelematicsLog::Log(TTelematicsLog::Incoming, this, data, size);
}

void NDrive::TTelematicsConnection::CreateProtocol(NDrive::NProtocol::EProtocolType protocolType) {
    if (protocolType == NProtocol::PT_INVALID) {
        ERROR_LOG << DebugString() << ": try create invalid protocol" << Endl;
        Drop();
        return;
    }
    auto sender = [this](const NDrive::NProtocol::IMessage& message) {
        Send(message);
    };

    Protocol = ::CreateProtocol(protocolType, sender, Server);

    if (Protocol) {
        Protocol->Launch();
        CurrentProtocolType = protocolType;
    } else {
        ERROR_LOG << DebugString() << ": error create protocol instance" << Endl;
        Drop();
    }
}

NDrive::NProtocol::TTaskPtr NDrive::TTelematicsConnection::CreateCommand(const TString& id, NProtocol::ECommandCode command, NProtocol::TArgument argument, const NProtocol::TCommandOptions& options) {
    const auto protocolType = GetProtocolType();
    Y_ENSURE(protocolType != NProtocol::PT_INVALID);

    if (protocolType == NProtocol::PT_VEGA) { // will delete
        return NDrive::CreateCommand(id, command, argument, options);
    }

    Y_ENSURE(Protocol);
    return Protocol->CreateCommand(id, command, argument, options);
}

void NDrive::TTelematicsConnection::RegisterSensorsSignals() {
    for (const auto sensorId : NDrive::NVega::GetSensorIds()) {
        SensorsPushSignals.RegisterSignal(sensorId, MakeHolder<NDrive::TSensorsPushSignal>(TTelematicsUnistatSignals::GetSensorsPushSignalName(sensorId)));
    }
}

void NDrive::TTelematicsConnection::RegisterSensorsFallback() {
    SensorsPushSignals.RegisterFallback(MakeHolder<NDrive::TSensorsPushSignal>("incorrect_sensor"));
}
