#pragma once

#include "weak_intrusive_ptr.h"

#include <drive/telematics/server/location/cache.h>
#include <drive/telematics/server/pusher/interface.h>
#include <drive/telematics/server/sensors/cache.h>

#include <drive/telematics/common/handler.h>
#include <drive/telematics/protocol/log.h>
#include <drive/telematics/protocol/protocol.h>
#include <drive/telematics/protocol/sensor.h>

#include <drive/library/cpp/network/tcp/server.h>

#include <library/cpp/threading/future/fwd.h>

#include <rtline/library/unistat/signals.h>

#include <util/datetime/base.h>
#include <util/generic/buffer.h>
#include <util/generic/hash_set.h>
#include <util/generic/string.h>
#include <util/system/mutex.h>

namespace NDrive {
    namespace NVega {
        class TMessage;
        class TCommandRequest;

        class TAuthorizationResponse;
        class TAuthorizationRequest;
        class TBlackboxRecords;
        class TPingResponse;
        class TCommandResponse;
    }

    namespace NNavTelecom {
        class TRecord;
        class TAdditional;
    }

    class TSensorCalculator;
    class TTelematicsServer;
    class TTelematicsConnection;
    using TTelematicsConnectionPtr = TIntrusivePtr<TTelematicsConnection>;

    class ITelematicsHandler
        : public IHandlerDescription
        , public TAtomicRefCount<ITelematicsHandler>
    {
    public:
        virtual ~ITelematicsHandler() = default;

        virtual bool OnAfterRegister(TTelematicsConnection* connection) = 0;
        virtual bool OnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) = 0;
        virtual void OnTermination() = 0;

        virtual bool Serializable() const {
            return false;
        }
    };

    using TTelematicsHandlerPtr = TIntrusivePtr<ITelematicsHandler>;
    using TTelematicsHandlers = TVector<TTelematicsHandlerPtr>;

    class TTelematicsConnection
        : public NProtocol::IConnection
        , public TWeakAtomicRefCount<TTelematicsConnection>
    {
    public:
        static TTelematicsConnectionPtr Create(TTelematicsServer* server, NUtil::TTcpServer::TConnectionPtr connection);

    private:
        TTelematicsConnection(TTelematicsServer* server, NUtil::TTcpServer::TConnectionPtr connection);
        TTelematicsConnection(const TTelematicsConnection& rhs) = delete;

    public:
        ~TTelematicsConnection();

        TInstant GetCreatedTime() const {
            return CreatedTime;
        }
        TInstant GetBlackboxTime() const {
            return BlackboxTime;
        }
        TInstant GetHeartbeatTime() const {
            return HeartbeatTime;
        }
        size_t GetInputBufferSize() const {
            return InputBufferSize;
        }
        const TString& GetIMEI() const {
            return IMEI;
        }
        const TString& GetRemoteAddr() const {
            return RemoteAddr;
        }

        TLocationCacheConstPtr GetLocations() const {
            return Locations;
        }
        const TSensorsCache& GetSensorsCache() const {
            return SensorsCache;
        }
        TSensorsCache& GetSensorsCache() {
            return SensorsCache;
        }

        NVega::TLogRecords GetLogRecords() {
            auto guard = Guard(Lock);
            return LogRecords;
        }
        void AddLogRecord(NVega::TLogRecord&& record);
        void AddLogRecords(NVega::TLogRecords&& records);

        bool Alive() const;
        void Drop() override;

        void AddHandler(TTelematicsHandlerPtr handler, TInstant time = TInstant::Zero());
        void ScheduleHandler(TTelematicsHandlerPtr handler, TInstant time);

        virtual const TString& GetId() const override;
        virtual void AddMessageHandler(NProtocol::THandlerPtr handler) override;
        virtual void SendMessage(THolder<NProtocol::IMessage>&& message) override;
        void OnSensor(ui16 id, ui16 subid, TSensorRef value);

        TString DebugString() const;
        TString GetServerAddr() const;

        NDrive::NProtocol::EProtocolType GetProtocolType() const {
            return CurrentProtocolType.load(std::memory_order_relaxed);
        }

        NProtocol::TTaskPtr CreateCommand(const TString& id, NProtocol::ECommandCode command, NProtocol::TArgument argument, const NProtocol::TCommandOptions& options);
        THolder<NProtocol::IMessage> CreateAuthorizationRequest() const;
        THolder<NProtocol::IMessage> CreateBlackboxAck(ui16 id) const;
        THolder<NProtocol::IMessage> CreateCommandRequest(NVega::TCommandRequest&& request, ui32 sequenceId = 0);
        THolder<NProtocol::IMessage> CreatePingRequest() const;
        THolder<NProtocol::IMessage> CreateListResponse() const;

        void Fuzz(const ui8* data, size_t size);

        inline static TSignalByKey<TSensorId, double> SensorsPushSignals;

    private:
        class TScheduledHandlerTick;
        class TScheduledOnAfterRegister;

        class TDeferredSensorState;
        using TDeferredSensorStatePtr = TAtomicSharedPtr<TDeferredSensorState>;
        using TDeferredSensors = TMap<NDrive::TSensorId, TDeferredSensorStatePtr>;

    private:
        enum class ETraceDirection {
            Input,
            Output
        };

    private:
        void Launch();
        void Loop(const TBuffer& buffer) noexcept;
        void Loop(const char* data, size_t length) noexcept;

        void InitializeHandler(TTelematicsHandlerPtr handler);

        void Handle(const NProtocol::IMessage& message);
        void Send(THolder<NProtocol::IMessage>&& message);
        void Send(const NProtocol::IMessage& message);

        void Register() noexcept;
        void Deregister() noexcept;

        void Process(const NVega::TAuthorizationResponse& response);
        void Process(const NVega::TPingResponse& response);
        void Process(const NVega::TBlackboxRecords& response);
        void Process(const NVega::TCommandResponse& response);
        void Process(THolder<NDrive::NProtocol::IMessage>&& message);
        void Process(TInstant start, const NDrive::NWialon::TShortData& shortData);
        void Process(TInstant timestamp, TInstant start, const NDrive::NWialon::TShortData& data);
        void Process(TInstant timestamp, TInstant start, const NDrive::NWialon::TAdditionalData& data);
        void Process(const NDrive::NNavTelecom::TICCIDAnswer& data);
        void Process(TInstant start, const TVector<NDrive::NNavTelecom::TRecord>& records);
        void Process(TInstant start, const TVector<NDrive::NNavTelecom::TAdditional>& records);
        void Process(TInstant start, const NDrive::NNavTelecom::TRecord& record);
        void Process(TInstant start, const NDrive::NNavTelecom::TAdditional& record);

        void Prepare(TMultiSensor& sensors);
        void ProcessCache(TMultiSensor&& sensors);
        void UpdateAge(TInstant start, TInstant timestamp);

        void RestoreState();
        void SetIMEI(const TString& imei);
        bool RegisterIMEI();
        void SetInputBufferSize(size_t size);

        template <class T>
        NThreading::TFuture<void> Push(TArrayRef<T> dataArray) const;
        NThreading::TFuture<void> Push(TArrayRef<TTelematicsHandlerPtr> handlers) const;
        NThreading::TFuture<void> PushOrDefer(TArrayRef<TSensor> sensors);
        NThreading::TFuture<void> PushDeferred(TArrayRef<TSensor> sensors, TVector<std::pair<TDeferredSensorStatePtr, TInstant>>&& statesAndTimestamps) const;
        NDrive::TMultiSensor Derive(const NDrive::TMultiSensor& sensor, const NDrive::TSensorCalculator& calculator) const;
        void PushDeferred();

        void Trace(ETraceDirection direction, const char* data, size_t size);

        void CreateProtocol(NDrive::NProtocol::EProtocolType protocolType);

        static void RegisterSensorsSignals();
        static void RegisterSensorsFallback();

    private:
        TTelematicsServer* Server;
        NUtil::TTcpServer::TConnectionPtr Connection;
        TString RemoteAddr;
        TString IMEI;
        TInstant CreatedTime;
        TInstant BlackboxTime;
        TInstant HeartbeatTime;

        TBuffer Buffer;
        TTelematicsHandlers Handlers;
        TMutex Lock;

        TDeferredSensors DeferredSensors;
        TMutex DeferredSensorsLock;

        TLocationCachePtr Locations;
        TSensorsCache SensorsCache;
        NVega::TLogRecords LogRecords;

        std::atomic<ui32> MessageId;
        std::atomic<size_t> InputBufferSize;
        std::atomic<ui64> ErrorsCount;
        mutable std::atomic<ui32> ExpectedNumber;
        std::atomic<bool> Registered;
        std::atomic<NDrive::NProtocol::EProtocolType> CurrentProtocolType;

        TMutex SendMutex;
        TMutex SequenceIdsMutex;
        THashSet<ui64> ExpectedSequenceIds;
        THolder<NProtocol::IProtocol> Protocol;
    };
}
