#include "fast_data.h"

#include "handlers.h"
#include "server.h"

#include <drive/telematics/server/common/settings.h>

#include <drive/telematics/common/firmware.h>

#include <rtline/util/algorithm/ptr.h>

#include <util/digest/fnv.h>

NDrive::TFastDataScheduler::TFastDataScheduler(const NDrive::TTelematicsServer& server, const TString& imei, TInstant now, TDuration interval)
    : TGlobalScheduler::TScheduledItem<TFastDataScheduler>(server.Name(), "fast_data_scheduler:" + imei, now)
    , Server(server)
    , IMEI(imei)
    , Interval(interval)
{
}

void NDrive::TFastDataScheduler::Process(void* /*threadSpecificResource*/) {
    auto cleanup = Hold(this);
    auto storage = Server.GetStorage();
    if (!storage) {
        ERROR_LOG << GetId() << ": storage is missing" << Endl;
        return;
    }

    auto connection = Server.GetConnection(IMEI);
    if (!connection || !connection->Alive()) {
        return;
    }

    const auto& sensorsCache = connection->GetSensorsCache();
    const auto& settings = Server.GetDynamicSettings();
    bool enabled = false;
    if (!enabled) {
        TString value;
        storage->GetValue("imei:" + IMEI + ":enable_fast_data", value);
        enabled = IsTrue(value);
    }
    if (!enabled) {
        auto fraction = settings.Get<float>("fast_data:fraction").GetOrElse(0);
        auto hash = FnvHash<ui32>(IMEI) % 10000;
        auto engineOn = sensorsCache.Get(CAN_ENGINE_IS_ON);
        enabled = engineOn && engineOn->ConvertTo<bool>() && (hash < fraction * 10000);
    }
    if (!enabled) {
        DEBUG_LOG << GetId() << ": FastData is not enabled" << Endl;
        return;
    }
    auto discretization = settings.Get<TDuration>("fast_data:discretization").GetOrElse(TDuration::MilliSeconds(100));
    auto durationMultiplier = settings.Get<float>("fast_data:duration_multiplier").GetOrElse(1.2);
    auto now = Now();

    NDrive::NVega::TCommandRequest::TFastDataConfig config;
    config.SetDiscretization(discretization);
    config.SetDuration(durationMultiplier * Interval);
    NDrive::NVega::TArgument argument;
    argument.Set(config);

    auto handler = MakeIntrusive<TSendCommandTask>(
        ToString(now.MicroSeconds()) + '-' + GetId(),
        NDrive::NVega::ECommandCode::FAST_DATA_CONFIG,
        argument
    );
    handler->DisableSerialization();
    INFO_LOG << GetId() << ": scheduling FastData using " << handler->GetId() << Endl;
    connection->AddHandler(handler);
}

THolder<TGlobalScheduler::IScheduledItem> NDrive::TFastDataScheduler::GetNextScheduledItem(TInstant now) const {
    auto connection = Server.GetConnection(IMEI);
    if (!connection || !connection->Alive()) {
        return nullptr;
    }
    return MakeHolder<TFastDataScheduler>(Server, IMEI, now + Interval, Interval);
}
