#include "handlers.h"

#include <drive/telematics/server/common/signals.h>

#include <drive/telematics/common/firmware.h>
#include <drive/telematics/protocol/errors.h>

#include <drive/library/cpp/threading/container.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/scheduler/global.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>

#include <util/digest/fnv.h>
#include <util/string/hex.h>

namespace {
    NJson::TJsonValue HandlerOrConnnectionFailure(bool handler, bool connection) {
        NJson::TJsonValue result;

        if (connection) {
            result["message"] = "No handler";
        } else if (handler) {
            result["message"] = "No connection";
        } else {
            result["message"] = "No connection, no handler";
        }

        return result;
    }
}

bool NDrive::TWrapperHandler::DoOnAfterRegister(TTelematicsConnection* connection) {
    if (Handler && connection) {
        Handler->OnAdd(*connection);
        return false;
    } else {
        return true;
    }
}

bool NDrive::TWrapperHandler::DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (Handler && connection) {
        switch (Handler->OnMessage(*connection, message)) {
        case NVega::EHandlerStatus::Continue:
            return false;
        case NVega::EHandlerStatus::Finish:
            return true;
        }
    } else {
        return true;
    }
}

void NDrive::TWrapperHandler::DoOnTermination() {
    if (Handler) {
        Handler->OnDrop();
    }
}

NDrive::TTaskPtr NDrive::TCommonTask::Restore(const NJson::TJsonValue& value) {
    auto type = NJson::FromJson<TString>(value["type"]);
    auto result = NDrive::TTaskPtr(TFactory::Construct(type));
    if (!result) {
        return nullptr;
    }
    Y_ASSERT(result->Serializable());
    result->Deserialize(value);
    return result;
}

NDrive::TCommonTask::TCommonTask()
    : SensorValue(TNull())
    , Status(EStatus::New)
{
}

NDrive::TCommonTask::TCommonTask(const TString& id, TDuration timeout /*= TDuration::Seconds(100)*/)
    : Id(id)
    , CreatedTime(Now())
    , Deadline(CreatedTime + timeout)
    , SensorValue(TNull())
    , Status(EStatus::New)
{
    TTelematicsUnistatSignals::Get().HandlersTotal.Signal(TCommonTask::ObjectCount());
}

NDrive::TCommonTask::~TCommonTask() {
    TTelematicsUnistatSignals::Get().HandlersTotal.Signal(TCommonTask::ObjectCount());
}

void NDrive::TCommonTask::AddCallback(TCallback&& callback, TInstant time, TInstant deadline) {
    if (deadline) {
        if (deadline < Now()) {
            InvokeCallback(std::move(callback));
        } else {
            auto callbackContainer = MakeThreadSafeContainer(std::move(callback));
            auto wrapper = [callbackContainer = std::move(callbackContainer)](const TCommonTask& task) {
                auto callback = callbackContainer->Release().GetOrElse(nullptr);
                if (callback) {
                    callback(task);
                }
            };
            ScheduleCallback(MakeCopy(wrapper), deadline);
            AddCallback(std::move(wrapper), time, /*deadline=*/TInstant::Zero());
        }
        return;
    }

    auto guard = Guard(Lock);
    Callbacks.emplace_back(std::move(callback), time);
    if (Finished()) {
        InvokeCallbacks();
    }
}

bool NDrive::TCommonTask::OnAfterRegister(TTelematicsConnection* connection) try {
    auto guard = Guard(Lock);
    IMEI = connection ? connection->GetIMEI() : IMEI;
    StartedTime = Now();
    if (Status == EStatus::New) {
        Status = EStatus::Processing;
    }
    return DoOnAfterRegister(connection);
} catch(const std::exception& e) {
    NJson::TJsonValue result;
    result["exception"] = FormatExc(e);
    result["stage"] = "OnAfterRegister";
    auto guard = Guard(Lock);
    SetResult(std::move(result));
    SetStatus(EStatus::Failure);
    return true;
}

bool NDrive::TCommonTask::OnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) try {
    auto guard = Guard(Lock);
    HeartbeatTime = Now();
    if (DoOnMessage(connection, message)) {
        return true;
    }
    if (Now() > Deadline) {
        WARNING_LOG << GetId() << ": NDrive::TCommonTask::DoOnMessage TIMEOUT" << Endl;
        SetStatus(EStatus::Timeouted);
        return true;
    }
    return false;
} catch(const std::exception& e) {
    ERROR_LOG << GetId() << ": NDrive::TCommonTask::DoOnMessage EXCEPTION: " << FormatExc(e) << Endl;
    NJson::TJsonValue result;
    result["exception"] = FormatExc(e);
    result["stage"] = "OnMessage";
    auto guard = Guard(Lock);
    SetResult(std::move(result));
    SetStatus(EStatus::Failure);
    return true;
}

void NDrive::TCommonTask::OnTermination() {
    auto guard = Guard(Lock);
    HeartbeatTime = Now();
    SetStatus(EStatus::Terminated);
    DoOnTermination();
}

void NDrive::TCommonTask::AddEvent(NJson::TJsonValue&& value) {
    if (value.IsMap()) {
        if (!value.Has("timestamp")) {
            value["timestamp"] = NJson::ToJson(Now());
        }
        Events.AppendValue(std::move(value));
    } else {
        NJson::TJsonValue ev;
        ev["data"] = std::move(value);
        AddEvent(std::move(ev));
    }
}

void NDrive::TCommonTask::SetResult(NJson::TJsonValue&& value) {
    Result = std::move(value);
}

void NDrive::TCommonTask::SetSensorValue(TSensorValue&& value) {
    SensorValue = std::move(value);
}

void NDrive::TCommonTask::SetSensorId(TSensorId value) {
    SensorId = value;
}

void NDrive::TCommonTask::SetLastInfo(TStringBuf info) {
    LastInfo = info;
}

void NDrive::TCommonTask::SetStatus(EStatus value) {
    FinishedTime = Now();
    Status = value;
    if (Status == EStatus::Failure && IgnoreFailure) {
        Status = EStatus::Success;
    }
    Ev.Signal();
    try {
        DoOnFinish();
    } catch (const std::exception& e) {
        ERROR_LOG << GetId() << ": an exception occurred in NDrive::TCommonTask::DoOnFinish: " << FormatExc(e) << Endl;
    }
}

void NDrive::TCommonTask::CopyState(const TCommonTask& other) {
    auto events = other.GetEvents();
    for (auto&& e : events.GetArraySafe()) {
        AddEvent(std::move(e));
    }
    SetResult(other.GetResult());
    SetSensorId(other.GetSensorId());
    SetSensorValue(other.GetSensorValue());
    SetLastInfo(other.GetLastInfo());
    SetStatus(other.GetStatus());
}

void NDrive::TCommonTask::InvokeCallbacks() {
    while (!Callbacks.empty()) {
        InvokeCallback(std::move(Callbacks.front()));
        Callbacks.pop_front();
    }
}

void NDrive::TCommonTask::InvokeCallback(TCallbackInfo&& info) {
    if (!info.Function) {
        WARNING_LOG << GetId() << ": zero callback encountered" << Endl;
        Y_ASSERT(info.Function);
        return;
    }
    if (info.Time && info.Time > Now()) {
        ScheduleCallback(std::move(info.Function), info.Time);
    } else {
        INFO_LOG << GetId() << ": invoking callback" << Endl;
        info.Function(*this);
    }
}

void NDrive::TCommonTask::ScheduleCallback(TCallback&& callback, TInstant time) {
    auto self = TTaskPtr(this);
    bool scheduled = TGlobalScheduler::Schedule(time, [callback = std::move(callback), self = std::move(self)] {
        CHECK_WITH_LOG(callback);
        CHECK_WITH_LOG(self);
        INFO_LOG << self->GetId() << ": invoking scheduled callback" << Endl;
        callback(*self);
    });
    Y_ENSURE_BT(scheduled);
}

void NDrive::TCommonTask::DoOnFinish() {
    InvokeCallbacks();
}

NJson::TJsonValue NDrive::TCommonTask::DoSerialize() const {
    NJson::TJsonValue result;
    result["type"] = GetType();
    result["id"] = Id;
    result["imei"] = NJson::ToJson(NJson::Nullable(IMEI));
    result["created"] = NJson::ToJson(CreatedTime);
    result["deadline"] = NJson::ToJson(Deadline);
    result["started"] = NJson::ToJson(StartedTime);
    result["heartbeat"] = NJson::ToJson(HeartbeatTime);
    result["finished"] = NJson::ToJson(FinishedTime);
    result["info"] = NJson::ToJson(NJson::Nullable(LastInfo));
    result["events"] = Events;
    result["result"] = Result;
    result["sensor_id"] = NJson::ToJson(NJson::Nullable(SensorId));
    result["sensor_value"] = NJson::ToJson(SensorValue);
    result["status"] = NJson::ToJson(NJson::Stringify(Status));
    return result;
}

void NDrive::TCommonTask::DoDeserialize(const NJson::TJsonValue& value) {
    NJson::ReadField(value, "id", Id, true);
    NJson::ReadField(value, "imei", IMEI);
    NJson::ReadField(value, "created", CreatedTime, true);
    NJson::ReadField(value, "deadline", Deadline, true);
    NJson::ReadField(value, "started", StartedTime);
    NJson::ReadField(value, "heartbeat", HeartbeatTime);
    NJson::ReadField(value, "finished", FinishedTime);
    NJson::ReadField(value, "info", LastInfo);
    NJson::ReadField(value, "events", Events);
    NJson::ReadField(value, "result", Result);
    NJson::ReadField(value, "sensor_id", SensorId);
    NJson::ReadField(value, "status", NJson::Stringify(Status));

    if (Finished()) {
        Ev.Signal();
    }
    if (SensorId) {
        SensorValue = NDrive::SensorValueFromJson(value["sensor_value"], SensorId.Id);
    }
}

bool NDrive::TPingTask::DoOnAfterRegister(TTelematicsConnection* connection) {
    auto message = MakeHolder<NVega::TMessage>(NVega::PING_REQUEST);
    AddEvent(message->DebugString());

    Y_ASSERT(connection);
    connection->SendMessage(std::move(message));
    return false;
}

bool NDrive::TPingTask::DoOnMessage(TTelematicsConnection* /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetMessageType() == NDrive::NVega::PING_RESPONSE) {
        AddEvent(message.DebugString());
        TDuration duration = GetHeartbeatTime() - GetStartedTime();
        TTelematicsUnistatSignals::Get().MessagesTimes.Signal(duration.MilliSeconds());
        SetStatus(EStatus::Success);
        return true;
    }
    return false;
}

bool NDrive::TSendCommandTask::DoOnAfterRegister(TTelematicsConnection* connection) {
    NDrive::NVega::TCommandRequest request;
    request.Code = Command;
    request.Argument = Argument;

    Y_ENSURE(connection);
    auto message = connection->CreateCommandRequest(std::move(request), FnvHash<NProtocol::TSequenceId>(GetId()));
    UniqueId = message->GetSequenceId();
    AddEvent(message->DebugString());

    connection->SendMessage(std::move(message));
    return false;
}

bool NDrive::TSendCommandTask::DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return false;
    }
    if (message.GetMessageType() == NDrive::NVega::COMMAND_RESPONSE) {
        const auto& response = message.As<NDrive::NVega::TCommandResponse>();
        if (message.GetSequenceId() == UniqueId) {
            AddEvent(message.DebugString());
            TDuration duration = GetHeartbeatTime() - GetStartedTime();

            TTelematicsUnistatSignals::Get().MessagesTimes.Signal(duration.MilliSeconds());
            TTelematicsUnistatSignals::Get().CommandResults.Signal(
                static_cast<NDrive::NVega::TCommandResponse::EResult>(response.Result),
                1
            );

            NJson::TJsonValue result = GetResult();
            switch (Command) {
            case NDrive::NVega::ECommandCode::GET_PARAM:
                if (!response.Argument.IsNull()) {
                    const auto& argument = response.Argument.Get<NDrive::NVega::TCommandResponse::TGetParameter>();
                    const auto& parameter = Argument.Get<NDrive::NVega::TCommandRequest::TGetParameter>();
                    Y_ENSURE(argument.Id == parameter.Id, "parameter Id mismatch: " << argument.Id << " " << parameter.Id);
                    {
                        NDrive::TSensor sensor;
                        sensor.Id = argument.Id;
                        sensor.SubId = parameter.SubId;
                        sensor.Value = NDrive::SensorValueFromRef(argument.GetValue());
                        sensor.Timestamp = GetHeartbeatTime();
                        result["sensor"] = sensor.ToJson();
                        SetSensorId(sensor);
                        SetSensorValue(std::move(sensor.Value));
                    }
                    connection->OnSensor(argument.Id, parameter.SubId, argument.GetValue());
                }
                break;
            case NDrive::NVega::ECommandCode::SET_PARAM:
                {
                    const auto& parameter = Argument.Get<NDrive::NVega::TCommandRequest::TSetParameter>();
                    connection->OnSensor(parameter.Id, parameter.SubId, parameter.GetValue());
                }
                // Fall through
            default:
                if (!response.Argument.IsNull()) {
                    const auto& argument = response.Argument.Get<NDrive::NVega::TCommandResponse::TInfo>();
                    const auto& info = argument.Get();
                    SetLastInfo(info);
                    result["messages"].AppendValue(argument.Get());
                }
                break;
            }
            result["result"] = ToString<ui32>(response.Result);
            SetResult(std::move(result));

            EStatus commandStatus = InterpretResult(response.Result);
            if (commandStatus == EStatus::Success) {
                TTelematicsUnistatSignals::Get().CommandTimes.Signal(Command, duration.MilliSeconds());
            }
            switch (response.Result) {
            case NDrive::NVega::TCommandResponse::IN_PROGRESS:
                return commandStatus == EStatus::Success;
            default:
                return true;
            }
        }
    }
    return false;
}

NJson::TJsonValue NDrive::TSendCommandTask::DoSerialize() const {
    NJson::TJsonValue result = TBase::DoSerialize();
    result["command"] = NJson::ToJson(NJson::Stringify(Command));
    result["argument"] = NJson::ToJson(Argument);
    result["unique_id"] = NJson::ToJson(UniqueId);
    return result;
}

void NDrive::TSendCommandTask::DoDeserialize(const NJson::TJsonValue& value) {
    TBase::DoDeserialize(value);
    NJson::ReadField(value, "command", NJson::Stringify(Command));
    NJson::ReadField(value, "argument", Argument);
    NJson::ReadField(value, "unique_id", UniqueId);
}

NDrive::TCommonTask::EStatus NDrive::TSendCommandTask::InterpretResult(ui8 result) {
    switch (result) {
    case NDrive::NVega::TCommandResponse::PROCESSED:
    case NDrive::NVega::TCommandResponse::READY_TO_CONTINUE:
    {
        const auto& last = GetLastInfo();
        if (last) { // workaround
            if (NDrive::IsError(last)) {
                SetStatus(EStatus::Failure);
                break;
            }
        }
        SetStatus(EStatus::Success);
        break;
    }
    case NDrive::NVega::TCommandResponse::BUSY:
        SetStatus(TCommonTask::EStatus::Retry);
        break;
    case NDrive::NVega::TCommandResponse::ERROR:
    case NDrive::NVega::TCommandResponse::INCORRECT:
        SetStatus(EStatus::Failure);
        break;
    case NDrive::NVega::TCommandResponse::IN_PROGRESS:
        if (Command == NDrive::NVega::ECommandCode::YADRIVE_WARMING) { // workaround
            const auto& last = GetLastInfo();
            ETelematicsNotification notification = NDrive::ParseError(last);
            if (notification == ETelematicsNotification::EngineRunning) {
                SetStatus(EStatus::Success);
            }
            if (notification == ETelematicsNotification::EngineRunFailure) {
                SetStatus(EStatus::Failure);
            }
        }
        if (Command == NDrive::NVega::ECommandCode::YADRIVE_PANIC) {
            SetStatus(EStatus::Success);
        }
        break;
    }
    return GetStatus();
}

namespace {
    NDrive::NProtocol::TArgument GetParameterArgument(ui16 id, ui16 subid) {
        NDrive::NVega::TCommandRequest::TGetParameter parameter;
        parameter.Id = id;
        parameter.SubId = subid;
        NDrive::NProtocol::TArgument argument;
        argument.Set(parameter);
        return argument;
    }
}

NDrive::TGetParameterTask::TGetParameterTask(const TString& taskId, NDrive::NProtocol::TArgument argument)
    : TSendCommandTask(taskId, NDrive::NVega::ECommandCode::GET_PARAM, std::move(argument))
{
}

NDrive::TGetParameterTask::TGetParameterTask(const TString& taskId, ui16 id, ui16 subid /*= 0*/)
    : TGetParameterTask(taskId, GetParameterArgument(id, subid))
{
}

bool NDrive::TInterfaceTask::DoOnAfterRegister(TTelematicsConnection* connection) {
    if (!Input.Empty()) {
        if (connection->GetProtocolType() == NProtocol::PT_VEGA) {
            auto message = MakeHolder<NDrive::NVega::TMessage>(NVega::INTERFACE_REQUEST);
            auto& payload = message->As<NDrive::NVega::TInterfaceRequest>();
            payload.Interface = Interface;
            payload.Data.assign(Input.Begin(), Input.End());
            Yensured(connection)->SendMessage(std::move(message));
        }
    }
    return false;
}

bool NDrive::TInterfaceTask::DoOnMessage(TTelematicsConnection* /*connection*/, const NProtocol::IMessage& message) {
    if (message.GetProtocolType() != NProtocol::PT_VEGA) {
        return false;
    }
    if (message.GetMessageType() == NDrive::NVega::INTERFACE_RESPONSE) {
        const auto& response = message.As<NDrive::NVega::TInterfaceResponse>();
        if (response.Interface == Interface) {
            Output.Assign(response.Data.begin(), response.Data.end());
            SetStatus(EStatus::Success);
            return true;
        }
    }
    return false;
}

NDrive::TCanRequestTask::TCanRequestTask(const TString& id, ui32 canId, ui8 canIndex, TConstArrayRef<ui8> data)
    : TCommonTask(id, TDuration::Seconds(10))
{
    Y_ENSURE(data.size() <= 8);
    NDrive::NVega::TCanRequest::TFrame frame;
    frame.Data.id = canId;
    frame.Data.itf_idx = canIndex;
    frame.Data.dlen = data.size();
    for (size_t i = 0; i < data.size(); ++i) {
        frame.Data.data[i] = data[i];
    }
    Request.Frames.push_back(frame);
}

bool NDrive::TCanRequestTask::DoOnAfterRegister(NDrive::TTelematicsConnection* connection) {
    if (connection->GetProtocolType() == NProtocol::PT_VEGA) {
        auto message = MakeHolder<NVega::TMessage>(MakeHolder<NDrive::NVega::TCanRequest>(Request));
        Yensured(connection)->SendMessage(std::move(message));
    }
    return true;
}

bool NDrive::TCanRequestTask::DoOnMessage(NDrive::TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (!Active()) {
        return false;
    }
    if (ResponseId) {
        if (message.GetMessageType() == NDrive::NVega::CAN_RESPONSE) {
            const auto& response = message.As<NDrive::NVega::TCanResponse>();
            for (auto&& frame : response.Frames) {
                if (frame.Data.id == ResponseId) {
                    Frames.push_back(frame);
                    SetStatus(EStatus::Success);
                    return true;
                }
            }
        }
        return false;
    }

    Y_UNUSED(connection);
    SetStatus(EStatus::Success);
    return true;
}

NJson::TJsonValue NDrive::TCanRequestTask::DoSerialize() const {
    NJson::TJsonValue result = TCommonTask::DoSerialize();
    NJson::InsertField(result, "frames", Frames);
    return result;
}

void NDrive::TCanRequestTask::DoDeserialize(const NJson::TJsonValue& value) {
    TCommonTask::DoDeserialize(value);
    NJson::ReadField(value, "frames", Frames);
}

NDrive::TCanSetupTask::TCanSetupTask(const TString& id, const NDrive::NProtocol::TArgument& argument)
    : TSendCommandTask(id, NDrive::NVega::ECommandCode::OBD_FORWARD_CONFIG)
{
    Config = argument.TryGet<NDrive::NVega::TCommandRequest::TObdForwardConfig>();
    CustomConfig = argument.TryGet<NDrive::NVega::TCommandRequest::TCustomObdForwardConfig>();
    Y_ENSURE(Config || CustomConfig);
}

NDrive::TCanSetupTask::TCanSetupTask(const TString& id, const NDrive::NVega::TCommandRequest::TCustomObdForwardConfig& config)
    : TSendCommandTask(id, NDrive::NVega::ECommandCode::OBD_FORWARD_CONFIG)
    , CustomConfig(config)
{
}

bool NDrive::TCanSetupTask::DoOnAfterRegister(NDrive::TTelematicsConnection* connection) {
    if (connection && !Config && CustomConfig) {
        auto firmware = connection->GetSensorsCache().Get(VEGA_MCU_FIRMWARE_VERSION);
        Y_ENSURE(firmware, "cannot determine Firmware");
        auto firmwareInfo = NDrive::NVega::ParseFirmwareInfo(firmware->ConvertTo<TString>());
        auto obdCount = firmwareInfo.GetCanCount();
        Config.ConstructInPlace();
        Config->Duration = CustomConfig->Duration;
        for (size_t i = 0; i < obdCount; ++i) {
            auto& setting = Config->Cans.emplace_back().Data;
            setting.is_enable = 1;
            setting.id_type = 1;
            setting.value = CustomConfig->Value;
            setting.mask = CustomConfig->Mask;
        }
    }
    SetArgument(*Config);
    return TBase::DoOnAfterRegister(connection);
}

NDrive::TCommonTask::TFactory::TRegistrator<NDrive::TCanRequestTask> CanRequestTaskRegistrator(NDrive::TCanRequestTask::Type());
NDrive::TCommonTask::TFactory::TRegistrator<NDrive::TSendCommandTask> SendCommandTaskRegistrator(NDrive::TSendCommandTask::Type());
NDrive::TCommonTask::TFactory::TRegistrator<NDrive::TGetParameterTask> GetParameterTaskRegistrator(NDrive::TGetParameterTask::Type());

bool NDrive::TUploadFileHandler::DoOnAfterRegister(TTelematicsConnection* connection) {
    if (Handler && connection) {
        Handler->OnAdd(*connection);
        return false;
    } else {
        return true;
    }
}

bool NDrive::TUploadFileHandler::DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (Handler && connection) {
        switch (Handler->OnMessage(*connection, message)) {
        case NVega::EHandlerStatus::Continue:
            return false;
        case NVega::EHandlerStatus::Finish:
            if (Handler->GetResult() == NVega::TFileChunkResponse::OK) {
                SetStatus(EStatus::Success);
            } else {
                NJson::TJsonValue result;
                result["error_code"] = ToString(Handler->GetResult());
                SetResult(std::move(result));
                SetStatus(EStatus::Failure);
            }
            return true;
        }
    } else {
        NJson::TJsonValue result = HandlerOrConnnectionFailure(!!Handler, !!connection);
        SetResult(std::move(result));
        SetStatus(EStatus::Failure);
        return true;
    }
}

void NDrive::TUploadFileHandler::DoOnTermination() {
    if (Handler) {
        Handler->OnDrop();
    }
}

bool NDrive::TDownloadFileHandler::DoOnAfterRegister(TTelematicsConnection* connection) {
    if (Handler && connection) {
        Handler->OnAdd(*connection);
        return false;
    } else {
        return true;
    }
}

bool NDrive::TDownloadFileHandler::DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (Handler && connection) {
        switch (Handler->OnMessage(*connection, message)) {
        case NVega::EHandlerStatus::Continue:
            return false;
        case NVega::EHandlerStatus::Finish:
            if (Handler->GetResult() == NVega::TGetFileResponse::OK) {
                auto& data = Handler->GetData();

                NJson::TJsonValue result;
                result["data"] = Base64Encode({data.Data(), data.Size()});
                SetResult(std::move(result));
                SetStatus(EStatus::Success);
            } else {
                NJson::TJsonValue result;
                result["error_code"] = ToString(Handler->GetResult());
                SetResult(std::move(result));
                SetStatus(EStatus::Failure);
            }
            return true;
        }

    } else {
        auto result = HandlerOrConnnectionFailure(!!Handler, !!connection);
        SetResult(std::move(result));
        SetStatus(EStatus::Failure);
        return true;
    }
}

void NDrive::TDownloadFileHandler::DoOnTermination() {
    if (Handler) {
        Handler->OnDrop();
    }
}

bool NDrive::TNavTelecomCommand::DoOnAfterRegister(TTelematicsConnection* /*connection*/) {
    return false;
}

bool NDrive::TNavTelecomCommand::DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (connection->GetProtocolType() != NDrive::NProtocol::PT_NAVTELECOM) {
        return false;
    }

    auto type = message.GetMessageTypeAs<NDrive::NNavTelecom::EMessageType>();

    if (type == NDrive::NNavTelecom::MT_DIGITAL_OUTPUT_ANSWER) {
        NJson::TJsonValue result = GetResult();
        const TString lastInfo = "complete";

        result["result"] = ToString<ui32>(NDrive::NVega::TCommandResponse::PROCESSED);
        result["messages"] = lastInfo;

        SetStatus(EStatus::Success);
        SetLastInfo(lastInfo);
        SetResult(std::move(result));
        return true;
    }

    return false;
}
