#pragma once

#include "connection.h"

#include <drive/telematics/common/file.h>
#include <drive/telematics/protocol/navtelecom.h>
#include <drive/telematics/protocol/vega.h>

#include <library/cpp/json/json_value.h>
#include <library/cpp/object_factory/object_factory.h>

#include <util/generic/object_counter.h>
#include <util/system/event.h>
#include <util/system/mutex.h>

namespace NDrive {
    class TCommonTask;
    using TTaskPtr = TIntrusivePtr<TCommonTask>;

    class TCommonTask
        : public ITelematicsHandler
        , public TObjectCounter<TCommonTask>
    {
    public:
        enum class EStatus {
            New        /* "new" */,
            Processing /* "processing" */,
            Success    /* "success" */,
            Failure    /* "failure" */,
            Terminated /* "terminated" */,
            Timeouted  /* "timeouted" */,
            Retry      /* "retry" */,
        };

        using TCallback = std::function<void(const TCommonTask&)>;
        struct TCallbackInfo {
            TCallback Function;
            TInstant Time;

            inline TCallbackInfo(TCallback&& function, TInstant time = TInstant::Zero())
                : Function(std::move(function))
                , Time(time)
            {
            }
        };
        using TCallbacks = std::deque<TCallbackInfo>;
        using TFactory = NObjectFactory::TObjectFactory<TCommonTask, TString>;

    public:
        static TTaskPtr Restore(const NJson::TJsonValue& value);

    public:
        TCommonTask();
        TCommonTask(const TString& id, TDuration timeout = TDuration::Seconds(1000));
        ~TCommonTask();

        bool Active() const {
            return Status == EStatus::Processing;
        }
        bool Finished() const {
            return Status != EStatus::Processing && Status != EStatus::New;
        }
        bool Terminated() const {
            return Status == EStatus::Terminated;
        }

        TInstant GetCreatedTime() const {
            return CreatedTime;
        }
        TInstant GetFinishedTime() const {
            return FinishedTime;
        }
        TInstant GetHeartbeatTime() const {
            return HeartbeatTime;
        }
        TInstant GetStartedTime() const {
            return StartedTime;
        }
        TInstant GetTimestamp() const final {
            return HeartbeatTime;
        }
        TString GetId() const final {
            return Id;
        }
        NJson::TJsonValue GetEvents() const {
            auto guard = Guard(Lock);
            return Events;
        }
        NJson::TJsonValue GetResult() const {
            auto guard = Guard(Lock);
            return Result;
        }
        TSensorId GetSensorId() const {
            auto guard = Guard(Lock);
            return SensorId;
        }
        TSensorValue GetSensorValue() const {
            auto guard = Guard(Lock);
            return SensorValue;
        }
        TString GetLastInfo() const {
            auto guard = Guard(Lock);
            return LastInfo;
        }
        EStatus GetStatus() const {
            return Status;
        }
        bool IsFinished() const final {
            return Finished();
        }

        bool Wait(TDuration timeout = TDuration::Max()) {
            return Ev.WaitT(timeout);
        }
        bool Wait(TInstant deadline) {
            return Ev.WaitD(deadline);
        }

        void AddCallback(TCallback&& callback, TInstant time = TInstant::Zero(), TInstant deadline = TInstant::Zero());

        virtual bool OnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool OnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;
        virtual void OnTermination() override;

        virtual TVector<NProtocol::TSequenceId> GetExpectedSequenceIds() const {
            return {};
        }
        virtual TString GetType() const {
            return "CommonTask";
        }
        virtual void SetDeadline(TInstant deadline) {
            Deadline = deadline;
        }
        void SetIgnoreFailure(bool value) {
            IgnoreFailure = value;
        }
        NJson::TJsonValue Serialize() const final {
            auto guard = Guard(Lock);
            return DoSerialize();
        }
        void Deserialize(const NJson::TJsonValue& value) {
            auto guard = Guard(Lock);
            DoDeserialize(value);
        }

    protected:
        void AddEvent(NJson::TJsonValue&& value);
        void SetResult(NJson::TJsonValue&& value);
        void SetSensorValue(TSensorValue&& value);
        void SetSensorId(TSensorId value);
        void SetLastInfo(TStringBuf info);
        void SetStatus(EStatus value);
        void CopyState(const TCommonTask& other);
        void InvokeCallbacks();
        void InvokeCallback(TCallbackInfo&& info);
        void ScheduleCallback(TCallback&& callback, TInstant time);

        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) = 0;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) = 0;
        virtual void DoOnTermination() {}
        virtual void DoOnFinish();

        virtual NJson::TJsonValue DoSerialize() const;
        virtual void DoDeserialize(const NJson::TJsonValue& value);

    private:
        TString Id;
        TString IMEI;
        TInstant CreatedTime;
        TCallbacks Callbacks;
        bool IgnoreFailure = false;

        TString LastInfo;
        TInstant Deadline;
        TInstant StartedTime;
        TInstant HeartbeatTime;
        TInstant FinishedTime;
        NJson::TJsonValue Events;
        NJson::TJsonValue Result;
        TSensorValue SensorValue;
        TSensorId SensorId;
        EStatus Status;

        TManualEvent Ev;
        TMutex Lock;
    };

    class TPingTask: public TCommonTask {
    public:
        using TCommonTask::TCommonTask;

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;
    };

    class TSendCommandTask: public TCommonTask {
    private:
        using TBase = TCommonTask;

    private:
        static constexpr auto IncorrectSequenceId = static_cast<NProtocol::TSequenceId>(-1);

    public:
        TSendCommandTask() = default;
        TSendCommandTask(const TString& id, NDrive::NVega::ECommandCode command, NDrive::NProtocol::TArgument argument = {})
            : TCommonTask(id)
            , Command(command)
            , Argument(argument)
        {
        }

        static TString Type() {
            return "SendCommandTask";
        }
        virtual TVector<NProtocol::TSequenceId> GetExpectedSequenceIds() const override {
            if (UniqueId != IncorrectSequenceId) {
                return { UniqueId };
            } else {
                return {};
            }
        }
        virtual TString GetType() const override {
            return Type();
        }
        virtual bool Serializable() const override {
            return IsSerializable;
        }

        void DisableSerialization() {
            IsSerializable = false;
        }

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;

        virtual NJson::TJsonValue DoSerialize() const override;
        virtual void DoDeserialize(const NJson::TJsonValue& value) override;

        NDrive::TCommonTask::EStatus InterpretResult(ui8 result);

        template <class T>
        void SetArgument(T&& value) {
            Argument.Set(std::forward<T>(value));
        }

    private:
        NDrive::NVega::ECommandCode Command = NDrive::NVega::ECommandCode::UNKNOWN;
        NDrive::NProtocol::TArgument Argument;

        NProtocol::TSequenceId UniqueId = IncorrectSequenceId;
        bool IsSerializable = true;
    };

    class TGetParameterTask: public TSendCommandTask {
    public:
        TGetParameterTask() = default;
        TGetParameterTask(const TString& taskId, NDrive::NProtocol::TArgument argument);
        TGetParameterTask(const TString& taskId, ui16 id, ui16 subid = 0);

        static TString Type() {
            return "GetParameterTask";
        }
        virtual TString GetType() const override {
            return Type();
        }
        virtual bool Serializable() const override {
            return false;
        }
    };

    class TInterfaceTask: public TCommonTask {
    public:
        TInterfaceTask(const TString& id, NVega::TInterfaceData::EInterface interface_, TBuffer&& data)
            : TCommonTask(id, TDuration::Seconds(10))
            , Interface(interface_)
            , Input(std::move(data))
        {
        }

        const TBuffer& GetOutput() const {
            return Output;
        }
        TBuffer ReleaseOutput() {
            return std::move(Output);
        }

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;

    private:
        NVega::TInterfaceData::EInterface Interface;
        TBuffer Input;
        TBuffer Output;
    };

    class TCanRequestTask: public TCommonTask {
    public:
        TCanRequestTask() = default;
        TCanRequestTask(const TString& id, ui32 canId, ui8 canIndex, TConstArrayRef<ui8> data);

        static TString Type() {
            return "CanRequestTask";
        }
        virtual TString GetType() const override {
            return Type();
        }
        void SetResponseId(ui32 value) {
            ResponseId = value;
        }

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;

        virtual NJson::TJsonValue DoSerialize() const override;
        virtual void DoDeserialize(const NJson::TJsonValue& value) override;

    private:
        NVega::TCanRequest Request;
        ui32 ResponseId = 0;

        NVega::TCanResponse::TFrames Frames;
    };

    class TCanSetupTask: public TSendCommandTask {
    private:
        using TBase = TSendCommandTask;

    public:
        TCanSetupTask(const TString& id, const NDrive::NProtocol::TArgument& argument);
        TCanSetupTask(const TString& id, const NDrive::NVega::TCommandRequest::TCustomObdForwardConfig& config);

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;

    private:
        TMaybe<NVega::TCommandRequest::TObdForwardConfig> Config;
        TMaybe<NVega::TCommandRequest::TCustomObdForwardConfig> CustomConfig;
    };

    class TWrapperHandler: public TCommonTask {
    public:
        TWrapperHandler(const TString& id, NProtocol::THandlerPtr handler)
            : TCommonTask(id)
            , Handler(handler)
        {
        }

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;
        virtual void DoOnTermination() override;

    private:
        NProtocol::THandlerPtr Handler;
    };

    class TUploadFileHandler: public TCommonTask {
    public:
        TUploadFileHandler(const TString& id, TAtomicSharedPtr<NVega::TChunkedFileHandler> handler)
            : TCommonTask(id)
            , Handler(handler)
        {
        }

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;
        virtual void DoOnTermination() override;

    private:
        TAtomicSharedPtr<NVega::TChunkedFileHandler> Handler;
    };

    class TDownloadFileHandler: public TCommonTask {
    public:
        TDownloadFileHandler(const TString& id, TAtomicSharedPtr<NVega::TGetFileHandler> handler)
            : TCommonTask(id)
            , Handler(handler)
        {
        }

        static TString Type() {
            return "DownloadFileTask";
        }
        virtual TString GetType() const override {
            return Type();
        }

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;
        virtual void DoOnTermination() override;

    private:
        TAtomicSharedPtr<NVega::TGetFileHandler> Handler;
    };

    class TNavTelecomCommand: public TSendCommandTask {
    public:
        TNavTelecomCommand(const TString& id, NDrive::NVega::ECommandCode command, NDrive::NProtocol::TArgument argument = {})
            : TSendCommandTask(id, command, std::move(argument))
        {
        }

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;
    };

}
