#include "log_loader.h"

#include "server.h"

#include <drive/telematics/common/file.h>

namespace {
    TVector<TStringBuf> SplitLogRecords(TStringBuf data, bool skipFirst) {
        size_t index = 0;
        TVector<TStringBuf> result;
        for (auto&& i : StringSplitter(data).Split('\n')) {
            if (index++ || !skipFirst) {
                result.push_back(i.Token());
            }
        }
        if (result.size()) {
            result.pop_back();
        }
        return result;
    }
}

NDrive::TLogLoader::TLogLoader(const NDrive::TTelematicsServer* server, const TString& imei, TInstant now, TDuration interval)
    : TGlobalScheduler::TScheduledItem<TLogLoader>(server->Name(), "log_loader:" + imei, now + interval)
    , Server(server)
    , IMEI(imei)
    , Interval(interval)
{
    CHECK_WITH_LOG(Server);
}

void NDrive::TLogLoader::Process(void* /*threadSpecificResource*/) {
    THolder<TLogLoader> cleanup(this);
    Y_ASSERT(Server);
    auto connection = Server->GetConnection(IMEI);
    if (connection && connection->Alive()) {
        auto since = GetLastTimestamp();
        auto step = GetStep();
        auto offset = -1 * static_cast<ssize_t>(GetStep());
        auto handler = MakeAtomicShared<NDrive::NVega::TGetFileHandler>("LOG", offset);

        auto callback = [handler, since, step, imei = IMEI, server = Server] (NProtocol::THandlerPtr) {
            const auto storage = server->GetStorage();
            NDrive::NVega::TGetFileResponse::EResult result = handler->GetResult();
            switch (result) {
                case NVega::TGetFileResponse::OK:
                    break;
                case NVega::TGetFileResponse::DATA_SIZE_ERROR:
                case NVega::TGetFileResponse::BUSY:
                case NVega::TGetFileResponse::OUT_OF_DATA:
                case NVega::TGetFileResponse::MEMORY_ERROR:
                case NVega::TGetFileResponse::FILENAME_ERROR:
                case NVega::TGetFileResponse::UNKNOWN_ERROR:
                    ERROR_LOG << imei << ": cannot get LOG file " << result << Endl;
                    break;
            }

            Y_ASSERT(server);
            auto connection = server->GetConnection(imei);

            TBuffer& data = handler->GetData();
            TStringBuf buf(data.Data(), data.Size());
            auto strings = SplitLogRecords(buf, /*skipFirst=*/handler->GetSize() > data.Size());
            if (strings.empty()) {
                return;
            }

            TInstant last = since;
            for (size_t i = 0; i < strings.size(); ++i) {
                auto record = NVega::ParseLogRecord(strings[i]);
                if (!record.Timestamp) {
                    ERROR_LOG << imei << ": no timestamp for log string " << strings[i] << Endl;
                    continue;
                }
                if (i == 0) {
                    if (record.Timestamp <= since) {
                        auto newStep = 2 * step;
                        if (storage->SetValue(GetStepKey(imei), ToString(newStep), /*storeHistory=*/false)) {
                            INFO_LOG << imei << ": set step to " << newStep << Endl;
                        } else {
                            ERROR_LOG << imei << ": unable to set step to " << newStep << Endl;
                        }
                        return;
                    }
                }
                if (record.Timestamp > since) {
                    last = std::max(last, record.Timestamp);
                    INFO_LOG << imei << ": " << strings[i] << Endl;
                    if (connection) {
                        connection->AddLogRecord(std::move(record));
                    }
                }
            }
            if (storage->SetValue(GetLastTimestampKey(imei), ToString(last.Seconds()), /*storeHistory=*/false)) {
                INFO_LOG << imei << ": set last timestamp to " << last << Endl;
            } else {
                ERROR_LOG << imei << ": unable to set last timestamp to " << last << Endl;
            }
            auto newStep = 1024;
            if (storage->SetValue(GetStepKey(imei), ToString(newStep), /*storeHistory=*/false)) {
                INFO_LOG << imei << ": set step to " << newStep << Endl;
            } else {
                ERROR_LOG << imei << ": unable to set step to " << newStep << Endl;
            }
        };
        auto wrapper = MakeAtomicShared<NProtocol::TCallbackHandler>(handler, std::move(callback));
        connection->AddMessageHandler(wrapper);
    }
}

THolder<TGlobalScheduler::IScheduledItem> NDrive::TLogLoader::GetNextScheduledItem(TInstant now) const {
    Y_ASSERT(Server);
    auto connection = Server->GetConnection(IMEI);
    if (connection && connection->Alive()) {
        return MakeHolder<TLogLoader>(Server, IMEI, now, Interval);
    }
    return nullptr;
}

TInstant NDrive::TLogLoader::GetLastTimestamp() const {
    TString value;
    auto storage = Server->GetStorage();
    if (storage && storage->GetValue(GetLastTimestampKey(IMEI), value)) {
        ui64 seconds;
        if (TryFromString(value, seconds)) {
            return TInstant::Seconds(seconds);
        } else {
            ERROR_LOG << IMEI << ": cannot parse LastTimestamp " << value << Endl;
        }
    } else {
        WARNING_LOG << IMEI << ": cannot read LastTimestamp" << Endl;
    }
    return TInstant::Zero();
}

size_t NDrive::TLogLoader::GetStep() const {
    TString value;
    auto storage = Server->GetStorage();
    if (storage && storage->GetValue(GetStepKey(IMEI), value)) {
        size_t size;
        if (TryFromString(value, size)) {
            return size;
        } else {
            ERROR_LOG << IMEI << ": cannot parse LastTimestamp " << value << Endl;
        }
    } else {
        WARNING_LOG << IMEI << ": cannot read LastTimestamp" << Endl;
    }
    return Max<ui32>();
}
