#include "logging.h"

#include "handlers.h"

#include <drive/telematics/server/tasks/lite.h>

#include <drive/telematics/protocol/navtelecom.h>
#include <drive/telematics/protocol/vega.h>
#include <drive/telematics/protocol/wialon.h>

#include <drive/library/cpp/searchserver/replier.h>

#include <library/cpp/json/json_value.h>

#include <rtline/library/json/cast.h>

#include <util/string/hex.h>

namespace {
    using namespace NDrive::NVega;

    const TMap<ui16, TString> PositionSensors = {
        { VEGA_LAT, "lat" },
        { VEGA_LON, "lon" },
        { VEGA_ALT, "height" },
        { VEGA_SPEED, "speed" },
        { VEGA_DIR, "course" },
        { VEGA_SAT_USED, "sats" },
    };

    NJson::TJsonValue GetPositionData(const NDrive::TMultiSensor& sensors) {
        NJson::TJsonValue result;

        for (auto&& sensor : sensors) {
            if (PositionSensors.contains(sensor.Id)) {
                TStringBuf key = PositionSensors.Value(sensor.Id, "");
                if (!key.Empty()) {
                    result[key] = sensor.GetJsonValue();
                }
            }
        }

        return result;
    }

    NJson::TJsonValue GetData(const NDrive::TMultiSensor& sensors, TInstant timestamp) {
        NJson::TJsonValue result;

        result["type"] = "record";
        result["timestamp"] = timestamp.Seconds();

        NJson::TJsonValue& pd = result["subrecords"].AppendValue(NJson::JSON_MAP);
        pd = GetPositionData(sensors);
        pd["type"] = "position_data";

        NJson::TJsonValue& cp = result["subrecords"].AppendValue(NJson::JSON_MAP);
        cp["type"] = "custom_parameters";
        NJson::TJsonValue& params = cp.InsertValue("params", NJson::JSON_MAP);
        for (auto&& sensor : sensors) {
            if (!PositionSensors.contains(sensor.Id)) {
                auto name = ToString(sensor.Id);
                auto value = sensor.GetJsonValue();
                params[name] = value;
            }
        }

        return result;
    }

    NJson::TJsonValue GetData(TConstArrayRef<NDrive::NWialon::TShortData> records) {
        NJson::TJsonValue result;
        result["id"] = 0;

        for (auto&& record : records) {
            NJson::TJsonValue& r = result["records"].AppendValue(NJson::JSON_MAP);
            const auto& dateTime = record.DateTime.Get();
            TInstant timestamp = dateTime ? *dateTime : Now();

            r = GetData(NDrive::NWialon::ToSensors(record), timestamp);
        }

        return result;
    }

    NJson::TJsonValue GetData(const NDrive::NWialon::TShortDataRequest& payload) {
        return GetData(NContainer::Scalar(payload.ShortData));
    }

    NJson::TJsonValue GetData(const NDrive::NWialon::TDataRequest& payload) {
        return GetData(NContainer::Scalar(payload.ShortData));
    }

    NJson::TJsonValue GetData(const NDrive::NWialon::TBlackBoxRequest& payload) {
        TVector<NDrive::NWialon::TShortData> records;
        const auto& data = payload.Data.Get();
        records.reserve(data.size());

        auto transformFunc = [&](const NDrive::NWialon::TBlackBoxRequest::TItem& item) {
            return item.ShortData;
        };
        Transform(data.begin(), data.end(), std::back_inserter(records), transformFunc);

        return GetData(MakeArrayRef(records.begin(), records.size()));
    }

    NJson::TJsonValue GetData(const TBlackboxRecords& payload) {
        NJson::TJsonValue result;
        result["id"] = payload.Id;
        for (auto&& record : payload.Records) {
            NJson::TJsonValue& r = result["records"].AppendValue(NJson::JSON_MAP);
            r = GetData(NDrive::NVega::ToSensors(record), TInstant::Seconds(record.Timestamp));
        }
        return result;
    }

    NJson::TJsonValue GetData(const TCanResponse& payload) {
        return NJson::ToJson(payload);
    }

    NJson::TJsonValue GetData(const TCommandRequest& payload) {
        NJson::TJsonValue result;
        result["id"] = payload.Id;
        result["code"] = static_cast<ui32>(payload.Code);
        result["argument"] = payload.Argument.DebugString();
        switch (payload.Code) {
        case ECommandCode::GET_PARAM: {
            TString error;
            TCommandRequest::TGetParameter parameter;
            if (payload.Argument.TryGet(parameter, &error)) {
                result["parameter"]["id"] = parameter.Id;
                result["parameter"]["subid"] = parameter.SubId;
            } else {
                result["error"] = error;
            }
            break;
        }
        case ECommandCode::SET_PARAM: {
            TString error;
            TCommandRequest::TSetParameter parameter;
            if (payload.Argument.TryGet(parameter, &error)) {
                result["parameter"]["id"] = parameter.Id;
                result["parameter"]["subid"] = parameter.SubId;
                result["parameter"]["value"] = NJson::ToJson(parameter.GetValue());
            } else {
                result["error"] = error;
            }
            break;
        }
        default:
            break;
        }
        return result;
    }

    NJson::TJsonValue GetData(const TCommandResponse& payload) {
        NJson::TJsonValue result;
        result["id"] = payload.Id;
        result["result"] = static_cast<ui32>(payload.Result);
        result["argument"] = payload.Argument.DebugString();

        if (!payload.Argument.IsNull()) {
            TCommandResponse::TGetParameter parameter;
            TCommandResponse::TInfo info;
            if (payload.Argument.TryGet(parameter)) {
                result["parameter"]["id"] = parameter.Id;
                result["parameter"]["value"] = NJson::ToJson(parameter.GetValue());
            } else if (payload.Argument.TryGet(info)) {
                result["info"] = info.Get();
            }
        }

        return result;
    }

    NJson::TJsonValue GetData(const TFastDataRecords& payload) {
        NJson::TJsonValue result;
        NJson::TJsonValue& records = result.InsertValue("records", NJson::JSON_ARRAY);
        for (auto&& record : payload.Records) {
            NJson::TJsonValue& r = records.AppendValue(NJson::JSON_MAP);
            r["drop"] = record.DropCounter;
            r["tick"] = record.TickCounter;
            r["timestamp"] = record.Timestamp;
            NJson::TJsonValue& params = r.InsertValue("params", NJson::JSON_MAP);
            params[ToString(VEGA_LAT)] = record.Latitude;
            params[ToString(VEGA_LON)] = record.Longitude;
            params[ToString(VEGA_SPEED)] = record.Speed;
            params[ToString(VEGA_DIR)] = record.GetCourse();
            params[ToString(VEGA_GPS_IS_ACTIVE)] = record.Active;
            params[ToString(VEGA_GPS_IS_VALID)] = record.Valid;

            params[ToString(VEGA_GSENSOR_AXIS_X)] = record.GetAccelerometerX();
            params[ToString(VEGA_GSENSOR_AXIS_Y)] = record.GetAccelerometerY();
            params[ToString(VEGA_GSENSOR_AXIS_Z)] = record.GetAccelerometerZ();

            params[ToString(CAN_SPEED)] = record.CanSpeed;
            params[ToString(CAN_ACCELERATOR)] = record.Accelerator;
            params[ToString(CAN_BRAKE)] = record.Brake;
            params[ToString(CAN_STEERING_WHEEL)] = record.SteeringWheel;
            params[ToString(CAN_ENGINE_RPM)] = record.Rpm;
            params[ToString(CAN_STEERING_ACCELERATION)] = record.SteeringAcceleration;
        }
        return result;
    }

    NJson::TJsonValue GetData(const NDrive::NWialon::TMessage& message) {
        NJson::TJsonValue result;
        try {
            switch (message.GetMessageType()) {
            case NDrive::NWialon::MT_SHORT_DATA_REQUEST:
                result = GetData(message.As<NDrive::NWialon::TShortDataRequest>());
                break;
            case NDrive::NWialon::MT_DATA_REQUEST:
                result = GetData(message.As<NDrive::NWialon::TDataRequest>());
                break;
            case NDrive::NWialon::MT_BLACK_BOX_REQUEST:
                result = GetData(message.As<NDrive::NWialon::TBlackBoxRequest>());
                break;
            default:
                result["debug"] = message.DebugString();
            }
        } catch (const std::exception& e) {
            result["error"] = FormatExc(e);
            TStringStream ss;
            message.Save(ss);
            result["debug"] = HexEncode(ss.Str());
        }
        return result;
    }

    NJson::TJsonValue GetData(const NDrive::NNavTelecom::THandShakeRequest& payload) {
        NJson::TJsonValue result;
        result["imei"] = payload.IMEI.Get();
        return result;
    }

    NJson::TJsonValue GetData(const NDrive::NNavTelecom::TProtocolSettingRequest& payload) {
        NJson::TJsonValue result;
        result["protocol_version"] = ToString(payload.ProtocolVersion);
        result["struct_version"] = ToString(payload.StructVersion);
        result["data_size"] = payload.DataSize;
        result["bit_field"] = payload.GetBitFieldDebugString();
        return result;
    }

    NJson::TJsonValue GetData(TConstArrayRef<NDrive::NNavTelecom::TRecord> payload) {
        NJson::TJsonValue result;
        result["id"] = 0;

        for (auto&& record : payload) {
            NJson::TJsonValue& r = result["records"].AppendValue(NJson::JSON_MAP);

            auto timestamp = Now();
            auto timestampParameter = record.Get(NDrive::NNavTelecom::PI_TIMESTAMP);
            if (std::holds_alternative<ui64>(timestampParameter)) {
                timestamp = TInstant::Seconds(std::get<ui64>(timestampParameter));
            }

            r = GetData(NDrive::NNavTelecom::ToSensors(record), timestamp);
        }

        return result;
    }

    NJson::TJsonValue GetData(const NDrive::NNavTelecom::TBlackBoxRequest& payload) {
        return GetData(payload.Records);
    }

    NJson::TJsonValue GetData(const NDrive::NNavTelecom::TSingleBlackBoxRequest& payload) {
        return GetData(NContainer::Scalar(payload.Record));
    }

    NJson::TJsonValue GetData(const NDrive::NNavTelecom::TMessage& message) {
        NJson::TJsonValue result;
        try {
            switch (message.GetMessageType()) {
            case NDrive::NNavTelecom::MT_HANDSHAKE_REQUEST:
                result = GetData(message.As<NDrive::NNavTelecom::THandShakeRequest>());
                break;
            case NDrive::NNavTelecom::MT_PROTOCOL_SETTING_REQUEST:
                result = GetData(message.As<NDrive::NNavTelecom::TProtocolSettingRequest>());
                break;
            case NDrive::NNavTelecom::MT_BLACKBOX_REQUEST:
                result = GetData(message.As<NDrive::NNavTelecom::TBlackBoxRequest>());
                break;
            case NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_REQUEST:
                result = GetData(message.As<NDrive::NNavTelecom::TSingleBlackBoxRequest>());
                break;
            default:
                result["debug"] = message.DebugString();
                break;
            }
        } catch (const std::exception& e) {
            result["error"] = FormatExc(e);
            TStringStream ss;
            message.Save(ss);
            result["debug"] = HexEncode(ss.Str());
        }
        return result;
    }

    NJson::TJsonValue GetType(const NDrive::NProtocol::IMessage& message) {
        const TString incorrect = "INCORRECT";

        auto blackBox = ToString(NDrive::NVega::BLACKBOX_RECORDS);

        if (message.GetProtocolType() == NDrive::NProtocol::PT_WIALON_IPS) {
            auto messageType = message.GetMessageTypeAs<NDrive::NWialon::EMessageType>();
            if (
                messageType == NDrive::NWialon::MT_DATA_REQUEST ||
                messageType == NDrive::NWialon::MT_BLACK_BOX_REQUEST ||
                messageType == NDrive::NWialon::MT_SHORT_DATA_REQUEST
            ) {
                return blackBox;
            }
            return ToString(messageType);
        } else if (message.GetProtocolType() == NDrive::NProtocol::PT_NAVTELECOM) {
            auto messageType = message.GetMessageTypeAs<NDrive::NNavTelecom::EMessageType>();
            if (
                messageType == NDrive::NNavTelecom::MT_BLACKBOX_REQUEST ||
                messageType == NDrive::NNavTelecom::MT_SINGLE_BLACKBOX_REQUEST
            ) {
                return blackBox;
            }
            return ToString(messageType);
        } else if (message.GetProtocolType() == NDrive::NProtocol::PT_VEGA) {
            auto messageType = message.GetMessageTypeAs<NDrive::NVega::EMessageType>();
            return ToString(messageType);
        }

        return incorrect;
    }

    NJson::TJsonValue GetData(const NDrive::NProtocol::IMessage& message) {
        using namespace NDrive;
        NJson::TJsonValue result;
        if (message.GetProtocolType() == NProtocol::PT_VEGA) {
            try {
                switch (message.GetMessageType()) {
                    case BLACKBOX_RECORDS:
                        result = GetData(message.As<TBlackboxRecords>());
                        break;
                    case CAN_RESPONSE:
                        result = GetData(message.As<TCanResponse>());
                        break;
                    case COMMAND_REQUEST:
                        result = GetData(message.As<TCommandRequest>());
                        break;
                    case COMMAND_RESPONSE:
                        result = GetData(message.As<TCommandResponse>());
                        break;
                    case FAST_DATA_RECORDS:
                        result = GetData(message.As<TFastDataRecords>());
                        break;
                    default:
                        result["debug"] = message.DebugString();
                        break;
                }
            } catch (const std::exception& e) {
                result["error"] = FormatExc(e);
                TStringStream ss;
                message.Save(ss);
                result["debug"] = HexEncode(ss.Str());
            }
        } else if (message.GetProtocolType() == NDrive::NProtocol::PT_WIALON_IPS) {
            result = GetData(*VerifyDynamicCast<const NDrive::NWialon::TMessage*>(&message));
        } else if (message.GetProtocolType() == NDrive::NProtocol::PT_NAVTELECOM) {
            result = GetData(*VerifyDynamicCast<const NDrive::NNavTelecom::TMessage*>(&message));
        }
        return result;
    }

    void FillCommon(NJson::TJsonValue& r) {
        r.InsertValue("unixtime", Seconds());
    }

    void FillConnection(NJson::TJsonValue& r, const NDrive::TTelematicsConnection* connection) {
        if (connection) {
            const auto& imeiRaw = connection->GetIMEI();
            i64 imei = 0;
            if (imeiRaw && !TryFromString(imeiRaw, imei)) {
                r.InsertValue("raw_imei", imeiRaw);
            }
            r.InsertValue("imei", imei);
            r.InsertValue("client", connection->GetRemoteAddr());
        } else {
            r.InsertValue("imei", 0);
            r.InsertValue("client", TString());
        }
    }
}

NDrive::TTelematicsLog::TTelematicsLog(THolder<TLogBackend> backend)
    : TSearchLog()
{
    ResetBackend(std::move(backend));
}

void NDrive::TTelematicsLog::Log(
    NDrive::TTelematicsLog::EEvent event,
    const NDrive::TTelematicsConnection* connection,
    const NDrive::NProtocol::IMessage& message
) {
    CHECK_WITH_LOG(connection);
    NJson::TJsonValue r;
    FillCommon(r);
    FillConnection(r, connection);
    r.InsertValue("event", ToString(event));
    r.InsertValue("type", GetType(message));
    r.InsertValue("protocol", ToString(message.GetProtocolType()));
    r.InsertValue("data", GetData(message));
    Log(r);
}

void NDrive::TTelematicsLog::Log(
    NDrive::TTelematicsLog::EEvent event,
    const NDrive::TTelematicsConnection* connection
) {
    CHECK_WITH_LOG(connection);
    NJson::TJsonValue r;
    FillCommon(r);
    FillConnection(r, connection);
    r.InsertValue("event", ToString(event));
    r.InsertValue("type", TString());
    r.InsertValue("data", TString());
    Log(r);
}

void NDrive::TTelematicsLog::Log(EHandlerEvent event, const TTelematicsConnection* connection, const ITelematicsHandler* handler) {
    NJson::TJsonValue r;
    FillCommon(r);
    FillConnection(r, connection);
    r.InsertValue("event", ToString(event));
    if (handler) {
        r.InsertValue("type", TypeName(*handler));
    } else {
        r.InsertValue("type", TString());
    }
    if (auto task = dynamic_cast<const NDrive::TCommonTask*>(handler)) {
        r.InsertValue("data", task->Serialize());
    } else {
        r.InsertValue("data", TString());
    }
    Log(r);
}

void NDrive::TTelematicsLog::Log(ETaskEvent event, const TTelematicsConnection* connection, const TCommonDistributedData& data) {
    NJson::TJsonValue r;
    FillCommon(r);
    FillConnection(r, connection);
    r.InsertValue("event", ToString(event));
    r.InsertValue("type", data.GetType());
    r.InsertValue("data", data.GetDataInfo());
    Log(r);
}

void NDrive::TTelematicsLog::Log(EMiscEvent event, const TTelematicsConnection* connection, const TString& type, NJson::TJsonValue&& data) {
    NJson::TJsonValue r;
    FillCommon(r);
    FillConnection(r, connection);
    r.InsertValue("event", ToString(event));
    r.InsertValue("type", type);
    r.InsertValue("data", std::move(data));
    Log(r);
}

void NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::EClientEvent event, const IReplyContext* context, const TString& imei, NJson::TJsonValue&& data) {
    NJson::TJsonValue r;
    FillCommon(r);
    r.InsertValue("imei", imei);
    r.InsertValue("client", context ? context->GetRequestData().RemoteAddr() : "unknown_client");
    r.InsertValue("event", ToString(event));
    r.InsertValue("type", context ? context->GetUri() : "unknown_handle");
    r.InsertValue("data", std::move(data));
    Log(r);
}

void NDrive::TTelematicsLog::Log(const NJson::TJsonValue& r) {
    if (TLoggerOperator<TTelematicsLog>::Usage()) {
        TLoggerOperator<TTelematicsLog>::Log() << r.GetStringRobust() << Endl;
    } else {
        INFO_LOG << r.GetStringRobust() << Endl;
    }
}

void NDrive::TTelematicsLog::Log(EEvent event, const TTelematicsConnection* connection, const char* data, size_t size) {
    CHECK_WITH_LOG(connection);
    NJson::TJsonValue r;
    FillCommon(r);
    FillConnection(r, connection);

    r.InsertValue("event", ToString(event));
    r.InsertValue("type", "raw");

    if (data && size > 0) {
        r.InsertValue("data", HexEncode(data, size));
    } else {
        r.InsertValue("data", ToString(""));
    }

    Log(r);
}
