#include "navtelecom.h"

#include "scenarios.h"

#include <drive/telematics/protocol/navtelecom.h>
#include <drive/telematics/protocol/vega.h>

#include <library/cpp/logger/global/global.h>

#include <util/generic/ptr.h>
#include <util/generic/yexception.h>

NDrive::NNavTelecom::TNavTelecomProtocol::TNavTelecomProtocol() {
    Type = NProtocol::PT_NAVTELECOM;
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Launch() noexcept {
}

THolder<NDrive::NProtocol::IMessage> NDrive::NNavTelecom::TNavTelecomProtocol::Load(IInputStream& stream) {
    auto result = MakeHolder<NNavTelecom::TMessage>(NNavTelecom::MT_INCORRECT, NNavTelecom::TMessage::ESource::Server, GetBitField());
    result->Load(stream);
    return std::move(result);
}

NDrive::NNavTelecom::TBitField NDrive::NNavTelecom::TNavTelecomProtocol::GetBitField() const {
    auto guard = Guard(DataLock);
    return BitField;
}
void NDrive::NNavTelecom::TNavTelecomProtocol::SetBitField(const NNavTelecom::TBitField& bitField) {
    auto guard = Guard(DataLock);
    BitField = bitField;
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NProtocol::IMessage& message) {
    Y_ENSURE(message.GetProtocolType() == NProtocol::PT_NAVTELECOM);

    auto messageType = message.GetMessageType();

    if (messageType == NNavTelecom::MT_HANDSHAKE_REQUEST) {
        Process(message.As<NNavTelecom::THandShakeRequest>());
    } else if (messageType == NNavTelecom::MT_PROTOCOL_SETTING_REQUEST) {
        Process(message.As<NNavTelecom::TProtocolSettingRequest>());
    } else if (messageType == NNavTelecom::MT_BLACKBOX_REQUEST) {
        Process(message.As<NNavTelecom::TBlackBoxRequest>());
    } else if (messageType == NNavTelecom::MT_ADDITIONAL_BLACKBOX_REQUEST) {
        Process(message.As<NNavTelecom::TAdditionalBlackBoxRequest>());
    } else if (messageType == NNavTelecom::MT_SINGLE_BLACKBOX_REQUEST) {
        Process(message.As<NNavTelecom::TSingleBlackBoxRequest>());
    } else if (messageType == NNavTelecom::MT_ONLINE_BLACKBOX_REQUEST) {
        Process(message.As<NNavTelecom::TOnlineBlackBoxRequest>());
    } else if (messageType == NNavTelecom::MT_PING) {
        Process(message.As<NNavTelecom::TPing>());
    } else if (messageType == NNavTelecom::MT_DIGITAL_OUTPUT_ANSWER) {
        Process(message.As<NNavTelecom::TDigitalOutputCommandAnswer>());
    } else if (messageType == NNavTelecom::MT_ICCID_ANSWER) {
        Process(message.As<NNavTelecom::TICCIDAnswer>());
    }
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Command(THolder<NProtocol::IMessage>&& message) {
    auto lock = Guard(MessagesQueueLock);

    auto messageType = message->GetMessageType();
    if (messageType != NVega::COMMAND_REQUEST) {
        return;
    }

    auto& payload = message->As<NDrive::NVega::TCommandRequest>();
    if (payload.Code != NVega::ECommandCode::SET_PARAM) {
        return;
    }

    NVega::TCommandRequest::TSetParameter argument;
    if (!payload.Argument.TryGet(argument) && argument.Id < NVega::DigitalOutput<1>() && argument.Id > NVega::DigitalOutput<4>()) {
        return;
    }

    ui8 outputNumber = argument.Id - NVega::DigitalOutput<1>() + 1;
    bool state = std::get<ui64>(argument.GetValue());
    auto answer = NProtocol::IPayload::Create<NNavTelecom::TDigitalOutputCommandRequest>();
    answer->Number = outputNumber;
    answer->State = state;

    Send<NNavTelecom::TMessage>(std::move(answer));
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::THandShakeRequest& /*message*/) {
    auto answer = NProtocol::IPayload::Create<NNavTelecom::THandShakeAnswer>();

    Send<NNavTelecom::TMessage>(std::move(answer));
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::TProtocolSettingRequest& message) {
    auto answer = NProtocol::IPayload::Create<NNavTelecom::TProtocolSettingAnswer>();

    SetBitField(message.BitField);

    answer->Protocol = message.Protocol;
    answer->ProtocolVersion = message.ProtocolVersion;
    answer->StructVersion = message.StructVersion;

    Send<NNavTelecom::TMessage>(std::move(answer));

    auto command = NProtocol::IPayload::Create<NDrive::NNavTelecom::TICCIDRequest>();
    AddCommand(std::move(command));
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::TBlackBoxRequest& message) {
    auto answer = NProtocol::IPayload::Create<NNavTelecom::TBlackBoxAnswer>();
    answer->Count = message.Count;

    CheckCommand(std::move(answer));
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::TAdditionalBlackBoxRequest& message) {
    auto answer = NProtocol::IPayload::Create<NNavTelecom::TAdditionalBlackBoxAnswer>();
    answer->Count = message.Count;

    CheckCommand(std::move(answer));
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::TSingleBlackBoxRequest& message) {
    auto answer = NProtocol::IPayload::Create<NNavTelecom::TSingleBlackBoxAnswer>();
    answer->EventIndex = message.EventIndex;

    CheckCommand(std::move(answer));
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::TAdditionalSingleBlackBoxRequest& message) {
    auto answer = NProtocol::IPayload::Create<NNavTelecom::TAdditionalSingleBlackBoxAnswer>();
    answer->EventIndex = message.EventIndex;

    CheckCommand(std::move(answer));
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::TOnlineBlackBoxRequest& /*message*/) {
    auto answer = NProtocol::IPayload::Create<NNavTelecom::TOnlineBlackBoxAnswer>();
    CheckCommand(std::move(answer));
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::TPing& /*message*/) {
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::TDigitalOutputCommandAnswer& /*message*/) {
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Process(const NNavTelecom::TICCIDAnswer& /*message*/) {
    CheckCommand();
}

void NDrive::NNavTelecom::TNavTelecomProtocol::AddCommand(THolder<NProtocol::IPayload> command) {
    auto guard = Guard(MessagesQueueLock);
    MessagesQueue.emplace(std::move(command));
}

void NDrive::NNavTelecom::TNavTelecomProtocol::CheckCommand(THolder<NProtocol::IPayload> answer /*= nullptr*/) {
    auto guard = Guard(MessagesQueueLock);

    THolder<NProtocol::IPayload> result;

    if (!MessagesQueue.empty()) {
        result = std::move(MessagesQueue.front());
        MessagesQueue.pop();

        if (answer) {
            INFO_LOG << "navtelecom replace answer to message queue: " << answer->DebugString() << Endl;
            MessagesQueue.emplace(std::move(answer));
        }

        if (result) {
            INFO_LOG << "navtelecom get message from queue: " << result->DebugString() << Endl;
        }
    } else {
        INFO_LOG << "navtelecom message queue is empty" << Endl;
    }

    if (!result) {
        result = std::move(answer);
    }

    if (result) {
        INFO_LOG << "navtelecom answer message: " << result->DebugString() << Endl;
        Send<NNavTelecom::TMessage>(std::move(result));
    }
}

void NDrive::NNavTelecom::TNavTelecomProtocol::Drop() noexcept {
}

NDrive::NProtocol::TTaskPtr NDrive::NNavTelecom::TNavTelecomProtocol::CreateCommand(const TString& id, NProtocol::ECommandCode command, NProtocol::TArgument argument, const TCommandOptions&) {
    if (command != NVega::SET_PARAM) {
        return {};
    }

    auto parameter = argument.TryGet<NVega::TCommandRequest::TSetParameter>();
    if (!parameter) {
        return {};
    }

    ui16 min = NDrive::NVega::DigitalOutput<1>();
    ui16 max = NDrive::NVega::DigitalOutput<4>();

    if (parameter->Id < min || parameter->Id > max) {
        return {};
    }

    auto value = parameter->GetValue();
    bool state = false;
    if (std::holds_alternative<ui64>(value)) {
        state = std::get<ui64>(value);
    }

    ui16 number = parameter->Id - NDrive::NVega::DigitalOutput<1>() + 1;

    auto payload = NProtocol::IPayload::Create<NNavTelecom::TDigitalOutputCommandRequest>();
    payload->Number = number;
    payload->State = state;
    AddCommand(std::move(payload));

    return MakeIntrusive<TNavTelecomCommand>(id, command, std::move(argument));
}
