#include "pin_resetter.h"

#include "logging.h"
#include "scenarios.h"
#include "server.h"

#include <drive/telematics/server/common/settings.h>

#include <drive/telematics/protocol/settings.h>
#include <drive/telematics/protocol/vega.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/library/json/cast.h>

namespace NDrive {
    const TString TPinResetter::IMEIWhitelistSettingName = "pin_resetter:imei_whitelist";
    const TString TPinResetter::DefaultPin = "new_car";

    TPinResetter::TPinResetter(const TTelematicsServer& server, TAtomicSharedPtr<TTelematicsDynamicSettings> dynamicSettings, const TString& imei, TInstant now, TDuration interval)
        : TGlobalScheduler::TScheduledItem<TPinResetter>(server.Name(),
                                    TStringBuilder() << "pin_resetter:" << imei << ":" << static_cast<const void*>(this), now)
        , Server(server)
        , DynamicSettings(dynamicSettings)
        , IMEI(imei)
        , Interval(interval)
    {
        Y_ENSURE(DynamicSettings);
    }

    void TPinResetter::Process(void* /*threadSpecificResource*/) {
        auto cleanup = Hold(this);
        auto maybeIMEIWhiteListStr = DynamicSettings->template Get<TString>(IMEIWhitelistSettingName);
        if (!maybeIMEIWhiteListStr) {
            return;
        }
        TSet<TString> IMEIWhiteList;
        NJson::TJsonValue value;
        if (!NJson::ReadJsonFastTree(*maybeIMEIWhiteListStr, &value) || !NJson::TryFromJson(value, IMEIWhiteList)) {
            ALERT_LOG << "can't parse telematics setting: " << IMEIWhitelistSettingName << Endl;
        }
        if (!IMEIWhiteList.contains(IMEI)) {
            return;
        }
        auto connection = Server.GetConnection(IMEI);
        if (!connection) {
            return;
        }

        TVector<TSensorId> sensorIds {
            {NVega::THardPasswordParameter::GetId(), 1},
            {NVega::TWiredPasswordParameter::GetId(), 0},
        };
        for (auto sensorId : sensorIds) {
            NVega::TCommandRequest::TSetParameter parameter{};
            parameter.Id = sensorId.Id;
            parameter.SubId = sensorId.SubId;
            NVega::THardPasswordParameter hardPasswordParam;
            hardPasswordParam.Value.Set(DefaultPin);
            parameter.SetValue(hardPasswordParam.Dump());

            NVega::TArgument argument;
            argument.Set(parameter);
            TCommandOptions options;
            auto id = NProtocol::GenerateId(IMEI);
            auto handler = CreateCommand(id, NVega::ECommandCode::SCENARIO_POLITE_SET_PARAM, argument, options);
            connection->AddHandler(std::move(handler));
        }
    }

    THolder<TGlobalScheduler::IScheduledItem> TPinResetter::GetNextScheduledItem(TInstant now) const {
        auto connection = Server.GetConnection(IMEI);
        if (!connection || !connection->Alive()) {
            return nullptr;
        }
        return MakeHolder<TPinResetter>(Server, DynamicSettings, IMEI, now + Interval, Interval);
    }
}
