#include "scenarios.h"

#include <drive/telematics/server/common/signals.h>

#include <drive/telematics/common/firmware.h>
#include <drive/telematics/protocol/settings.h>

#include <library/cpp/logger/global/global.h>

#include <rtline/library/json/parse.h>
#include <rtline/library/scheduler/global.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/digest/fnv.h>
#include <util/random/entropy.h>
#include <util/string/builder.h>

namespace {
    NDrive::TConditionalTask::TCheckResult BleExists(NDrive::TTelematicsConnection* connection) {
        if (!connection) {
            return NDrive::TConditionalTask::Failure("null Connection");
        }
        const auto& sensorsCache = connection->GetSensorsCache();
        const auto mac = sensorsCache.Get(BLE_EXT_BOARD_MAC);
        if (!mac) {
            return NDrive::TConditionalTask::Success("ble_mac is missing");
        }
        if (mac->IsZero()) {
            return NDrive::TConditionalTask::Success("ble_mac is zero");
        }
        return NDrive::TConditionalTask::Continue();
    }

    NDrive::TConditionalTask::TCheckResult EngineIsRunning(NDrive::TTelematicsConnection* connection) {
        if (!connection) {
            return NDrive::TConditionalTask::Failure("null Connection");
        }
        const auto& sensorsCache = connection->GetSensorsCache();
        const auto engine = sensorsCache.Get(CAN_ENGINE_IS_ON);
        if (!engine) {
            return NDrive::TConditionalTask::Continue(); // engine status is missing, assuming it is on
        }
        if (engine->ConvertTo<bool>()) {
            return NDrive::TConditionalTask::Continue();
        }
        return NDrive::TConditionalTask::Success();
    }

    NDrive::TConditionalTask::TCheckResult EngineIsOff(NDrive::TTelematicsConnection* connection) {
        if (!connection) {
            return NDrive::TConditionalTask::Failure("null Connection");
        }
        const auto& sensorsCache = connection->GetSensorsCache();
        const auto engine = sensorsCache.Get(CAN_ENGINE_IS_ON);
        if (!engine) {
            return NDrive::TConditionalTask::Failure("engine_on is missing"); // engine status is missing, assuming it is on
        }
        if (engine->ConvertTo<bool>()) {
            return NDrive::TConditionalTask::Failure("engine is on");
        }
        return NDrive::TConditionalTask::Continue();
    }

    NDrive::TConditionalTask::TCheckResult IsCarTanker(NDrive::TTelematicsConnection* connection) {
        if (!connection) {
            return NDrive::TConditionalTask::Failure("null Connection");
        }
        const auto& sensorsCache = connection->GetSensorsCache();
        const auto firmware = sensorsCache.Get(VEGA_MCU_FIRMWARE_VERSION);
        if (!firmware) {
            return NDrive::TConditionalTask::Failure( "cannot determine Firmware");
        }
        auto firmwareInfo = NDrive::NVega::ParseFirmwareInfo(firmware->ConvertTo<TString>());
        if (!firmwareInfo.CanBeTanker()) {
            return NDrive::TConditionalTask::Failure( "the firmware is not compatible with tanker firmware");
        }
        return NDrive::TConditionalTask::Continue();
    }

    NDrive::TConditionalTask::TCheckResult SpeedIsZero(NDrive::TTelematicsConnection* connection) {
        if (!connection) {
            return NDrive::TConditionalTask::Failure("null Connection");
        }

        const auto& sensorsCache = connection->GetSensorsCache();
        const auto current = sensorsCache.Get(VEGA_SPEED);
        if (!current) {
            return NDrive::TConditionalTask::Failure("current speed sensor is missing");
        }

        const auto now = Now();
        const auto threshold = now - TDuration::Seconds(30);
        if (current->Timestamp < threshold) {
            return NDrive::TConditionalTask::Failure("current speed sensor is outdated: " + ToString(current->Timestamp.Get()));
        }

        auto currentSpeed = current->ConvertTo<double>();
        if (currentSpeed > 0.001) {
            return NDrive::TConditionalTask::Failure("current speed is nonzero: " + ToString(currentSpeed));
        }

        auto previous = sensorsCache.Get(VEGA_SPEED, threshold);
        if (!previous) {
            return NDrive::TConditionalTask::Failure("previous speed sensor is missing");
        }

        auto previousSpeed = previous->ConvertTo<double>();
        if (previousSpeed > 0.001) {
            return NDrive::TConditionalTask::Failure("previous speed is nonzero: " + ToString(previousSpeed));
        }

        return NDrive::TConditionalTask::Continue();
    }

    NDrive::TConditionalTask::TCheckResult Stationary(NDrive::TTelematicsConnection* connection) {
        return EngineIsOff(connection) & SpeedIsZero(connection);
    }

    NDrive::TConditionalTask::TCheckResult IsCarSupportFuelUpdate(NDrive::TTelematicsConnection* connection) {
        if (!connection) {
            return NDrive::TConditionalTask::Failure("null Connection");
        }
        const auto& sensorsCache = connection->GetSensorsCache();
        const auto firmware = sensorsCache.Get(VEGA_MCU_FIRMWARE_VERSION);
        if (!firmware) {
            return NDrive::TConditionalTask::Failure( "cannot determine Firmware");
        }
        auto firmwareInfo = NDrive::NVega::ParseFirmwareInfo(firmware->ConvertTo<TString>());
        if (!firmwareInfo.CanUpdateFuel()) {
            return NDrive::TConditionalTask::Failure( "the firmware is not support fuel auto update");
        }
        return Stationary(connection);
    }

    NDrive::TConditionalTask::TCondition SensorIsMissing(NDrive::TSensorId sensorId) {
        return [sensorId](NDrive::TTelematicsConnection* connection) {
            if (!connection) {
                return NDrive::TConditionalTask::Failure("null Connection");
            }
            const auto& sensorsCache = connection->GetSensorsCache();
            const auto sensor = sensorsCache.Get(sensorId.Id, sensorId.SubId);
            if (sensor) {
                return NDrive::TConditionalTask::Success(TStringBuilder() << sensorId << " is found");
            }
            return NDrive::TConditionalTask::Continue();
        };
    }

    NDrive::TConditionalTask::TCondition SensorIsNotEqual(NDrive::TSensor expected) {
        return [expected = std::move(expected)](NDrive::TTelematicsConnection* connection) {
            if (!connection) {
                return NDrive::TConditionalTask::Failure("null Connection");
            }
            const auto& sensorsCache = connection->GetSensorsCache();
            const auto current = sensorsCache.Get(expected.Id, expected.SubId);
            if (!current) {
                return NDrive::TConditionalTask::Failure(TStringBuilder() << static_cast<NDrive::TSensorId>(expected) << " is not found");
            }
            Y_ASSERT(*current == expected);
            if (current->Value == expected.Value) {
                return NDrive::TConditionalTask::Success(
                    TStringBuilder() << static_cast<NDrive::TSensorId>(expected) << " has expected value " << expected.GetJsonValue().GetStringRobust()
                );
            }
            return NDrive::TConditionalTask::Continue();
        };
    }

    NDrive::TConditionalTask::TCondition FirmwareTypeIsEqual(NDrive::NVega::TFirmwareInfo::EType expectedFirmwareType) {
        using namespace NDrive;
        return [expectedFirmwareType](TTelematicsConnection* connection) {
            if (!connection) {
                return TConditionalTask::Failure("null connection");
            }
            const auto& sensorsCache = connection->GetSensorsCache();
            const auto maybeFirmwareSensor = sensorsCache.Get(VEGA_MCU_FIRMWARE_VERSION);
            if (!maybeFirmwareSensor) {
                return TConditionalTask::Failure(TStringBuilder() << static_cast<TSensorId>(VEGA_MCU_FIRMWARE_VERSION) << " sensor wasn't found");
            }
            auto currentFirmwareType = NVega::ParseFirmwareInfo(std::get<TString>(maybeFirmwareSensor->Value)).Type;
            if (currentFirmwareType != expectedFirmwareType) {
                return TConditionalTask::Success(
                    TStringBuilder() << static_cast<TSensorId>(VEGA_MCU_FIRMWARE_VERSION)
                                     << " doesn't have expected value " << expectedFirmwareType << "; actual is " << currentFirmwareType
                );
            }
            return TConditionalTask::Continue();
        };
    }

    template <class F>
    NDrive::TTaskPtr RunIf(F&& f, NDrive::TTaskPtr task) {
        return MakeIntrusive<NDrive::TConditionalTask>(task, std::forward<F>(f));
    }

    template <class F>
    NDrive::TTaskPtr RunIfElse(F&& f, NDrive::TTaskPtr task, NDrive::TTaskPtr elseTask) {
        return MakeIntrusive<NDrive::TConditionalTask>(task, std::forward<F>(f), elseTask);
    }

    NDrive::TTaskPtr CanScript(const TString& id, NDrive::NVega::TCommandRequest::TCanScript::EType type, const NDrive::TCommandOptions& options) {
        using namespace NDrive;
        NProtocol::TArgument canScriptArgument;
        NVega::TCommandRequest::TCanScript request;
        request.Id = static_cast<ui8>(type);
        canScriptArgument.Set(request);
        return CreateCommand(id, NVega::ECommandCode::CAN_SCRIPT, canScriptArgument, options);
    }

    NDrive::TTaskPtr GetParameter(const TString& id, NDrive::TSensorId sensor, const NDrive::TCommandOptions& options) {
        NDrive::NVega::TCommandRequest::TGetParameter getParameter;
        getParameter.Id = sensor.Id;
        getParameter.SubId = sensor.SubId;
        NDrive::NProtocol::TArgument arg;
        arg.Set(getParameter);
        return NDrive::CreateCommand(id, NDrive::NVega::ECommandCode::GET_PARAM, std::move(arg), options);
    }

    NDrive::TTaskPtr SetParameter(const TString& id, NDrive::NProtocol::TArgument argument, const NDrive::TCommandOptions& options) {
        return NDrive::CreateCommand(id, NDrive::NVega::ECommandCode::SET_PARAM, std::move(argument), options);
    }

    template <class T>
    NDrive::TTaskPtr SetParameter(const TString& id, NDrive::TSensorId sensor, T value, const NDrive::TCommandOptions& options) {
        NDrive::TCommandOptions opts = options;
        opts.Retries = 1;
        NDrive::NVega::TCommandRequest::TSetParameter setParameter;
        setParameter.Id = sensor.Id;
        setParameter.SubId = sensor.SubId;
        setParameter.SetValue(std::move(value));
        NDrive::NProtocol::TArgument arg;
        arg.Set(setParameter);
        return SetParameter(id, std::move(arg), opts);
    }

    NDrive::TTaskPtr CreateToggleCommand(
        const TString& id,
        const TString& subId,
        NDrive::TSensorId sensor,
        TVector<TDuration>&& delayTable,
        const NDrive::TCommandOptions& options,
        bool controlOff = false
    ) {
        auto tries = options.Retries;
        auto opts = options;
        opts.Retries = 1;

        NDrive::TSequentialTask::TTasks subtasks;

        for (ui32 i = 0; i < tries; ++i) {
            subtasks.push_back(SetParameter<ui8>(id + "-" + subId + "-on-" + ToString(i), sensor, 1, opts));
            subtasks.push_back(SetParameter<ui8>(id + "-" + subId + "-off-" + ToString(i), sensor, 0, opts));
        }

        if (controlOff) {
            subtasks.push_back(SetParameter<ui8>(id + "-" + subId + "-off", sensor, 0, opts));
        }

        auto result = MakeIntrusive<NDrive::TSequentialTask>(id, std::move(subtasks));
        result->SetDelayTable(std::move(delayTable));
        return result;
    }
}

NDrive::TConditionalTask::TCheckResult NDrive::TConditionalTask::TCheckResult::operator&(const NDrive::TConditionalTask::TCheckResult& other) const {
    if (IsError()) {
        return *this;
    }
    if (other.IsError()) {
        return other;
    }
    if (IsSuccess() && other.IsSuccess()) {
        return NDrive::TConditionalTask::Success(Info + other.Info);
    }
    return Continue();
}

NDrive::TConditionalTask::TCheckResult NDrive::TConditionalTask::Continue() {
    return { EStatus::Processing };
}

NDrive::TConditionalTask::TCheckResult NDrive::TConditionalTask::Success(TString info) {
    return { EStatus::Success, std::move(info) };
}

NDrive::TConditionalTask::TCheckResult NDrive::TConditionalTask::Failure(TString info) {
    return { EStatus::Failure, std::move(info) };
}

NDrive::TConditionalTask::TConditionalTask(NDrive::TTaskPtr task, TCondition condition, NDrive::TTaskPtr elseTask)
    : TCommonTask(task ? task->GetId() : "unknown-conditional")
    , Task(task)
    , Condition(condition)
    , ElseTask(elseTask)
{
}

TVector<NDrive::NProtocol::TSequenceId> NDrive::TConditionalTask::GetExpectedSequenceIds() const {
    TVector<NDrive::NProtocol::TSequenceId> result;
    if (Task) {
        auto ids = Task->GetExpectedSequenceIds();
        result.insert(result.end(), ids.begin(), ids.end());
    }
    if (ElseTask) {
        auto ids = ElseTask->GetExpectedSequenceIds();
        result.insert(result.end(), ids.begin(), ids.end());
    }
    return result;
}

bool NDrive::TConditionalTask::DoOnAfterRegister(NDrive::TTelematicsConnection* connection) {
    CheckResult = Condition ? Condition(connection) : Continue();
    if (!CheckResult.ShouldContinue()) {
        if (ElseTask) {
            return ElseTask->OnAfterRegister(connection);
        }
        return true;
    }

    if (!Task) {
        return true;
    }
    return Task->OnAfterRegister(connection);
}

bool NDrive::TConditionalTask::DoOnMessage(NDrive::TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (!CheckResult.ShouldContinue()) {
        if (ElseTask) {
            AddEvent("doing else branch");
            bool result = ElseTask->OnMessage(connection, message);
            if (result) {
                CopyState(*ElseTask);
            }
            return result;
        }
        AddEvent("skipped by condition");
        if (CheckResult.Info) {
            SetLastInfo(CheckResult.Info);
        }
        SetStatus(CheckResult.Status);
        return true;
    }
    if (!Task) {
        AddEvent("null Task");
        SetStatus(EStatus::Failure);
        return true;
    }

    bool result = Task->OnMessage(connection, message);
    if (result) {
        CopyState(*Task);
    }
    return result;
}

void NDrive::TConditionalTask::DoOnTermination() {
    if (CheckResult.ShouldContinue()) {
        if (Task) {
            Task->OnTermination();
        }
    } else {
        if (ElseTask) {
            ElseTask->OnTermination();
        }
    }
}

NDrive::TMultiTask::TMultiTask(const TString& id, TTasks&& tasks)
    : TCommonTask(id)
    , Tasks(std::move(tasks))
    , Next(Tasks.begin())
{
    for (auto&& task : Tasks) {
        Y_ENSURE(task, "null task encountered");
        AddEvent(CreateEvent("register", task->GetId()));
    }
}

TVector<NDrive::NProtocol::TSequenceId> NDrive::TMultiTask::GetExpectedSequenceIds() const {
    TVector<NDrive::NProtocol::TSequenceId> result;
    for (auto&& task : Tasks) {
        if (!task) {
            continue;
        }
        auto ids = task->GetExpectedSequenceIds();
        if (result.empty()) {
            result = std::move(ids);
        } else {
            result.insert(result.end(), ids.begin(), ids.end());
        }
    }
    return result;
}

void NDrive::TMultiTask::SetDeadline(TInstant deadline) {
    TBase::SetDeadline(deadline);
    for (auto&& task : Tasks) {
        if (task) {
            task->SetDeadline(deadline);
        }
    }
}

bool NDrive::TMultiTask::Serializable() const {
    for (auto&& task : Tasks) {
        if (!task || !task->Serializable()) {
            return false;
        }
    }
    return true;
}

bool NDrive::TMultiTask::DoOnAfterRegister(TTelematicsConnection* connection) {
    return ScheduleNextEx(connection).second;
}

NJson::TJsonValue NDrive::TMultiTask::DoSerialize() const {
    NJson::TJsonValue result = TBase::DoSerialize();
    NJson::TJsonValue& tasks = result.InsertValue("tasks", NJson::JSON_ARRAY);
    for (auto&& task : Tasks) {
        if (task) {
            tasks.AppendValue(task->Serialize());
        }
    }
    NJson::TJsonValue& active = result.InsertValue("active", NJson::JSON_ARRAY);
    for (auto&& task : Active) {
        if (task) {
            active.AppendValue(task->GetId());
        }
    }
    NJson::TJsonValue& next = result.InsertValue("next", NJson::JSON_NULL);
    if (Next != Tasks.end()) {
        auto task = *Next;
        if (task) {
            next = task->GetId();
        }
    }
    NJson::TJsonValue& last = result.InsertValue("last", NJson::JSON_NULL);
    if (Last) {
        last = Last->GetId();
    }
    return result;
}

void NDrive::TMultiTask::DoDeserialize(const NJson::TJsonValue& value) {
    TBase::DoDeserialize(value);
    for (auto&& i : value["tasks"].GetArraySafe()) {
        auto task = TBase::Restore(i);
        Y_ENSURE(task, "cannot restore task from " << i.GetStringRobust());
        Tasks.push_back(std::move(task));
    }
    auto active = NJson::FromJson<TSet<TString>>(value["active"]);
    auto next = NJson::FromJson<TMaybe<TString>>(value["next"]);
    auto last = NJson::FromJson<TMaybe<TString>>(value["last"]);
    for (auto&& task : Tasks) {
        if (task && active.contains(task->GetId())) {
            Active.push_back(task);
        }
        if (task && last && task->GetId() == *last) {
            Last = task;
        }
    }
    Next = Tasks.end();
    for (auto i = Tasks.begin(); next && i != Tasks.end(); ++i) {
        auto task = *i;
        if (task && task->GetId() == *next) {
            Next = i;
            break;
        }
    }
}

bool NDrive::TSequentialTask::DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (IsValid()) {
        INFO_LOG << "TSequentialTask::DoOnMessage: " << GetId() << " / " << message.DebugString() << Endl;
        auto current = Feed(connection, message);
        if (!current) {
            INFO_LOG << "TSequentialTask::DoOnMessage return false: " << GetId() << " / " << message.DebugString() << Endl;
            return false;
        }

        AddEvent(CreateEvent("finished", current->GetId()));
        if (auto last = current->GetLastInfo()) {
            AddEvent(last);
            SetLastInfo(last);
        }
        auto status = current->GetStatus();
        switch (status) {
        case TCommonTask::EStatus::Success:
        {
            auto delay = GetDelay();
            auto time = delay ? (Now() + delay) : TInstant::Zero();
            if (ScheduleNext(connection, time)) {
                Index++;
                return false;
            } else {
                // fallthrough
            }
        }
        default:
            SetLast(current);
            SetStatus(status);
            return true;
        }
    } else {
        INFO_LOG << "FAIL TSequentialTask::DoOnMessage: " << GetId() << Endl;
        SetStatus(TCommonTask::EStatus::Failure);
        return true;
    }
}

NJson::TJsonValue NDrive::TSequentialTask::DoSerialize() const {
    NJson::TJsonValue result = TBase::DoSerialize();
    result["delay_table"] = NJson::ToJson(DelayTable);
    result["index"] = Index;
    return result;
}

void NDrive::TSequentialTask::DoDeserialize(const NJson::TJsonValue& value) {
    TBase::DoDeserialize(value);
    NJson::ReadField(value, "delay_table", DelayTable);
    NJson::ReadField(value, "index", Index);
}

bool NDrive::TFirstTask::DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (IsValid()) {
        auto current = Feed(connection, message);
        if (!current) {
            auto now = Now();
            auto hedge = GetNextHedge();
            if (now > hedge) {
                if (auto task = ScheduleNext(connection)) {
                    INFO_LOG << connection->DebugString() << ": invoking hedge subtask " << task->GetId() << Endl;
                    Index++;
                    TTelematicsUnistatSignals::Get().HandlersHedge.Signal(1);
                }
            }
            return false;
        }

        AddEvent(CreateEvent("finished", current->GetId()));
        switch (current->GetStatus()) {
        case EStatus::Retry:
            if (ScheduleNext(connection, Now() + GetTimeout())) {
                Index++;
                TTelematicsUnistatSignals::Get().HandlersRetry.Signal(1);
                return false;
            } else {
                // fallthrough
            }
        default:
            SetLast(current);
            CopyState(*current);
            return true;
        }
    } else {
        SetStatus(TCommonTask::EStatus::Failure);
        return true;
    }
}

NJson::TJsonValue NDrive::TFirstTask::DoSerialize() const {
    NJson::TJsonValue result = TBase::DoSerialize();
    result["index"] = Index;
    return result;
}

void NDrive::TFirstTask::DoDeserialize(const NJson::TJsonValue& value) {
    TBase::DoDeserialize(value);
    NJson::ReadField(value, "index", Index);
}

void NDrive::TMultiTask::DoOnTermination() {
    for (auto&& task : Active) {
        CHECK_WITH_LOG(task);
        task->OnTermination();
    }
    for (auto i = Next; i != Tasks.end(); ++i) {
        auto task = *i;
        CHECK_WITH_LOG(task);
        task->OnTermination();
    }
}

NJson::TJsonValue NDrive::TMultiTask::CreateEvent(const TString& action, const TString& id) const {
    NJson::TJsonValue result;
    result["action"] = action;
    result["id"] = id;
    return result;
}

NDrive::TTaskPtr NDrive::TMultiTask::Feed(TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    for (auto i = Active.begin(); i != Active.end(); ++i) {
        auto task = *i;
        CHECK_WITH_LOG(task);
        if (task->OnMessage(connection, message)) {
            Active.erase(i);
            return task;
        }
    }
    return nullptr;
}

NDrive::TTaskPtr NDrive::TMultiTask::ScheduleNext(TTelematicsConnection* connection, TInstant time) {
    return ScheduleNextEx(connection, time).first;
}

std::pair<NDrive::TTaskPtr, bool> NDrive::TMultiTask::ScheduleNextEx(TTelematicsConnection* connection, TInstant time) {
    if (Next == Tasks.end()) {
        return { nullptr, true };
    }

    auto next = *Next;
    bool tick = false;
    if (time) {
        CHECK_WITH_LOG(connection);
        connection->ScheduleHandler(next, time);
    } else {
        tick = next->OnAfterRegister(connection);
    }
    Active.push_back(next);
    Next++;
    return { next, tick };
}

void NDrive::TMultiTask::SetLast(TTaskPtr task) {
    Last = task;
}

bool NDrive::TMultiTask::IsInitialized() const {
    return Next != Tasks.begin() || Tasks.empty();
}

bool NDrive::TMultiTask::IsValid() const {
    return !Active.empty() || !IsInitialized();
}

size_t NDrive::TMultiTask::GetTasksCount() const {
    return Tasks.size();
}

bool NDrive::TParallelTask::DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) {
    if (IsValid()) {
        INFO_LOG << "TParallelTask::DoOnMessage " << GetId() << " / " << message.DebugString() << Endl;
        auto current = Feed(connection, message);
        if (!current) {
            INFO_LOG << "TParallelTask::Feed return false: " << GetId() << " / " << message.DebugString() << Endl;
            return false;
        }
        ++FinishedTasksCount;
        AddEvent(CreateEvent("finished", current->GetId()));
        if (auto last = current->GetLastInfo()) {
            AddEvent(last);
            SetLastInfo(last);
        }
        SetLast(current);

        auto status = current->GetStatus();
        if (status != TCommonTask::EStatus::Success) {
            // don't wait for other tasks, if one failed
            SetStatus(status);
            return true;
        }
        if (FinishedTasksCount == GetTasksCount()) {
            SetStatus(TCommonTask::EStatus::Success);
            return true;
        }
        return false;
    } else {
        INFO_LOG << "FAIL TParallelTask::DoOnMessage: " << GetId() << Endl;
        SetStatus(TCommonTask::EStatus::Failure);
        return true;
    }
}

bool NDrive::TParallelTask::DoOnAfterRegister(TTelematicsConnection* connection) {
    bool ready = false;
    for (size_t i = 0; i < GetTasksCount(); ++i) {
        if (ScheduleNext(connection)) {
            ready = true;
        }
    }
    return ready;
}

bool NDrive::TNoWaitMultiTask::DoOnAfterRegister(TTelematicsConnection* connection) {
    INFO_LOG << "TNoWaitMultiTask::DoOnAfterRegister: " << GetId() << Endl;
    for (auto it = std::make_move_iterator(TasksBegin()); it != std::make_move_iterator(TasksEnd()); ++it) {
        connection->AddHandler(*it);
    }
    return true;
}

bool NDrive::TNoWaitMultiTask::DoOnMessage(TTelematicsConnection* /* connection */, const NProtocol::IMessage& message) {
    DEBUG_LOG << "TNoWatiMultiTask::DoOnMessage " << GetId() << " / " << message.DebugString() << Endl;
    SetStatus(TCommonTask::EStatus::Success);
    return true;
}

NDrive::TTaskPtr NDrive::CreateBasicCommand(const TString& id, NVega::ECommandCode command, NProtocol::TArgument argument, const TCommandOptions& options) {
    Y_UNUSED(options);
    switch (command) {
    case NVega::ECommandCode::GET_PARAM:
        return MakeIntrusive<TGetParameterTask>(id, std::move(argument));
    case NVega::ECommandCode::OBD_FORWARD_CONFIG:
        return MakeIntrusive<TCanSetupTask>(id, argument);
    default:
        return MakeIntrusive<TSendCommandTask>(id, command, std::move(argument));
    }
}

NDrive::TTaskPtr NDrive::CreateCommand(const TString& id, NVega::ECommandCode command, NProtocol::TArgument argument, const TCommandOptions& options) {
    switch (command) {
    case NVega::ECommandCode::SCENARIO_STOP_WARMING_AND_OPEN_DOORS: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            RunIf(EngineIsRunning, NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_STOP_WARMING, argument, options)),
            NDrive::CreateCommand(id, NVega::ECommandCode::OPEN_DOORS, argument, options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_STOP_WARMING_AND_END_OF_LEASE: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            RunIf(EngineIsRunning, NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_STOP_WARMING, argument, options)),
            NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_END_OF_LEASE, argument, options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_UNLOCK_DOORS_AND_HOOD: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            NDrive::CreateCommand(id, NVega::ECommandCode::OPEN_DOORS, argument, options),
            NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_UNLOCK_HOOD, argument, options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_LOCK_DOORS_AND_HOOD: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            NDrive::CreateCommand(id, NVega::ECommandCode::CLOSE_DOORS, argument, options),
            NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_LOCK_HOOD, argument, options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_UNLOCK_HOOD_AND_START_OF_LEASE: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_START_OF_LEASE, argument, options),
            NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_UNLOCK_HOOD, argument, options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_STOP_WARMING_END_OF_LEASE_AND_LOCK_HOOD: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            RunIf(EngineIsRunning, NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_STOP_WARMING, argument, options)),
            NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_END_OF_LEASE, argument, options),
            NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_LOCK_HOOD, argument, options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_ENABLE_LONGHORN: {
        return SetParameter<ui8>(id, NVega::DigitalOutput<3>(), 1, options);
    }
    case NVega::ECommandCode::SCENARIO_DISABLE_LONGHORN: {
        return SetParameter<ui8>(id, NVega::DigitalOutput<3>(), 0, options);
    }
    case NVega::ECommandCode::SCENARIO_POLITE_FORCED_END_OF_LEASE: {
        auto opts = options;
        opts.Retries = 1;
        return RunIf(SpeedIsZero, NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_FORCED_END_OF_LEASE, std::move(argument), opts));
    }
    case NVega::ECommandCode::SCENARIO_BLE_RESET: {
        auto passkey = FnvHash<ui32>(id) % 1000000;
        auto passkeyString = Sprintf("%06d", passkey);
        NVega::TBlePasskeyParameter value;
        value.Set(passkeyString);
        NVega::TBleSessionKeyParameter sessionKey;
        EntropyPool().Load(sessionKey.Value.data(), sessionKey.Value.size());
        auto subtasks = NDrive::TSequentialTask::TTasks{
            SetParameter(id, NVega::BlePasskey, value.Raw(), options),
            SetParameter(id, NVega::BleSessionKey, sessionKey.Raw(), options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_PREPARE: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            RunIf(EngineIsRunning, NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_STOP_WARMING, argument, options)),
            NDrive::CreateCommand(id, NVega::ECommandCode::OPEN_DOORS, argument, options),
            RunIf(BleExists, NDrive::CreateCommand(id, NVega::ECommandCode::SCENARIO_BLE_RESET, argument, options)),
        };
        Yensured(subtasks.back())->SetIgnoreFailure(true);
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_RESET: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            RunIf(EngineIsRunning, NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_STOP_WARMING, argument, options)),
            NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_END_OF_LEASE, argument, options),
            RunIf(BleExists, NDrive::CreateCommand(id, NVega::ECommandCode::SCENARIO_BLE_RESET, argument, options)),
        };
        Yensured(subtasks.back())->SetIgnoreFailure(true);
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_FORCED_RESET: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            NDrive::CreateCommand(id, NVega::ECommandCode::SCENARIO_POLITE_FORCED_END_OF_LEASE, argument, options),
            NDrive::CreateCommand(id, NVega::ECommandCode::SCENARIO_BLE_RESET, argument, options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_POLITE_RESTART: {
        auto opts = options;
        opts.Retries = 1;
        return RunIf(Stationary, NDrive::CreateCommand(id, NVega::ECommandCode::RESTART, std::move(argument), opts));
    }
    case NVega::ECommandCode::SCENARIO_DIN2_SHUTDOWN: {
        TVector delayTable = {
            TDuration::Seconds(2),
            TDuration::Seconds(3),
            TDuration::Seconds(2)
        };
        auto opts = options;
        opts.Retries = 2;
        return CreateToggleCommand(id, "di2", NVega::DigitalOutput<2>(), std::move(delayTable), opts);
    }
    case NVega::ECommandCode::SCENARIO_POLITE_SET_PARAM: {
        const auto& setParameter = argument.Get<NDrive::NVega::TCommandRequest::TSetParameter>();
        NDrive::TSensorId sensorId(setParameter.Id, setParameter.SubId);
        NDrive::TSensor sensor(sensorId);
        sensor.Value = NDrive::SensorValueFromRef(setParameter.GetValue());
        auto subtasks = NDrive::TSequentialTask::TTasks{
            RunIf(SensorIsMissing(sensor), GetParameter(id + "-get", sensor, options)),
            RunIf(SensorIsNotEqual(sensor), SetParameter(id + "-set", std::move(argument), options))
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_BLE_RESET_FORCED_END_OF_LEASE: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            NDrive::CreateCommand(id, NVega::ECommandCode::SCENARIO_BLE_RESET, argument, options),
            NDrive::CreateCommand(id, NVega::ECommandCode::YADRIVE_FORCED_END_OF_LEASE, argument, options)
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_QUERY_BEACONS: {
        auto subtasks = NDrive::TMultiTask::TTasks{
            GetParameter(id + "-1", BLE_EXT_BOARD_BEACONS_INFO1, options),
            GetParameter(id + "-2", BLE_EXT_BOARD_BEACONS_INFO2, options),
            GetParameter(id + "-3", BLE_EXT_BOARD_BEACONS_INFO3, options),
            GetParameter(id + "-4", BLE_EXT_BOARD_BEACONS_INFO4, options),
            GetParameter(id + "-5", BLE_EXT_BOARD_BEACONS_INFO5, options),
            GetParameter(id + "-6", BLE_EXT_BOARD_BEACONS_INFO6, options),
            GetParameter(id + "-7", BLE_EXT_BOARD_BEACONS_INFO7, options),
            GetParameter(id + "-8", BLE_EXT_BOARD_BEACONS_INFO8, options),
            GetParameter(id + "-9", BLE_EXT_BOARD_BEACONS_INFO9, options),
            GetParameter(id + "-10", BLE_EXT_BOARD_BEACONS_INFO10, options)
        };
        return MakeIntrusive<TParallelTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_QUERY_BEACONS_NO_WAIT: {
        auto subtasks = NDrive::TMultiTask::TTasks{
            GetParameter(id + "-1", BLE_EXT_BOARD_BEACONS_INFO1, options),
            GetParameter(id + "-2", BLE_EXT_BOARD_BEACONS_INFO2, options),
            GetParameter(id + "-3", BLE_EXT_BOARD_BEACONS_INFO3, options),
            GetParameter(id + "-4", BLE_EXT_BOARD_BEACONS_INFO4, options),
            GetParameter(id + "-5", BLE_EXT_BOARD_BEACONS_INFO5, options),
            GetParameter(id + "-6", BLE_EXT_BOARD_BEACONS_INFO6, options),
            GetParameter(id + "-7", BLE_EXT_BOARD_BEACONS_INFO7, options),
            GetParameter(id + "-8", BLE_EXT_BOARD_BEACONS_INFO8, options),
            GetParameter(id + "-9", BLE_EXT_BOARD_BEACONS_INFO9, options),
            GetParameter(id + "-10", BLE_EXT_BOARD_BEACONS_INFO10, options)
        };
        return MakeIntrusive<TNoWaitMultiTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_START_IGNITION: {
        return SetParameter<ui8>(id, NVega::DigitalOutput<1>(), 1, options);
    }
    case NVega::ECommandCode::SCENARIO_STOP_IGNITION: {
        return SetParameter<ui8>(id, NVega::DigitalOutput<1>(), 0, options);
    }
    case NVega::ECommandCode::SCENARIO_QUERY_FUEL_LEVEL: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            NDrive::CreateCommand(id, NVega::ECommandCode::SCENARIO_START_IGNITION, argument, options),
            GetParameter(id, CAN_FUEL_LEVEL_P, options),
            NDrive::CreateCommand(id, NVega::ECommandCode::SCENARIO_STOP_IGNITION, argument, options)
        };
        auto result = MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
        const auto& queryFuelLevelArgs = argument.Get<NDrive::NVega::TCommandRequest::TQueryFuelLevelArgs>();
        result->SetDelayTable({
            queryFuelLevelArgs.GetIgnitionDuration()
        });
        return result;
    }
    case NVega::ECommandCode::SCENARIO_OBD_REQUEST: {
        const auto& arg = argument.Get<NDrive::NVega::TCommandRequest::TObdRequest>();
        if (arg.ResponseId) {
            auto request = MakeIntrusive<TCanRequestTask>(id + "-request", arg.Id, arg.Index, arg.Data);
            request->SetResponseId(arg.ResponseId);

            NDrive::NVega::TCommandRequest::TCustomObdForwardConfig config;
            config.Duration = options.Timeout.Seconds();
            config.Value = arg.ResponseId;
            config.Mask = 0x7FF;
            auto setup = MakeIntrusive<TCanSetupTask>(id + "-setup", config);
            auto subtasks = NDrive::TSequentialTask::TTasks{
                setup,
                request,
            };
            return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
        }
        auto result = MakeIntrusive<TCanRequestTask>(id, arg.Id, arg.Index, arg.Data);
        return result;
    }
    case NVega::ECommandCode::SCENARIO_POLITE_WIRELESS_BLOCK: {
        return RunIf(Stationary, CreateCommand(id, NVega::ECommandCode::SCENARIO_WIRELESS_BLOCK, argument, options));
    }
    case NVega::ECommandCode::SCENARIO_WIRELESS_BLOCK: {
        return SetParameter<ui8>(id, VEGA_NRF_DESIRED_RELAY_STATE, 1, options);
    }
    case NVega::ECommandCode::SCENARIO_WIRELESS_UNBLOCK: {
        return SetParameter<ui8>(id, VEGA_NRF_DESIRED_RELAY_STATE, 0, options);
    }
    case NVega::ECommandCode::SCENARIO_OPEN_DOORS: {
        return RunIfElse(
            FirmwareTypeIsEqual(NVega::TFirmwareInfo::EType::MT32K_MTX),
            CanScript(id, NVega::TCommandRequest::TCanScript::EType::OPEN_DOORS, options),
            CreateCommand(id, NVega::ECommandCode::OPEN_DOORS, argument, options)
        );
    }
    case NVega::ECommandCode::SCENARIO_CLOSE_DOORS: {
        return RunIfElse(
            FirmwareTypeIsEqual(NVega::TFirmwareInfo::EType::MT32K_MTX),
            CanScript(id, NVega::TCommandRequest::TCanScript::EType::CLOSE_DOORS, options),
            CreateCommand(id, NVega::ECommandCode::CLOSE_DOORS, argument, options)
        );
    }
    case NVega::ECommandCode::SCENARIO_RAPID_OFF_EMERGENCY_LIGHTS: {
        TVector<TDuration> delayTable = {
            TDuration::Seconds(1),
            TDuration::Seconds(1)
        };
        auto opts = options;
        opts.Retries = 1;
        return CreateToggleCommand(id, "el", NVega::DigitalOutput<9>(), std::move(delayTable), opts, true);
    }
    case NVega::ECommandCode::SCENARIO_POLITE_WIRED_BLOCK: {
        return RunIf(Stationary, CreateCommand(id, NVega::ECommandCode::SCENARIO_WIRED_BLOCK, argument, options));
    }
    case NVega::ECommandCode::SCENARIO_WIRED_BLOCK: {
        return SetParameter<ui8>(id, NVega::DigitalOutput<4>(), 1, options);
    }
    case NVega::ECommandCode::SCENARIO_WIRED_UNBLOCK: {
        return SetParameter<ui8>(id, NVega::DigitalOutput<4>(), 0, options);
    }
    case NVega::ECommandCode::SCENARIO_ALL_BLOCK: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            CreateCommand(id + "-wireless", NVega::ECommandCode::SCENARIO_WIRELESS_BLOCK, argument, options),
            CreateCommand(id + "-wired", NVega::ECommandCode::SCENARIO_WIRED_BLOCK, argument, options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_ALL_UNBLOCK: {
        auto subtasks = NDrive::TSequentialTask::TTasks{
            CreateCommand(id + "-wired", NVega::ECommandCode::SCENARIO_WIRED_UNBLOCK, argument, options),
            CreateCommand(id + "-wireless", NVega::ECommandCode::SCENARIO_WIRELESS_UNBLOCK, argument, options),
        };
        return MakeIntrusive<TSequentialTask>(id, std::move(subtasks));
    }
    case NVega::ECommandCode::SCENARIO_POLITE_ALL_BLOCK: {
        return RunIf(Stationary, CreateCommand(id, NVega::ECommandCode::SCENARIO_ALL_BLOCK, argument, options));
    }
    case NVega::ECommandCode::SCENARIO_FTANKER_SET_GASOLINE_LITERS:{
        const auto& FuelQuantity = argument.Get<NDrive::NVega::TCommandRequest::TTankerFuelQuantity>();
        return RunIf(IsCarTanker, SetParameter<decltype(FuelQuantity.GetFuelQuantity())>(id, FuelQuantity.GetGasolineSensorId(), FuelQuantity.GetFuelQuantity(), options));
    }
    case NVega::ECommandCode::SCENARIO_FTANKER_SET_DIESEL_LITERS:{
        const auto& FuelQuantity = argument.Get<NDrive::NVega::TCommandRequest::TTankerFuelQuantity>();
        return RunIf(IsCarTanker, SetParameter<decltype(FuelQuantity.GetFuelQuantity())>(id, FuelQuantity.GetDieselSensorId(), FuelQuantity.GetFuelQuantity(), options));
    }
    case NVega::ECommandCode::SCENARIO_START_FUEL_AUTO_UPDATE:{
        const auto& UpdatePeriod = argument.Get<NDrive::NVega::TCommandRequest::TFuelUpdatePeriod>();
        return RunIf(IsCarSupportFuelUpdate, SetParameter<decltype(UpdatePeriod.GetPeriod())>(id, UpdatePeriod.GetSensorId(), UpdatePeriod.GetPeriod(), options));
    }
    case NVega::ECommandCode::SCENARIO_STOP_FUEL_AUTO_UPDATE:{
        NDrive::NVega::TCommandRequest::TFuelUpdatePeriod UpdatePeriod;
        UpdatePeriod.SetPeriodForStop();
        return RunIf(IsCarSupportFuelUpdate, SetParameter<decltype(UpdatePeriod.GetPeriod())>(id, UpdatePeriod.GetSensorId(), UpdatePeriod.GetPeriod(), options));
    }
    default:
        break;
    }

    if (options.Retries > 1) {
        const TString subid = id + '-' + ToString(command);
        TMultiTask::TTasks subtasks;
        for (ui32 i = 0; i < options.Retries; ++i) {
            subtasks.push_back(
                CreateBasicCommand(subid + '-' + ToString(i), command, argument, options)
            );
        }
        return MakeIntrusive<TFirstTask>(id, std::move(subtasks));
    } else {
        return CreateBasicCommand(id, command, std::move(argument), options);
    }
}

NDrive::TCommonTask::TFactory::TRegistrator<NDrive::TFirstTask> FirstTaskRegistrator(NDrive::TFirstTask::Type());
NDrive::TCommonTask::TFactory::TRegistrator<NDrive::TSequentialTask> SequentialTaskRegistrator(NDrive::TSequentialTask::Type());
