#pragma once

#include "handlers.h"

namespace NDrive {
    class TConditionalTask: public TCommonTask {
    public:
        struct TCheckResult {
        public:
            EStatus Status = EStatus::New;
            TString Info;

        public:
            inline TCheckResult() = default;
            inline TCheckResult(EStatus status, TString info = {})
                : Status(status)
                , Info(std::move(info))
            {
            }

            inline bool ShouldContinue() const {
                return Status == EStatus::Processing;
            }

            inline bool IsError() const {
                return Status != EStatus::Processing && Status != EStatus::Success;
            }

            inline bool IsSuccess() const {
                return Status == EStatus::Success;
            }

            TCheckResult operator&(const TCheckResult& other) const;
        };

        using TCondition = std::function<TCheckResult(TTelematicsConnection*)>;

    public:
        static TCheckResult Continue();
        static TCheckResult Success(TString info = {});
        static TCheckResult Failure(TString info);

    public:
        TConditionalTask(TTaskPtr task, TCondition condition, TTaskPtr elseTask = nullptr);

        virtual TString GetType() const override {
            return "ConditionalTask";
        }

        virtual TVector<NProtocol::TSequenceId> GetExpectedSequenceIds() const override;

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;
        virtual void DoOnTermination() override;

    private:
        TTaskPtr Task;
        TCondition Condition;
        TTaskPtr ElseTask;

        TCheckResult CheckResult;
    };

    class TMultiTask: public TCommonTask {
    private:
        using TBase = TCommonTask;

    public:
        using TTasks = TVector<TTaskPtr>;

    public:
        TMultiTask() = default;
        TMultiTask(const TString& id, TTasks&& tasks);

        virtual TVector<NProtocol::TSequenceId> GetExpectedSequenceIds() const override;
        virtual void SetDeadline(TInstant deadline) override;
        virtual bool Serializable() const override;

    protected:
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
        virtual void DoOnTermination() override;

        virtual NJson::TJsonValue DoSerialize() const override;
        virtual void DoDeserialize(const NJson::TJsonValue& value) override;

        NJson::TJsonValue CreateEvent(const TString& action, const TString& id) const;
        TTaskPtr Feed(TTelematicsConnection* connection, const NProtocol::IMessage& message);
        TTaskPtr ScheduleNext(TTelematicsConnection* connection, TInstant time = TInstant::Zero());
        std::pair<TTaskPtr, bool> ScheduleNextEx(TTelematicsConnection* connection, TInstant time = TInstant::Zero());
        void SetLast(TTaskPtr task);

        bool IsInitialized() const;
        bool IsValid() const;
        size_t GetTasksCount() const;

        TTasks::iterator TasksBegin() {
            return Tasks.begin();
        }
        TTasks::iterator TasksEnd() {
            return Tasks.end();
        }
        TTasks::const_iterator TasksBegin() const {
            return Tasks.begin();
        }
        TTasks::const_iterator TasksEnd() const {
            return Tasks.end();
        }

    private:
        TTasks Tasks;

        TTasks Active;
        TTasks::const_iterator Next;
        TTaskPtr Last;
    };

    class TFirstTask: public TMultiTask {
    private:
        using TBase = TMultiTask;

    public:
        using TMultiTask::TMultiTask;

        static TString Type() {
            return "FirstTask";
        }
        virtual TString GetType() const override {
            return Type();
        }

    protected:
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;

        virtual NJson::TJsonValue DoSerialize() const override;
        virtual void DoDeserialize(const NJson::TJsonValue& value) override;

    private:
        TInstant GetNextHedge() const {
            if (Index < HedgeTable.size()) {
                return GetStartedTime() + HedgeTable[Index];
            } else if (!HedgeTable.empty()) {
                return GetStartedTime() + HedgeTable.back();
            } else {
                return TInstant::Max();
            }
        }
        TDuration GetTimeout() const {
            if (Index < TimeoutTable.size()) {
                return TimeoutTable[Index];
            } else {
                return TimeoutTable.size() > 0 ? TimeoutTable.back() : TDuration::Zero();
            }
        }

    private:
        const TVector<TDuration> HedgeTable = {
            TDuration::Seconds(20)
        };
        const TVector<TDuration> TimeoutTable = {
            TDuration::Seconds(2),
            TDuration::Seconds(4),
            TDuration::Seconds(8)
        };

        ui32 Index = 0;
    };

    class TSequentialTask: public TMultiTask {
    private:
        using TBase = TMultiTask;

    public:
        using TMultiTask::TMultiTask;

        static TString Type() {
            return "SequentialTask";
        }
        virtual TString GetType() const override {
            return Type();
        }

        void SetDelayTable(TVector<TDuration>&& value) {
            DelayTable = std::move(value);
        }

    protected:
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;

        virtual NJson::TJsonValue DoSerialize() const override;
        virtual void DoDeserialize(const NJson::TJsonValue& value) override;

        TDuration GetDelay() const {
            if (Index < DelayTable.size()) {
                return DelayTable[Index];
            }
            if (!DelayTable.empty()) {
                return DelayTable.back();
            } else {
                return TDuration::Zero();
            }
        }

    private:
        TVector<TDuration> DelayTable;

        ui32 Index = 0;
    };

    class TParallelTask : public TMultiTask {
    private:
        using TBase = TMultiTask;

    public:
        using TMultiTask::TMultiTask;

        static TString Type() {
            return "ParallelTask";
        }
        virtual TString GetType() const override {
            return Type();
        }

    protected:
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;

    private:
        ui32 FinishedTasksCount = 0;
    };

    class TNoWaitMultiTask : public TMultiTask {
    public:
        using TMultiTask::TMultiTask;

        static TString Type() {
            return "NoWaitMultiTask";
        }
        virtual TString GetType() const override {
            return Type();
        }

    protected:
        virtual bool DoOnMessage(TTelematicsConnection* connection, const NProtocol::IMessage& message) override;
        virtual bool DoOnAfterRegister(TTelematicsConnection* connection) override;
    };

    struct TCommandOptions {
        TDuration Timeout = TDuration::Seconds(60);
        ui32 Retries = 1;
    };

    TTaskPtr CreateBasicCommand(const TString& id, NVega::ECommandCode command, NProtocol::TArgument argument, const TCommandOptions& options);
    TTaskPtr CreateCommand(const TString& id, NVega::ECommandCode command, NProtocol::TArgument argument, const TCommandOptions& options);
}
