#include "sensors.h"

#include "handlers.h"
#include "server.h"

#include <util/random/random.h>

NDrive::TSensorsRefresher::TSensorsRefresher(const NDrive::TTelematicsServer* server, const TString& imei, TInstant now, TDuration interval, const TSensorsDict& sensors)
    : TGlobalScheduler::TScheduledItem<TSensorsRefresher>(server->Name(), "sensors:" + ToString(interval) + ":" + imei, now)
    , Server(server)
    , IMEI(imei)
    , Interval(interval)
    , Sensors(sensors)
{
    CHECK_WITH_LOG(Server);
    if (Sensors.empty()) {
        Sensors = Server->GetConfig().GetSensors();
    }
}

void NDrive::TSensorsRefresher::Process(void* /*threadSpecificResource*/) {
    THolder<TSensorsRefresher> cleanup(this);
    Y_ASSERT(Server);
    auto connection = Server->GetConnection(IMEI);
    auto now = Now();
    if (connection && connection->Alive()) {
        for (auto&& [alias, sensor] : Sensors) {
            TString taskId = ToString(now.Seconds()) + '-' + IMEI + '-' + alias + "-refresh";
            connection->AddHandler(MakeIntrusive<TGetParameterTask>(taskId, sensor.Id, sensor.SubId));
        }
    }
}

THolder<TGlobalScheduler::IScheduledItem> NDrive::TSensorsRefresher::GetNextScheduledItem(TInstant now) const {
    Y_ASSERT(Server);
    auto connection = Server->GetConnection(IMEI);
    if (connection && connection->Alive()) {
        auto randomPeriod = Max<TDuration>(Interval / 100, TDuration::Seconds(1));
        auto sampledPeriod = Interval + TDuration::Seconds(RandomNumber<ui32>(randomPeriod.Seconds() + 1));
        return MakeHolder<TSensorsRefresher>(Server, IMEI, now + sampledPeriod, Interval, Sensors);
    } else {
        return nullptr;
    }
}
