#include "server.h"

#include "beacons_refresher.h"
#include "ble_session_key_watcher.h"
#include "can_scanner.h"
#include "cdg.h"
#include "client.h"
#include "fast_data.h"
#include "handlers.h"
#include "log_loader.h"
#include "logging.h"
#include "pin_resetter.h"
#include "pinger.h"
#include "sensors.h"
#include "sim_switcher.h"
#include "tasks.h"

#include <drive/telematics/server/common/settings.h>
#include <drive/telematics/server/common/signals.h>
#include <drive/telematics/server/location/beacon_recognizer.h>
#include <drive/telematics/server/location/locator.h>
#include <drive/telematics/server/metadata/client.h>
#include <drive/telematics/server/metadata/local.h>
#include <drive/telematics/server/pusher/local.h>
#include <drive/telematics/server/pusher/pusher.h>
#include <drive/telematics/server/sensors/calculator.h>

#include <drive/telematics/api/client.h>

#include <drive/library/cpp/logger/formatter.h>

#include <kernel/daemon/messages.h>

#include <library/cpp/logger/global/common.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <logbroker/unified_agent/client/cpp/logger/backend.h>

#include <rtline/library/scheduler/global.h>
#include <rtline/library/unistat/log/backend.h>
#include <rtline/util/algorithm/type_traits.h>

#include <util/generic/iterator_range.h>
#include <util/string/builder.h>
#include <util/system/hostname.h>

namespace {
    class TTvmLogger : public NTvmAuth::ILogger {
        void Log(int lvl, const TString& msg) override {
            TEMPLATE_LOG(static_cast<ELogPriority>(lvl)) << msg << Endl;
        }
    };

    THolder<TTaskExecutor> CreateTaskExecutor(const TTaskExecutorConfig& config, IDistributedTaskContext* context) {
        if (config.IsValid()) {
            CHECK_WITH_LOG(context);
            return MakeHolder<TTaskExecutor>(config, context);
        } else {
            WARNING_LOG << "TaskExecutor config is invalid" << Endl;
            return nullptr;
        }
    }

    THolder<NDrive::TLocator> CreateLocator(const NDrive::TLocatorOptions& options, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm) {
        NDrive::TLocator::TOptions opts;
        opts.LBSToken = options.LBSToken;
        opts.LinkerHost = options.LinkerHost;
        opts.SensorHost = options.SensorHost;
        opts.SensorPort = options.SensorPort ? options.SensorPort : opts.SensorPort;
        opts.SensorSpMetaSearch = options.SensorSpMetaSearch;
        opts.SensorSaasTvmId = options.SensorSaasTvmId;
        opts.SensorService = options.SensorService ? options.SensorService : opts.SensorService;
        opts.SensorAPIName = options.SensorAPIName;
        opts.TracksHost = options.TracksHost;
        opts.GeocoderHost = options.GeocoderHost;
        opts.GeocoderPort = options.GeocoderPort;
        opts.GeocoderPath = options.GeocoderPath;
        opts.GeocoderClientId = options.GeocoderClientId;
        opts.SpeedThreshold = options.SpeedThreshold;
        opts.EnableClusterization = options.EnableClusterization;
        opts.RestrictedAreas = options.RestrictedAreas;
        opts.LinkDeviationLengthThreshold = options.LinkDeviationLengthThreshold;
        opts.LinkEnableProjection = options.LinkEnableProjection;
        opts.LinkFilterByEngine = options.LinkFilterByEngine;
        opts.LinkFilterBySpeed = options.LinkFilterBySpeed;
        opts.LinkFilterByRandomFraction = options.LinkFilterByRandomFraction;
        opts.LinkTailLength = options.LinkTailLength;
        return MakeHolder<NDrive::TLocator>(opts, tvm);
    }

    THolder<NDrive::ITelematicsMetadataClient> CreateMetadataClient(const NRTLine::TNehSearchClient* client) {
        if (client) {
            return MakeHolder<NDrive::TTelematicsMetadataClient>(*client);
        } else {
            return MakeHolder<TLocalMetadataClient>();
        }
    }

    THolder<NDrive::IPusher> CreatePusher(const NDrive::IPusherOptions& options, const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) {
        return options.BuildPusher(tvm);
    }

    THolder<NTvmAuth::TTvmClient> CreateTvmClient(const NDrive::TTelematicsConfig& options) {
        if (!options.GetTvmOptions().SelfClientId) {
            return nullptr;
        }

        NTvmAuth::NTvmApi::TClientSettings settings;
        settings.SetSelfTvmId(options.GetTvmOptions().SelfClientId);
        if (const auto& cache = options.GetTvmOptions().Cache) {
            settings.SetDiskCacheDir(cache);
        }
        NTvmAuth::NTvmApi::TClientSettings::TDstVector destinations;
        for (const auto& destinationId : options.GetTvmOptions().DestinationClientIds) {
            destinations.emplace_back(destinationId);
        }
        if (!destinations.empty()) {
            settings.EnableServiceTicketsFetchOptions(options.GetTvmOptions().Token, std::move(destinations));
        }
        settings.EnableServiceTicketChecking();
        return MakeHolder<NTvmAuth::TTvmClient>(settings, MakeIntrusive<TTvmLogger>());
    }

    NUtil::TTcpServer::TOptions GetTcpServerOptions(const THttpServerOptions& options) {
        NUtil::TTcpServer::TOptions result;
        result.ListenBacklog = options.ListenBacklog;
        result.Threads = options.nThreads;
        result.ReadTimeout = options.ClientTimeout ? options.ClientTimeout : result.ReadTimeout;
        result.WriteTimeout = options.ClientTimeout ? options.ClientTimeout : result.WriteTimeout;
        return result;
    }

    class TTelematicsServerContext: public NDrive::ITelematicsServerContext {
    public:
        TTelematicsServerContext(NDrive::TTelematicsServer& server)
            : Server(server)
        {
        }

        NDrive::TTelematicsServer& GetServer() const override {
            return Server;
        }

    private:
        NDrive::TTelematicsServer& Server;
    };

    class TTelematicsQueueViewSelector: public IQueueViewSelector {
    public:
        TTelematicsQueueViewSelector(IDistributedTaskContext& context, const NDrive::TTelematicsServer& server)
            : IQueueViewSelector(&context)
            , Server(server)
        {
        }

        bool IsAvailable(const TString& taskId) const override {
            TStringBuf task = taskId;
            TStringBuf imei = task.Before('-');
            auto connection = Server.GetConnection(TString{imei});
            if (connection && connection->Alive()) {
                return true;
            } else {
                DEBUG_LOG << "Skip task " << taskId << Endl;
                return false;
            }
        }

    private:
        const NDrive::TTelematicsServer& Server;
    };
}

namespace NDrive {
    class TTelematicsServerCallback: public NUtil::TTcpServer::ICallback {
    public:
        TTelematicsServerCallback(NDrive::TTelematicsServer* server)
            : Server(server)
        {
            CHECK_WITH_LOG(Server);
        }

        void OnConnection(NUtil::TTcpServer::TConnectionPtr connection) override {
            auto c = TTelematicsConnection::Create(Server, connection);
        }

    private:
        NDrive::TTelematicsServer* Server;
    };

    class TTelematicsServer::TTaskGarbageCollector: public TGlobalScheduler::TScheduledItem<TTaskGarbageCollector> {
    private:
        using TBase = TGlobalScheduler::TScheduledItem<TTaskGarbageCollector>;

    public:
        TTaskGarbageCollector(TTelematicsServer& server, TInstant now, TDuration interval)
            : TBase(server.Name(), server.Name() + ":task_garbage_collector", now + interval)
            , Server(server)
            , Interval(interval)
        {
        }

        void Process(void* /*threadSpecificResource*/) override {
            auto cleanup = Hold(this);
            INFO_LOG << GetId() << ": started" << Endl;
            auto now = Now();
            auto successLifetime = Server.GetConfig().GetSuccessfulTaskLifetime();
            auto failureLifetime = Server.GetConfig().GetFailedTaskLifetime();
            auto until = now - std::min(successLifetime, failureLifetime);
            auto candidates = Server.GetTasks(until);
            INFO_LOG << GetId() << ": " << candidates.size() << " discovered" << Endl;
            for (auto&& task : candidates) {
                if (!task) {
                    continue;
                }
                if (!task->IsFinished()) {
                    continue;
                }
                auto deadline = task->GetCreatedTime() + (task->GetStatus() == NDrive::TCommonTask::EStatus::Success
                    ? successLifetime
                    : failureLifetime
                );
                if (deadline < now) {
                    INFO_LOG << GetId() << ": collecting " << task->GetId() << Endl;
                    auto collected = Server.ForgetTask(task->GetId());
                    if (!collected) {
                        WARNING_LOG << GetId() << ": could not collect " << task->GetId() << Endl;
                    } else {
                        Y_ASSERT(collected->GetId() == task->GetId());
                    }
                    continue;
                }
            }
            INFO_LOG << GetId() << ": finished" << Endl;
        }
        THolder<IScheduledItem> GetNextScheduledItem(TInstant now) const override {
            return MakeHolder<TTaskGarbageCollector>(Server, now, Interval);
        }

    private:
        TTelematicsServer& Server;
        TDuration Interval;
    };
}

NDrive::TTelematicsServer::TTelematicsServer(const TTelematicsConfig& config)
    : TSearchServerBase(config.GetClientServerOptions())
    , Config(config)
    , Created(Now())
    , TelematicsServer(
        config.GetTelematicsServerOptions().Port,
        GetTcpServerOptions(config.GetTelematicsServerOptions()),
        MakeAtomicShared<NDrive::TTelematicsServerCallback>(this)
    )
    , Tvm(CreateTvmClient(config))
    , TaskContext(MakeHolder<TTelematicsServerContext>(*this))
    , TaskExecutor(CreateTaskExecutor(config.GetTaskExecutorConfig(), TaskContext.Get()))
    , Locator(CreateLocator(config.GetLocatorOptions(), Tvm))
    , Pusher(CreatePusher(config.GetPusherOptions(), Tvm))
    , SensorCalculator(MakeHolder<TSensorCalculator>())
    , Sensors(Locator->GetSensors())
    , TelematicsMetadataClient(CreateMetadataClient(Locator->GetSensorSearchClient().Get()))
{
    if (const auto& globalLog = Config.GetDaemonConfig().GetLoggerType(); globalLog || !GlobalLogInitialized()) {
        DoInitGlobalLog(MakeHolder<TUnistatLogBackend>(
            Config.GetDaemonConfig().GetLoggerType(),
            static_cast<ELogPriority>(Config.GetDaemonConfig().GetLogLevel())
        ), MakeHolder<NDrive::TLoggerFormatter>());
    }
    if (config.GetStorageOptions().GetStorageConfig()) {
        Storage = config.GetStorageOptions().ConstructStorage();
    }
    if (const TString& eventLog = Config.GetEventLog()) {
        Config.GetDaemonConfig().StartLoggingUnfiltered<TTelematicsLog>(eventLog);
    }
    if (const TString& unifiedAgentEventLog = Config.GetUnifiedAgentEventLog()) {
        auto parameters = NUnifiedAgent::TClientParameters(unifiedAgentEventLog);
        if (TLoggerOperator<TTelematicsLog>::Usage()) {
            parameters.SetLog(TLoggerOperator<TTelematicsLog>::Log());
        }
        if (size_t grpcMaxMessageSize = Config.GetUnifiedAgentGrpcMaxMessageSize()) {
            parameters.SetGrpcMaxMessageSize(grpcMaxMessageSize);
        }
        if (TDuration grpcReconnectDelay = Config.GetUnifiedAgentGrpcReconnectDelay()) {
            parameters.SetGrpcReconnectDelay(grpcReconnectDelay);
        }
        if (TDuration grpcSendDelay = Config.GetUnifiedAgentGrpcSendDelay()) {
            parameters.SetGrpcSendDelay(grpcSendDelay);
        }
        if (size_t maxInflightBytes = Config.GetUnifiedAgentMaxInflightBytes()) {
            parameters.SetMaxInflightBytes(maxInflightBytes);
        }
        auto backend = NUnifiedAgent::MakeLogBackend(parameters);
        TLoggerOperator<TTelematicsLog>::Set(new TTelematicsLog(std::move(backend)));
    }
    if (const auto threads = Config.GetSchedulerThreads()) {
        TGlobalScheduler::SetThreadsCount(threads);
    }
    TGlobalScheduler::SetDelegateInstantTasks(true);
    if (TaskExecutor) {
        TaskExecutor->SetSelector(MakeAtomicShared<TTelematicsQueueViewSelector>(*TaskContext, *this));
    }
    for (auto&& category : Config.GetConnectionCategories()) {
        const TString& name = category.Name;
        const TString& base = "connection-" + name + "-";
        for (auto&& value : category.Values) {
            auto names = TVector<TString>({ base + value });
            auto signal = MakeHolder<TUnistatSignal<>>(names, EAggregationType::LastValue, "ammv");
            ConnectionCategories.emplace(value, std::move(signal));
        }
    }
    TelematicsDynamicSettings = MakeHolder<TTelematicsDynamicSettings>(Storage);
    Locator->SetBeaconRecognizer(MakeHolder<TDynamicBeaconRecognizer>(TelematicsDynamicSettings, config.GetKnownBeaconsSynchronizationInterval()));
    Locator->SetDynamicSettings(TelematicsDynamicSettings);
}

NDrive::TTelematicsServer::~TTelematicsServer() {
}

TString NDrive::TTelematicsServer::GetClientEndpoint() const {
    return FQDNHostName() + ":" + ToString(GetConfig().GetClientServerOptions().Port);
}

TString NDrive::TTelematicsServer::Name() const {
    return TStringBuilder() << "TelematicsServer:" << Config.GetTelematicsServerOptions().Port << ':' << Created.MicroSeconds();
}

bool NDrive::TTelematicsServer::Process(IMessage* message_) {
    if (auto message = message_->As<TMessageUpdateUnistatSignals>()) {
        TMap<TString, ui64> counts;

        auto connections = GetConnections();
        for (auto&& connection : connections) {
            if (!connection) {
                continue;
            }
            for (auto&& category : Config.GetConnectionCategories()) {
                bool complete = true;
                TString value;
                for (auto&& sensorId : category.Sensors) {
                    auto optionalSensor = connection->GetSensorsCache().Get(sensorId.Id, sensorId.SubId);
                    if (!optionalSensor) {
                        complete = false;
                        break;
                    }
                    auto fallback = TString();
                    auto sensorValue = optionalSensor->ConvertTo<TString>(fallback);
                    if (!sensorValue) {
                        complete = false;
                        break;
                    }
                    if (value) {
                        value.append('-');
                    }
                    value.append(sensorValue);
                }
                if (complete) {
                    counts[value] += 1;
                }
            }
        }

        for (auto&& [value, count] : counts) {
            auto p = ConnectionCategories.find(value);
            if (p != ConnectionCategories.end()) {
                p->second->Signal(count);
            }
        }
        return true;
    }
    if (auto message = message_->As<TSystemStatusMessage>()) {
        Cerr << "SystemStatus: " << message->GetMessage() << Endl;
        return true;
    }
    return false;
}

void NDrive::TTelematicsServer::Run() try {
    TelematicsServer.Start();
    TSearchServerBase::Start();
    if (TaskExecutor) {
        TaskExecutor->Start();
    }
    RegisterGlobalMessageProcessor(this);
    GlobalSchedulerRegistrator.emplace(Name());
    if (Config.GetRigorMortisThreshold()) {
        Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<NDrive::TCoupDeGrace>(*this, Now(), TDuration::Seconds(1))));
    }
    if (Config.GetTaskGarbageCollectionInterval()) {
        Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<TTaskGarbageCollector>(*this, Now(), Config.GetTaskGarbageCollectionInterval())));
    }
} catch (const std::exception& e) {
    auto message = FormatExc(e);
    AbortFromCorruptedIndex("cannot start: %s", message.c_str());
}

void NDrive::TTelematicsServer::Stop(ui32 rigidStopLevel, const TCgiParameters* cgiParams /*= nullptr*/) {
    Y_UNUSED(cgiParams);
    GlobalSchedulerRegistrator.reset();
    UnregisterGlobalMessageProcessor(this);
    if (TaskExecutor) {
        TaskExecutor->Stop();
    }
    if (rigidStopLevel) {
        TSearchServerBase::Stop();
    } else {
        TSearchServerBase::Shutdown();
    }
    TelematicsServer.Stop();
    {
        TReadGuard guard(ConnectionsLock);
        for (auto&& [imei, connection] : Connections) {
            if (connection) {
                INFO_LOG << imei << ": dropping connection" << Endl;
                connection->Drop();
            }
        }
    }
    size_t count = 0;
    do {
        TReadGuard guard(ConnectionsLock);
        count = Connections.size();
        if (count) {
            WARNING_LOG << "waiting for " << count << " connections" << Endl;
            Sleep(TDuration::Seconds(1));
        }
    } while (count);
}

NDrive::TTelematicsConnectionPtr NDrive::TTelematicsServer::GetConnection(const TString& imei) const {
    TReadGuard guard(ConnectionsLock);
    return GetConnectionUnsafe(imei);
}

NDrive::TTelematicsConnectionPtr NDrive::TTelematicsServer::GetConnectionUnsafe(const TString& imei) const {
    auto connections = Connections.equal_range(imei);
    TTelematicsConnection* connection = nullptr;
    for (auto&&[d, c] : MakeIteratorRange(connections)) {
        connection = c;
        if (connection && connection->Alive()) {
            break;
        } else {
            WARNING_LOG << imei << ": encountered null or dead connection " << static_cast<void*>(connection) << Endl;
        }
    }
    if (connection) {
        auto weak = MakeWeakIntrusive(connection);
        if (!weak) {
            WARNING_LOG << imei << ": WeakPtr acquisition miss" << Endl;
        }
        return weak;
    } else {
        return nullptr;
    }
}

NDrive::TTelematicsServer::TConnectionPtrs NDrive::TTelematicsServer::GetConnections() const {
    TReadGuard guard(ConnectionsLock);
    TConnectionPtrs result;
    result.reserve(Connections.size());
    for (auto&& connection : Connections) {
        auto weak = MakeWeakIntrusive(connection.second);
        if (weak) {
            result.push_back(weak);
        }
    }
    return result;
}

NDrive::TTelematicsServer::TConnectionFuture NDrive::TTelematicsServer::WaitConnection(const TString& imei) const {
    {
        auto existing = GetConnection(imei);
        if (existing) {
            return NThreading::MakeFuture(std::move(existing));
        }
    }
    TWriteGuard g(ConnectionsLock);
    {
        auto existing = GetConnectionUnsafe(imei);
        if (existing) {
            return NThreading::MakeFuture(std::move(existing));
        }
    }
    return ConnectionWaiters.Get(imei);
}

void NDrive::TTelematicsServer::Register(TTelematicsConnection* connection) {
    NDrive::TTelematicsConnectionPtr expelled; // make sure that an expelled connection is destroyed not under WriteGuard
    CHECK_WITH_LOG(connection);
    auto imei = connection->GetIMEI();

    TWriteGuard guard(ConnectionsLock);
    auto p = Connections.find(imei);
    if (p != Connections.end()) {
        WARNING_LOG << "Expel " << imei << Endl;
        TTelematicsConnection* previous = p->second;
        CHECK_WITH_LOG(previous);
        expelled = MakeWeakIntrusive(previous);
        if (expelled) {
            expelled->Drop();
        }
        TTelematicsUnistatSignals::Get().ConnectionExpelled.Signal(1);
    }

    Connections.emplace(imei, connection);
    INFO_LOG << "Register " << imei << ": " << static_cast<const void*>(connection) << Endl;
    TTelematicsUnistatSignals::Get().ConnectionRegistered.Signal(1);
    TTelematicsUnistatSignals::Get().ConnectionTotal.Signal(Connections.size());
    NDrive::TTelematicsConnectionPtr registered = MakeWeakIntrusive(connection);

    for (auto&& [interval, sensors] : Config.GetSensorIntervals()) {
        DEBUG_LOG << "Schedule " << sensors.size() << " sensors every " << interval << Endl;
        bool scheduled = TGlobalScheduler::Schedule(MakeHolder<NDrive::TSensorsRefresher>(this, imei, Now(), interval, sensors));
        if (!scheduled) {
            ERROR_LOG << connection->DebugString() << ": cannot schedule SensorsRefresher" << Endl;
        }
    }
    if (auto interval = Config.GetCanScannerInterval()) {
        bool scheduled = TGlobalScheduler::Schedule(MakeHolder<NDrive::TCanScannerScheduler>(*this, imei, Now(), interval));
        if (!scheduled) {
            ERROR_LOG << connection->DebugString() << ": cannot schedule CanScannerScheduler" << Endl;
        }
    }
    if (auto interval = Config.GetPingerInterval()) {
        bool scheduled = TGlobalScheduler::Schedule(MakeHolder<NDrive::TPinger>(this, imei, Now(), interval));
        if (!scheduled) {
            ERROR_LOG << connection->DebugString() << ": cannot schedule Pinger" << Endl;
        }
    }
    if (auto interval = Config.GetFastDataInterval()) {
        bool scheduled = TGlobalScheduler::Schedule(MakeHolder<NDrive::TFastDataScheduler>(*this, imei, Now(), interval));
        if (!scheduled) {
            ERROR_LOG << connection->DebugString() << ": cannot schedule FastDataScheduler" << Endl;
        }
    }
    if (auto interval = Config.GetLogLoaderInterval()) {
        bool scheduled = TGlobalScheduler::Schedule(MakeHolder<NDrive::TLogLoader>(this, imei, Now(), interval));
        if (!scheduled) {
            ERROR_LOG << connection->DebugString() << ": cannot schedule LogLoader" << Endl;
        }
    }
    if (auto interval = Config.GetSimSwitcherInterval()) {
        bool scheduled = TGlobalScheduler::Schedule(MakeHolder<NDrive::TSimSwitcher>(*this, imei, Now(), interval));
        if (!scheduled) {
            ERROR_LOG << connection->DebugString() << ": cannot schedule SimSwitcher" << Endl;
        }
    }
    if (auto interval = Config.GetBeaconsRefreshInterval()) {
        bool scheduled = TGlobalScheduler::Schedule(MakeHolder<NDrive::TBeaconsRefresher>(*this, imei, Now(), interval));
        if (!scheduled) {
            ERROR_LOG << connection->DebugString() << ": cannot schedule BeaconsRefresher" << Endl;
        }
    }
    if (auto interval = Config.GetBleSessionKeyWatchInterval()) {
        bool scheduled = TGlobalScheduler::Schedule(MakeHolder<NDrive::TBleSessionKeyWatcher>(*this, imei, Now(), interval));
        if (!scheduled) {
            ERROR_LOG << connection->DebugString() << ": cannot schedule BleSessionKeyWatcher" << Endl;
        }
    }
    if (auto interval = Config.GetPinResetterInterval()) {
        try {
            bool scheduled = TGlobalScheduler::Schedule(MakeHolder<NDrive::TPinResetter>(*this, TelematicsDynamicSettings, imei, Now(), interval));
            if (!scheduled) {
                ERROR_LOG << connection->DebugString() << ": cannot schedule PinResetter" << Endl;
            }
        } catch (const std::exception& e) {
            ERROR_LOG << connection->DebugString() << ": cannot schedule PinResetter: " << FormatExc(e) << Endl;
        }
    }
    ConnectionWaiters.Set(imei, std::move(registered));
}

void NDrive::TTelematicsServer::Deregister(TTelematicsConnection* connection) {
    CHECK_WITH_LOG(connection);
    auto imei = connection->GetIMEI();

    TWriteGuard guard(ConnectionsLock);
    ui32 removed = 0;
    auto range = Connections.equal_range(imei);
    for (auto i = range.first; i != range.second;) {
        if (i->second == connection) {
            auto victim = i;
            ++i;
            ++removed;
            Connections.erase(victim);
        } else {
            ++i;
        }
    }
    CHECK_WITH_LOG(removed == 1) << imei;
    INFO_LOG << "Deregister " << imei << ": " << static_cast<const void*>(connection) << Endl;
    TTelematicsUnistatSignals::Get().ConnectionDeregistered.Signal(1);
    TTelematicsUnistatSignals::Get().ConnectionTotal.Signal(Connections.size());
}

TClientRequest* NDrive::TTelematicsServer::CreateClient() {
    return new NDrive::TTelematicsServerHttpClient(this);
}

NDrive::TTaskPtr NDrive::TTelematicsServer::AddTask(TTaskPtr task) {
    TStringBuf id = task->GetId();
    TWriteGuard guard(TasksLock);
    INFO_LOG << "Adding task " << id << Endl;
    auto p = Tasks.find(id);
    if (p != Tasks.end()) {
        NJson::TJsonValue serialized = p->second ? p->second->Serialize() : NJson::JSON_NULL;
        TTelematicsLog::Log(TTelematicsLog::Warning, nullptr, "ExpelTask", std::move(serialized));
        p->second = task;
    } else {
        Y_ENSURE(Tasks.emplace(id, task).second);
    }
    OrderedTasks[std::make_pair(task->GetCreatedTime(), id)] = task;
    TaskWaiters.Set(id, MakeCopy(task));
    return task;
}

NDrive::TTaskPtr NDrive::TTelematicsServer::GetTask(TStringBuf id) const {
    TReadGuard guard(TasksLock);
    auto p = Tasks.find(id);
    if (p != Tasks.end()) {
        auto result = p->second;
        Y_ASSERT(result);
        Y_ASSERT(result->GetId() == id);
        return result;
    } else {
        return nullptr;
    }
}

NDrive::TTaskPtr NDrive::TTelematicsServer::ForgetTask(TStringBuf id) {
    NDrive::TTaskPtr result;
    {
        TWriteGuard guard(TasksLock);
        auto p = Tasks.find(id);
        if (p != Tasks.end()) {
            result = std::move(p->second);
            Tasks.erase(p);
            OrderedTasks.erase(std::make_pair(result->GetCreatedTime(), id));
        }
    }
    Y_ASSERT(!result || result->GetId() == id);
    return result;
}

NDrive::TTelematicsServer::TTaskFuture NDrive::TTelematicsServer::WaitTask(TStringBuf id, bool ignoreTerminated) const {
    {
        auto existing = GetTask(id);
        if (existing && (!ignoreTerminated || !existing->Terminated())) {
            return NThreading::MakeFuture(std::move(existing));
        }
    }
    TWriteGuard guard(TasksLock);
    {
        auto p = Tasks.find(id);
        if (p != Tasks.end() && (!ignoreTerminated || !p->second->Terminated())) {
            return NThreading::MakeFuture(p->second);
        }
    }
    return TaskWaiters.Get(id);
}

TVector<NDrive::TTaskPtr> NDrive::TTelematicsServer::GetTasks(TInstant until) const {
    TVector<NDrive::TTaskPtr> result;
    result.reserve(Tasks.size());
    {
        TReadGuard guard(TasksLock);
        auto begin = OrderedTasks.begin();
        auto end = OrderedTasks.upper_bound(std::make_pair(until, TStringBuf{}));
        for (auto&& [orderedKey, task] : MakeIteratorRange(begin, end)) {
            if (!task) {
                continue;
            }
            result.push_back(task);
        }
    }
    return result;
}
