#pragma once

#include "config.h"
#include "connection.h"
#include "waiters.h"

#include <drive/telematics/server/pusher/interface.h>

#include <drive/library/cpp/network/tcp/server.h>
#include <drive/library/cpp/searchserver/server.h>

#include <kernel/daemon/base_controller.h>
#include <kernel/daemon/server.h>

#include <rtline/library/scheduler/registrator.h>

#include <util/system/rwlock.h>

namespace NTvmAuth {
    class TTvmClient;
}

namespace NDrive {
    class TCommonTask;
    using TTaskPtr = TIntrusivePtr<TCommonTask>;

    class IPusher;
    class ISensorApi;
    class ITelematicsMetadataClient;
    class TLocator;
    class TSensorCalculator;
    class TTelematicsDynamicSettings;

    class TTelematicsServer
        : public NController::IServer
        , public TSearchServerBase
        , public IMessageProcessor
    {
    public:
        using TConfig = TTelematicsConfig;
        using TController = NController::TController;
        using TInfoCollector = TCollectServerInfo;

        using TConnections = std::multimap<TString, TTelematicsConnection*>;
        using TConnectionPtrs = TVector<TTelematicsConnectionPtr>;
        using TConnectionFuture = NThreading::TFuture<TTelematicsConnectionPtr>;

        using TTasks = TMap<TStringBuf, TTaskPtr>;
        using TTaskFuture = NThreading::TFuture<TTaskPtr>;
        using TOrderedTasks = TMap<std::pair<TInstant, TStringBuf>, TTaskPtr>;

    public:
        TTelematicsServer(const TTelematicsConfig& config);
        ~TTelematicsServer();

        const TTelematicsConfig& GetConfig() const {
            return Config;
        }
        const TTelematicsDynamicSettings& GetDynamicSettings() const {
            return *TelematicsDynamicSettings;
        }
        TLocator& GetLocator() const {
            return *Locator;
        }
        IPusher* GetPusher() const {
            return Pusher.Get();
        }
        const ITelematicsMetadataClient* GetMetadataClient() const {
            return TelematicsMetadataClient.Get();
        }
        const ISensorApi* GetSensors() const {
            return Sensors.Get();
        }
        TSensorCalculator* GetSensorCalculator() const {
            return SensorCalculator.Get();
        }
        const NTvmAuth::TTvmClient* GetTvmClient() const {
            return Tvm.Get();
        }
        TAtomicSharedPtr<NRTLine::IVersionedStorage> GetStorage() const {
            return Storage;
        }

        void SetPusher(THolder<IPusher>&& pusher) {
            Pusher = std::move(pusher);
        }

        TString GetClientEndpoint() const;

        virtual TString Name() const override;
        virtual bool Process(IMessage* message) override;
        void Run() override;
        void Stop(ui32 rigidStopLevel, const TCgiParameters* cgiParams = nullptr) override;

        TTelematicsConnectionPtr GetConnection(const TString& imei) const;
        TConnectionPtrs GetConnections() const;
        TConnectionFuture WaitConnection(const TString& imei) const;
        void Register(TTelematicsConnection* connection);
        void Deregister(TTelematicsConnection* connection);

        template <class T, class... TArgs>
        TTaskPtr CreateTask(TArgs... args) {
            auto task = MakeIntrusive<T>(std::forward<TArgs>(args)...);
            return AddTask(task);
        }
        TTaskPtr AddTask(TTaskPtr task);
        TTaskPtr GetTask(TStringBuf id) const;
        TTaskPtr ForgetTask(TStringBuf id);
        TTaskFuture WaitTask(TStringBuf id, bool ignoreTerminated = true) const;

    private:
        class TTaskGarbageCollector;

    private:
        using TSearchServerBase::Start;
        using TSearchServerBase::Stop;
        using TSearchServerBase::Shutdown;

        virtual TClientRequest* CreateClient() override;

        TTelematicsConnectionPtr GetConnectionUnsafe(const TString& imei) const;
        TVector<TTaskPtr> GetTasks(TInstant until) const;

    private:
        const TTelematicsConfig& Config;
        const TInstant Created;

        NUtil::TTcpServer TelematicsServer;
        TAtomicSharedPtr<NTvmAuth::TTvmClient> Tvm;
        THolder<IDistributedTaskContext> TaskContext;
        THolder<TTaskExecutor> TaskExecutor;
        THolder<TLocator> Locator;
        THolder<IPusher> Pusher;
        THolder<TSensorCalculator> SensorCalculator;
        TAtomicSharedPtr<ISensorApi> Sensors;
        TAtomicSharedPtr<NRTLine::IVersionedStorage> Storage;
        TAtomicSharedPtr<TTelematicsDynamicSettings> TelematicsDynamicSettings;
        TAtomicSharedPtr<ITelematicsMetadataClient> TelematicsMetadataClient;
        NGlobalScheduler::TOptionalRegistrator GlobalSchedulerRegistrator;

        TConnections Connections;
        TRWMutex ConnectionsLock;
        TMap<TString, THolder<TUnistatSignal<>>> ConnectionCategories;
        NThreading::TWaiters<TTelematicsConnectionPtr> ConnectionWaiters;

        TTasks Tasks;
        TOrderedTasks OrderedTasks;
        TRWMutex TasksLock;
        NThreading::TWaiters<TTaskPtr> TaskWaiters;
    };
}
