#include "sim_switcher.h"

#include "handlers.h"
#include "server.h"

#include <drive/telematics/server/common/settings.h>

#include <drive/telematics/common/firmware.h>
#include <drive/telematics/protocol/actions.h>

#include <util/digest/fnv.h>

NDrive::TSimSwitcher::TSimSwitcher(const NDrive::TTelematicsServer& server, const TString& imei, TInstant now, TDuration interval)
    : TGlobalScheduler::TScheduledItem<TSimSwitcher>(server.Name(), "sim_switcher:" + imei, now + interval)
    , Server(server)
    , IMEI(imei)
    , Interval(interval)
{
}

void NDrive::TSimSwitcher::Process(void* /*threadSpecificResource*/) {
    auto cleanup = Hold(this);
    const auto& settings = Server.GetDynamicSettings();
    auto connection = Server.GetConnection(IMEI);
    if (!connection || !connection->Alive()) {
        return;
    }
    const auto& sensorsCache = connection->GetSensorsCache();
    const auto firmware = sensorsCache.Get(VEGA_MCU_FIRMWARE_VERSION);
    if (!firmware) {
        ERROR_LOG << GetId() << ": firmware sensor is missing" << Endl;
        return;
    }
    auto firmwareValue = firmware->ConvertTo<TString>(TString{});
    if (!firmwareValue) {
        WARNING_LOG << GetId() << ": bad firmware sensor value " << firmware->GetJsonValue() << Endl;
        return;
    }
    auto firmwareInfo = NDrive::NVega::ParseFirmwareInfo(firmwareValue);
    if (!firmwareInfo.IsFeatureSupported(NVega::TFirmwareInfo::SwitchSim)) {
        WARNING_LOG << GetId() << ": firmware " << firmwareValue << " does not support SwitchSim command" << Endl;
        return;
    }

    auto mnc = sensorsCache.Get(VEGA_MNC);
    if (!mnc) {
        ERROR_LOG << GetId() << ": MNC sensor is missing" << Endl;
        return;
    }
    auto mncValue = mnc->ConvertTo<ui64>();
    auto mncTarget = settings.Get<ui64>("imei:" + IMEI + ":mnc");
    if (!mncTarget) {
        auto bucket = FnvHash<ui32>(IMEI) % 100;
        auto mncList = { 2, 99 };
        ui32 border = 0;
        for (auto&& mnc : mncList) {
            auto fraction = settings.Get<ui32>("mnc:" + ToString(mnc) + ":fraction");
            if (!fraction) {
                continue;
            }
            border += *fraction;
            if (bucket < border) {
                mncTarget = mnc;
                break;
            }
        }
    }
    if (!mncTarget) {
        DEBUG_LOG << GetId() << ": target MNC is undefined" << Endl;
        return;
    }
    if (mncValue == *mncTarget) {
        DEBUG_LOG << GetId() << ": target MNC is satisfied" << Endl;
        return;
    }
    const auto now = Now();
    const auto switchTimeout = settings.Get<TDuration>("sim_switcher:switch_timeout").GetOrElse(TDuration::Minutes(30));
    if (mnc->Since + switchTimeout > now) {
        DEBUG_LOG << GetId() << ": switch timeout is not exceeded" << Endl;
        return;
    }
    auto command = NDrive::ParseCommand("switch-sim");
    auto handler = MakeIntrusive<TSendCommandTask>(ToString(now.MicroSeconds()) + '-' + GetId(), command->Code, command->Argument);
    INFO_LOG << GetId() << ": switching sim using " << handler->GetId() << Endl;
    connection->AddHandler(handler);
}

THolder<TGlobalScheduler::IScheduledItem> NDrive::TSimSwitcher::GetNextScheduledItem(TInstant now) const {
    auto connection = Server.GetConnection(IMEI);
    if (!connection || !connection->Alive()) {
        return nullptr;
    }
    return MakeHolder<TSimSwitcher>(Server, IMEI, now, Interval);
}
