#include "tasks.h"

#include "handlers.h"
#include "logging.h"
#include "scenarios.h"
#include "server.h"

#include <drive/telematics/common/file.h>
#include <drive/telematics/protocol/log.h>

#include <rtline/library/json/exception.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/system/hostname.h>

namespace {
    void SetStatus(NDrive::TCommonDistributedData& data, const NDrive::TCommonTask& task) {
        switch (task.GetStatus()) {
        case NDrive::TCommonTask::EStatus::Success:
            data.SetStatus(NDrive::NProto::TTelematicsTask::SUCCESS, task.GetLastInfo());
            break;
        case NDrive::TCommonTask::EStatus::Failure:
        case NDrive::TCommonTask::EStatus::Retry:
            data.SetStatus(NDrive::NProto::TTelematicsTask::FAILURE, task.GetLastInfo());
            break;
        case NDrive::TCommonTask::EStatus::Terminated:
            data.SetStatus(NDrive::NProto::TTelematicsTask::TERMINATED, "connection lost");
            break;
        case NDrive::TCommonTask::EStatus::Timeouted:
            data.SetStatus(NDrive::NProto::TTelematicsTask::TIMEOUTED, "timeout exceeded");
            break;
        default:
            data.SetStatus(NDrive::NProto::TTelematicsTask::INTERNAL_ERROR, "unknown handler status: " + ToString(task.GetStatus()));
            break;
        }
        data.SetFinished(Now());
    }
}

bool NDrive::TBaseDistributedTask::DoExecute(IDistributedTask::TPtr self) noexcept {
    INFO_LOG << "Processing " << GetType() << " " << self->GetIdentifier() << Endl;
    const auto& context = Executor->GetContextAs<ITelematicsServerContext>();
    const auto d = GetData();
    if (!d) {
        ERROR_LOG << "Cannot acquire data for " << GetType() << " " << self->GetIdentifier() << Endl;
        return true;
    }

    auto& data = d->GetDataAs<TCommonDistributedData>();
    switch (data.GetStatus()) {
    case NDrive::NProto::TTelematicsTask::NEW:
    case NDrive::NProto::TTelematicsTask::PROCESSING:
        if (data.GetMetaInfo().GetDeadline() > Now()) {
            break;
        } else {
            data.SetStatus(NDrive::NProto::TTelematicsTask::TIMEOUTED, "initialization timeout");
            if (!Executor->StoreData2(&data)) {
                ERROR_LOG << "cannot store data in storage" << Endl;
                return true;
            }
            // fallthrough
        }
    default:
        TTelematicsLog::Log(TTelematicsLog::Dropped, nullptr, data);
        WARNING_LOG << "Drop " << GetType() << " " << self->GetIdentifier() << ": status " << NDrive::NProto::TTelematicsTask::EStatus_Name(data.GetStatus()) << Endl;
        return true;
    }

    const TString& imei = data.GetMetaInfo().GetIMEI();
    const auto connection = context.GetServer().GetConnection(imei);
    if (!connection || !connection->Alive()) {
        Executor->RescheduleTask(self.Get(), Now(), /*unlockData=*/true);
        WARNING_LOG << "Rescheduling " << GetType() << " " << self->GetIdentifier() << ": connection to " << imei << " lost" << Endl;
        return true;
    }

    TTelematicsLog::Log(TTelematicsLog::Started, connection.Get(), data);
    try {
        return DoExecuteImpl(self, d, connection);
    } catch (const std::exception& e) {
        auto error = CurrentExceptionInfo(true).GetStringRobust();
        data.SetStatus(NDrive::NProto::TTelematicsTask::INTERNAL_ERROR, error);
        ERROR_LOG << "Exception in " << GetType() << " " << self->GetIdentifier() << ": " << error << Endl;
        if (!Executor->StoreData2(&data)) {
            ERROR_LOG << "cannot store data in storage" << Endl;
        }
        return true;
    }
}

bool NDrive::TPingDistributedTask::DoExecuteImpl(IDistributedTask::TPtr self, IDDataStorage::TGuard::TPtr d, TTelematicsConnectionPtr connection) {
    CHECK_WITH_LOG(connection);
    CHECK_WITH_LOG(d);
    const auto& context = Executor->GetContextAs<ITelematicsServerContext>();
    const auto& data = d->GetDataAs<TPingDistributedData>();
    const auto& id = data.GetIdentifier();
    auto handler = context.GetServer().CreateTask<TPingTask>(id);
    handler->AddCallback([self, this] (const TCommonTask& cmd) {
            IDistributedTask::TRemoveGuard guard(self);
            auto& data = GetData<TPingDistributedData>();
            data.OnHeartbeat();
            SetStatus(data, cmd);
            Executor->StoreData2(&data);
            TTelematicsLog::Log(
                TTelematicsLog::Finished,
                Executor->GetContextAs<ITelematicsServerContext>().GetServer().GetConnection(data.GetMetaInfo().GetIMEI()).Get(),
                data
            );
            INFO_LOG << "Processed PingDistributedTask " << self->GetIdentifier() << Endl;
    });
    connection->AddHandler(handler);

    return false;
}

bool NDrive::TSendCommandDistributedTask::DoExecuteImpl(IDistributedTask::TPtr self, IDDataStorage::TGuard::TPtr d, TTelematicsConnectionPtr connection) {
    CHECK_WITH_LOG(connection);
    CHECK_WITH_LOG(d);
    auto& server = Executor->GetContextAs<ITelematicsServerContext>().GetServer();
    auto& data = d->GetDataAs<TSendCommandDistributedData>();

    const auto command = data.GetCommand();
    const auto id =
        data.GetIdentifier() + '-' +
        FQDNHostName() + ':' + ToString(server.GetConfig().GetTelematicsServerOptions().Port);
    NDrive::NProtocol::TArgument argument;
    try {
        Y_ENSURE(GetEnumNames<NDrive::NVega::ECommandCode>().contains(command), "unknown command code " << static_cast<i64>(command));
        if (auto arg = data.GetGenericArgument()) {
            argument = *arg;
        } else switch (command) {
        case NDrive::NVega::ECommandCode::GET_PARAM: {
            auto arg = Yensured(data.GetParameter());
            NDrive::NVega::TCommandRequest::TGetParameter parameter;
            parameter.Id = arg->Id;
            parameter.SubId = arg->SubId;
            argument.Set(parameter);
            break;
        }
        case NDrive::NVega::ECommandCode::SCENARIO_POLITE_SET_PARAM:
        case NDrive::NVega::ECommandCode::SET_PARAM: {
            auto arg = Yensured(data.GetParameter());
            argument.Set(*arg);
            break;
        }
        case NDrive::NVega::ECommandCode::MOVE_TO_COORD: {
            const NDrive::NVega::TCommandRequest::TMoveToCoordParameter* parameter = data.MoveToCoordParameter();
            Y_ENSURE(parameter, "incorrect task: no MoveToCoordParameter");

            argument.Set(*parameter);
            break;
        }
        case NDrive::NVega::ECommandCode::ELECTRIC_CAR_COMMAND:
        {
            auto arg = Yensured(data.GetElectricCarCommandArgument());
            argument.Set(*arg);
            break;
        }
        case NDrive::NVega::ECommandCode::YADRIVE_PANIC:
        {
            auto arg = Yensured(data.GetPanicArgument());
            argument.Set(*arg);
            break;
        }
        case NDrive::NVega::ECommandCode::YADRIVE_WARMING: {
            auto arg = data.GetWarmingArgument();
            if (arg) {
                argument.Set(*arg);
            }
            break;
        }
        default:
            break;
        }
    } catch (const std::exception& e) {
        data.SetStatus(NProto::TTelematicsTask::INCORRECT, FormatExc(e));
        data.SetFinished(Now());
        Executor->StoreData2(&data);
        return true;
    }

    auto callback = [self, this](const TCommonTask& cmd) {
        IDistributedTask::TRemoveGuard guard(self);
        auto& server = Executor->GetContextAs<ITelematicsServerContext>().GetServer();
        auto& data = GetData<TSendCommandDistributedData>();
        data.AddHandlerId(cmd.GetId());
        data.OnHeartbeat(server.GetClientEndpoint());

        switch (cmd.GetStatus()) {
        case TCommonTask::EStatus::Success: {
            switch (data.GetCommand()) {
            case NDrive::NVega::ECommandCode::GET_PARAM: {
                data.SetId(cmd.GetSensorId().Id);
                data.SetSubId(cmd.GetSensorId().SubId);
                data.SetValue(cmd.GetSensorValue());
                break;
            }
            default:
                break;
            }
            break;
        }
        default:
            break;
        }
        SetStatus(data, cmd);
        Executor->StoreData2(&data);
        TTelematicsLog::Log(
            TTelematicsLog::Finished,
            server.GetConnection(data.GetMetaInfo().GetIMEI()).Get(),
            data
        );
        INFO_LOG << "Processed SendCommandDistributedTask " << self->GetIdentifier() << Endl;
    };

    TCommandOptions options;
    options.Retries = data.GetRetriesCount();

    try {
        auto handler = NDrive::CreateCommand(id, command, std::move(argument), options);
        auto now = Now();
        handler->AddCallback(std::move(callback), now + data.GetPostlinger());
        server.AddTask(handler);
        connection->AddHandler(handler, now + data.GetPrelinger());
    } catch (const std::exception& e) {
        data.SetStatus(NProto::TTelematicsTask::INCORRECT, FormatExc(e));
        data.SetFinished(Now());
        Executor->StoreData2(&data);
        return true;
    }

    return false;
}

bool NDrive::TDownloadFileDistributedTask::DoExecuteImpl(IDistributedTask::TPtr self, IDDataStorage::TGuard::TPtr d, TTelematicsConnectionPtr connection) {
    auto& data = Checked(d)->GetDataAs<TDownloadFileDistributedData>();

    auto handler = MakeAtomicShared<NDrive::NVega::TGetFileHandler>(
        data.GetName(),
        /*offset=*/0,
        std::min<ui16>(Checked(connection)->GetInputBufferSize() - 24, Max<ui16>())
    );
    auto callback = [d, handler, self, this](NProtocol::THandlerPtr h) {
        CHECK_WITH_LOG(h == handler);
        IDistributedTask::TRemoveGuard guard(self);
        auto& server = Executor->GetContextAs<ITelematicsServerContext>().GetServer();
        auto& data = d->GetDataAs<TDownloadFileDistributedData>();
        data.OnHeartbeat(server.GetClientEndpoint());

        NDrive::NVega::TGetFileResponse::EResult result = handler->GetResult();
        switch (result) {
        case NDrive::NVega::TGetFileResponse::OK:
            if (auto connection = server.GetConnection(data.GetMetaInfo().GetIMEI())) {
                const TBuffer& buffer = handler->GetData();
                const TStringBuf str(buffer.data(), buffer.size());
                connection->AddLogRecords(NDrive::NVega::ParseLogRecords(str));
            }
            data.SetData(std::move(handler->GetData()));
            data.SetStatus(NDrive::NProto::TTelematicsTask::SUCCESS);
            break;
        case NDrive::NVega::TGetFileResponse::UNKNOWN_ERROR:
            data.SetStatus(NDrive::NProto::TTelematicsTask::INTERNAL_ERROR, "unknown server error");
            break;
        default:
            data.SetStatus(NDrive::NProto::TTelematicsTask::FAILURE, ToString(result));
            break;
        }

        if (!d->Store2()) {
            ERROR_LOG << "cannot store data for " << self->GetIdentifier() << Endl;
        }
        INFO_LOG << "Processed " << self->GetType() << " " << self->GetIdentifier() << Endl;
    };

    auto wrapper = MakeAtomicShared<NProtocol::TCallbackHandler>(handler, std::move(callback));
    connection->AddMessageHandler(wrapper);

    return false;
}

bool NDrive::TUploadFileDistributedTask::DoExecuteImpl(IDistributedTask::TPtr self, IDDataStorage::TGuard::TPtr d, TTelematicsConnectionPtr connection) {
    CHECK_WITH_LOG(connection);
    CHECK_WITH_LOG(d);
    auto& data = d->GetDataAs<TUploadFileDistributedData>();

    auto handler = MakeAtomicShared<NDrive::NVega::TChunkedFileHandler>(
        data.GetName(),
        data.ReleaseData(),
        std::min<ui16>(Yensured(connection)->GetInputBufferSize() - 24, Max<ui16>())
    );
    auto callback = [handler, self, d, this] (NProtocol::THandlerPtr) {
        IDistributedTask::TRemoveGuard guard(self);
        auto& server = Executor->GetContextAs<ITelematicsServerContext>().GetServer();
        auto& data = d->GetDataAs<TUploadFileDistributedData>();
        data.OnHeartbeat(server.GetClientEndpoint());

        NDrive::NVega::TFileChunkResponse::EResult result = handler->GetResult();
        switch (result) {
        case NDrive::NVega::TFileChunkResponse::OK:
            data.SetStatus(NDrive::NProto::TTelematicsTask::SUCCESS);
            break;
        case NDrive::NVega::TFileChunkResponse::UNKNOWN_ERROR:
            data.SetStatus(NDrive::NProto::TTelematicsTask::INTERNAL_ERROR, "unknown server error");
            break;
        default:
            data.SetStatus(NDrive::NProto::TTelematicsTask::FAILURE, ToString(result));
            break;
        }

        if (!d->Store2()) {
            ERROR_LOG << "cannot store data for " << self->GetIdentifier() << Endl;
        }
        INFO_LOG << "Processed " << self->GetType() << " " << self->GetIdentifier() << Endl;
    };

    auto wrapper = MakeAtomicShared<NProtocol::TCallbackHandler>(handler, std::move(callback));
    connection->AddMessageHandler(wrapper);

    return false;
}

bool NDrive::TInterfaceCommunicationDistributedTask::DoExecuteImpl(IDistributedTask::TPtr self, IDDataStorage::TGuard::TPtr d, TTelematicsConnectionPtr connection) {
    auto& data = Checked(d)->GetDataAs<TInterfaceCommunicationDistributedData>();

    auto input = data.GetInput();
    auto handler = MakeIntrusive<TInterfaceTask>(data.GetIdentifier(), data.GetInterface(), std::move(input));
    handler->AddCallback([hndl = handler.Get(), self, d, this](const TCommonTask& cmd) {
        IDistributedTask::TRemoveGuard guard(self);
        auto& server = Executor->GetContextAs<ITelematicsServerContext>().GetServer();
        auto& data = Checked(d)->GetDataAs<TInterfaceCommunicationDistributedData>();
        data.OnHeartbeat(server.GetClientEndpoint());
        switch (cmd.GetStatus()) {
        case NDrive::TCommonTask::EStatus::Success:
            data.SetOutput(hndl->ReleaseOutput());
            [[fallthrough]];
        default:
            SetStatus(data, cmd);
            break;
        }
        if (!d->Store2()) {
            ERROR_LOG << "cannot store data for " << cmd.GetId() << Endl;
        }
    });

    Checked(connection)->AddHandler(handler);
    return false;
}

bool NDrive::TCanRequestDistributedTask::DoExecuteImpl(IDistributedTask::TPtr self, IDDataStorage::TGuard::TPtr d, TTelematicsConnectionPtr connection) {
    auto& data = Checked(d)->GetDataAs<TCanRequestDistributedData>();

    auto canData = MakeArrayRef(
        reinterpret_cast<const ui8*>(data.GetData().Begin()),
        reinterpret_cast<const ui8*>(data.GetData().End())
    );
    auto handler = MakeIntrusive<TCanRequestTask>(data.GetIdentifier(), data.GetCanId(), data.GetCanIndex(), canData);
    handler->AddCallback([hndl = handler.Get(), self, d, this](const TCommonTask& cmd) {
        IDistributedTask::TRemoveGuard guard(self);
        auto& server = Executor->GetContextAs<ITelematicsServerContext>().GetServer();
        auto& data = Checked(d)->GetDataAs<TCanRequestDistributedData>();
        data.OnHeartbeat(server.GetClientEndpoint());
        SetStatus(data, cmd);
        if (!d->Store2()) {
            ERROR_LOG << "cannot store data for " << cmd.GetId() << Endl;
        }
    });

    auto now = Now();
    Checked(connection)->AddHandler(handler);

    auto duration = data.GetDuration();
    auto interval = data.GetInterval();
    auto since = now + interval;
    auto until = (duration && interval) ? since + duration : TInstant::Zero();
    for (auto timestamp = since; timestamp <= until; timestamp += interval) {
        auto echo = MakeIntrusive<TCanRequestTask>(data.GetIdentifier(), data.GetCanId(), data.GetCanIndex(), canData);
        connection->AddHandler(echo, timestamp);
    }
    return false;
}

IDistributedTask::TFactory::TRegistrator<NDrive::TPingDistributedTask> PingTask("Ping");
IDistributedTask::TFactory::TRegistrator<NDrive::TSendCommandDistributedTask> SendCommandTask("SendCommand");
IDistributedTask::TFactory::TRegistrator<NDrive::TDownloadFileDistributedTask> DownloadFileTask("DownloadFile");
IDistributedTask::TFactory::TRegistrator<NDrive::TUploadFileDistributedTask> UploadFileTask("UploadFile");
IDistributedTask::TFactory::TRegistrator<NDrive::TInterfaceCommunicationDistributedTask> InterfaceCommunicationTask("Interface");
IDistributedTask::TFactory::TRegistrator<NDrive::TCanRequestDistributedTask> CanRequestTask("CanRequest");
