#include "wialon.h"

#include "scenarios.h"
#include "server.h"

#include <drive/telematics/server/common/settings.h>

#include <drive/telematics/protocol/wialon.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/logger/global/global.h>

#include <rtline/library/json/cast.h>

#include <util/generic/ptr.h>
#include <util/generic/yexception.h>

NDrive::NWialon::TWialonProtocol::TWialonProtocol(TTelematicsServerPtr server /*= nullptr*/)
    : Server(server)
{
    Type = NProtocol::PT_WIALON_IPS;
}

void NDrive::NWialon::TWialonProtocol::Launch() noexcept {
}

THolder<NDrive::NProtocol::IMessage> NDrive::NWialon::TWialonProtocol::Load(IInputStream& stream) {
    auto result = MakeHolder<NWialon::TMessage>();
    result->Load(stream);
    return std::move(result);
}

void NDrive::NWialon::TWialonProtocol::Process(const NProtocol::IMessage& message) {
    Y_ENSURE(message.GetProtocolType() == NProtocol::PT_WIALON_IPS);

    if (message.GetMessageType() == NWialon::MT_LOGIN_REQUEST) {
        Process(message.As<NWialon::TLoginRequest>());
    } else if (message.GetMessageType() == NWialon::MT_PING_REQUEST) {
        Process(message.As<NWialon::TPingRequest>());
    } else if (message.GetMessageType() == NWialon::MT_SHORT_DATA_REQUEST) {
        Process(message.As<NWialon::TShortDataRequest>());
    } else if (message.GetMessageType() == NWialon::MT_DATA_REQUEST) {
        Process(message.As<NWialon::TDataRequest>());
    } else if (message.GetMessageType() == NWialon::MT_BLACK_BOX_REQUEST) {
        Process(message.As<NWialon::TBlackBoxRequest>());
    }
}

void NDrive::NWialon::TWialonProtocol::Process(const NWialon::TLoginRequest& message) {
    auto answer= NProtocol::IPayload::Create<NWialon::TLoginAnswer>();
    answer->Answer = NWialon::TLoginAnswer::Success;

    SetIMEI(message.IMEI.Get());

    Send<NWialon::TMessage>(std::move(answer));
}

void NDrive::NWialon::TWialonProtocol::Process(const NWialon::TPingRequest& /*message*/) {
    auto answer = NProtocol::IPayload::Create<NWialon::TPingAnswer>();
    Send<NWialon::TMessage>(std::move(answer));
}

void NDrive::NWialon::TWialonProtocol::Process(const NWialon::TShortDataRequest& /*message*/) {
    auto answer = NProtocol::IPayload::Create<NWialon::TShortDataAnswer>();
    answer->Answer = NWialon::TShortDataAnswer::EAnswer::Success;

    Send<NWialon::TMessage>(std::move(answer));
}

void NDrive::NWialon::TWialonProtocol::Process(const NWialon::TDataRequest& /*message*/) {
    const auto onSearch = TBuffer("TP", 2);
    const auto offSearch = TBuffer("TS", 2);

    auto answer = NProtocol::IPayload::Create<NWialon::TSettingFileRequest>();

    if (EnableSearch()) {
        answer->SetData(onSearch);
    } else {
        answer->SetData(offSearch);
    }

    Send<NWialon::TMessage>(std::move(answer));
}

void NDrive::NWialon::TWialonProtocol::Process(const NWialon::TBlackBoxRequest& message) {
    auto answer = NProtocol::IPayload::Create<NWialon::TBlackBoxAnswer>();
    answer->Answer = message.Data.Get().size();

    Send<NWialon::TMessage>(std::move(answer));
}

void NDrive::NWialon::TWialonProtocol::Drop() noexcept {
}

void NDrive::NWialon::TWialonProtocol::SetIMEI(const TString& imei) {
    auto guard = Guard(Mutex);
    IMEI = imei;
}
const TString& NDrive::NWialon::TWialonProtocol::GetIMEI() const {
    auto guard = Guard(Mutex);
    return IMEI;
}

bool NDrive::NWialon::TWialonProtocol::EnableSearch() const {
    if (!Server) {
        return false;
    }

    auto storage = Server->GetStorage();
    if (!storage) {
        ERROR_LOG << "storage is missing" << Endl;
        return false;
    }

    if (EnableSearchFromList()) {
        NOTICE_LOG << "beacon imei contains in list for search" << Endl;
        return true;
    }

    TString value;
    storage->GetValue("imei:" + IMEI + ":enable_beacon_search", value);
    bool result = IsTrue(value);

    INFO_LOG << IMEI << ": beacon search value: " << value << " " << IsTrue(value) << Endl;

    return result;
}

bool NDrive::NWialon::TWialonProtocol::EnableSearchFromList() const {
    const auto& settings = Server->GetDynamicSettings();
    auto maybeNeedSearchIMEI = settings.Get<TString>(IMEISearchName);

    if (!maybeNeedSearchIMEI) {
        ERROR_LOG << "cannot find settings from " << IMEISearchName;
        return false;
    }

    TSet<TString> imeiList;
    NJson::TJsonValue value;
    if (!NJson::ReadJsonFastTree(*maybeNeedSearchIMEI, &value) || !NJson::TryFromJson(value, imeiList)) {
        ALERT_LOG << "cannot parse telematics setting: " << IMEISearchName << Endl;
    }

    if (!imeiList.contains(GetIMEI())) {
        ERROR_LOG << "imei not contains in list " << GetIMEI() << Endl;
        return false;
    }

    return true;
}

NDrive::NProtocol::TTaskPtr NDrive::NWialon::TWialonProtocol::CreateCommand(const TString& id, NProtocol::ECommandCode command, NProtocol::TArgument argument, const TCommandOptions& options) {
    Y_UNUSED(id);
    Y_UNUSED(command);
    Y_UNUSED(argument);
    Y_UNUSED(options);
    return {};
}
