#include "beacon_recognizer.h"

#include <library/cpp/string_utils/base64/base64.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/types/uuid.h>

#include <util/string/builder.h>

TString NDrive::IBeaconRecognizer::ConvertToKey(const NDrive::NVega::TBeaconInfo& beaconInfo) {
    TStringBuilder builder;
    builder << "uuid:" << beaconInfo.GetUUID().AsUuidString() << ":";
    builder << "major:" << beaconInfo.GetMajor() << ":";
    builder << "minor:" << beaconInfo.GetMinor();
    return builder;
}

NJson::TJsonValue NDrive::IBeaconRecognizer::TLocationData::ToJson() const {
    NJson::TJsonValue result;
    result.InsertValue("name", Name);
    result.InsertValue("latitude", Latitude);
    result.InsertValue("longitude", Longitude);
    return result;
}

void NDrive::IBeaconRecognizer::TLocationData::FromJson(const NJson::TJsonValue& value) {
    Name = value["name"].GetStringRobust();
    Latitude = value["latitude"].GetDouble();
    Longitude = value["longitude"].GetDouble();
}

bool NDrive::IBeaconRecognizer::TLocationData::TryFromJson(const NJson::TJsonValue& value) {
    if (!value.Has("name")) {
        return false;
    }
    if (!value.Has("latitude") || !value["latitude"].IsDouble()) {
        return false;
    }
    if (!value.Has("longitude") || !value["longitude"].IsDouble()) {
        return false;
    }
    FromJson(value);
    return true;
}

TMaybe<NDrive::IBeaconRecognizer::TLocationData> NDrive::TStaticBeaconRecognizer::TryRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const {
    auto it = LocationsDataByKey.find(ConvertToKey(beaconInfo));
    if (it == LocationsDataByKey.end()) {
        return {};
    }
    return it->second;
}

bool NDrive::TStaticBeaconRecognizer::CanRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const {
    return LocationsDataByKey.find(ConvertToKey(beaconInfo)) != LocationsDataByKey.end();
}

TMaybe<NDrive::IBeaconRecognizer::TLocationData> NDrive::TDynamicBeaconRecognizer::TryRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const {
    auto key = ConvertToKey(beaconInfo);
    TReadGuard guard(MapLock);
    auto it = LocationsDataByKey.find(key);
    if (it != LocationsDataByKey.end()) {
        return it->second;
    } else {
        return {};
    }
}

NDrive::TDynamicBeaconRecognizer::TDynamicBeaconRecognizer(TAtomicSharedPtr<TTelematicsDynamicSettings> settings, TDuration updateInterval)
    : IAutoActualization("TDynamicBeaconRecognizer", updateInterval)
    , Settings(settings)
{
    if (!Start()) {
        ythrow yexception() << "can't launch TDynamicBeaconRecognizer";
    }
}

NDrive::TDynamicBeaconRecognizer::~TDynamicBeaconRecognizer() {
    if (!Stop()) {
        ERROR_LOG << "can't stop TDynamicBeaconRecognizer";
    }
}

bool NDrive::TDynamicBeaconRecognizer::CanRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const {
    auto key = ConvertToKey(beaconInfo);
    bool foundKey;
    {
        TReadGuard guard(MapLock);
        foundKey = LocationsDataByKey.count(key);
    }
    return foundKey;
}

bool NDrive::TDynamicBeaconRecognizer::Refresh() {
    auto maybeString = Settings->template Get<TString>(BeaconsSettingsName);
    if (maybeString.Defined()) {
        INFO_LOG << "TDynamicBeaconRecognizer: " << "found beacons settings in TelematicsDynamicSettings" << Endl;
        TMap<TString, TLocationData> data;
        NJson::TJsonValue value;
        if (!NJson::ReadJsonFastTree(*maybeString, &value) || !TryFromJson(value, data)) {
            ALERT_LOG << "TDynamicBeaconRecognizer: " << "can't parse beacons settings" << Endl;
            return false;
        }
        TWriteGuard guard(MapLock);
        LocationsDataByKey = std::move(data);
    } else {
        ALERT_LOG << "TDynamicBeaconRecognizer: " << "can't find beacons settings in TelematicsDynamicSettings" << Endl;
    }
    return true;
}

bool NDrive::TryFromJson(const NJson::TJsonValue& jsonValue, TMap<TString, IBeaconRecognizer::TLocationData>& locationData) {
    if (!jsonValue.IsMap()) {
        return false;
    }
    for (const auto& [key, value] : jsonValue.GetMap()) {
        if (!locationData[key].TryFromJson(value)) {
            return false;
        }
    }
    return true;
}
