#pragma once

#include <drive/telematics/server/common/settings.h>

#include <drive/telematics/common/beacon.h>

#include <library/cpp/json/json_reader.h>

#include <util/generic/map.h>
#include <util/generic/maybe.h>
#include <util/generic/string.h>

namespace NDrive {

    class IBeaconRecognizer {
    public:
        class TLocationData {
        public:
            NJson::TJsonValue ToJson() const;
            void FromJson(const NJson::TJsonValue& value);
            bool TryFromJson(const NJson::TJsonValue& value);

        public:
            TString Name;
            double Latitude;
            double Longitude;
        };
    public:
        static TString ConvertToKey(const NDrive::NVega::TBeaconInfo& beaconInfo);

    public:
        virtual ~IBeaconRecognizer() {}

        virtual TMaybe<TLocationData> TryRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const = 0;
        virtual bool CanRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const = 0;
    };

    class TStaticBeaconRecognizer : public IBeaconRecognizer {
    public:
        TStaticBeaconRecognizer(TMap<TString, TLocationData> locationsDataByKey)
            : LocationsDataByKey(std::move(locationsDataByKey))
        {
        }

        TMaybe<TLocationData> TryRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const override;
        bool CanRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const override;

    private:
        TMap<TString, TLocationData> LocationsDataByKey;
    };

    class TDynamicBeaconRecognizer final : public IBeaconRecognizer, public IAutoActualization {
    public:
        TDynamicBeaconRecognizer(TAtomicSharedPtr<TTelematicsDynamicSettings> settings, TDuration updateInterval);
        ~TDynamicBeaconRecognizer();

        TMaybe<TLocationData> TryRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const override;
        bool CanRecognizeLocation(const NVega::TBeaconInfo& beaconInfo) const override;

    protected:
        bool Refresh() override;

    private:
        TAtomicSharedPtr<TTelematicsDynamicSettings> Settings;
        TMap<TString, TLocationData> LocationsDataByKey;
        const TString BeaconsSettingsName = "known_beacons";
        mutable TRWMutex MapLock;
    };

    bool TryFromJson(const NJson::TJsonValue& jsonValue, TMap<TString, IBeaconRecognizer::TLocationData>& locationData);
}
