#include "history.h"

#include <drive/telematics/server/location/proto/history.pb.h>

#include <rtline/util/algorithm/container.h>

namespace {
    class TCompareByTimestamp {
    public:
        bool operator()(const NDrive::TGpsLocation& left, const NDrive::TGpsLocation& right) const {
            return left.Timestamp < right.Timestamp;
        }
        bool operator()(const NDrive::TGpsLocation& left, TInstant right) const {
            return left.Timestamp < right;
        }
        bool operator()(TInstant left, const NDrive::TGpsLocation& right) const {
            return left < right.Timestamp.Get();
        }
    };
    auto CompareByTimestamp = TCompareByTimestamp{};
    bool Equal(const NDrive::TGpsLocation& left, const NDrive::TGpsLocation& right, double precision = 0.001) {
        return
            left.Speed == right.Speed &&
            std::abs(left.Latitude - right.Latitude) < precision &&
            std::abs(left.Longitude - right.Longitude) < precision;
    }
}

size_t NDrive::TLocationHistory::GetCount() const {
    return Locations.size();
}

TInstant NDrive::TLocationHistory::GetTimestamp() const {
    TReadGuard guard(Lock);
    if (!Locations.empty()) {
        return Locations.back().Timestamp;
    } else {
        return {};
    }
}

TMaybe<NDrive::TGpsLocation> NDrive::TLocationHistory::GetValueAt(TInstant timestamp, bool skipZeros) const {
    TReadGuard guard(Lock);
    auto lk = LowerKey(Locations, timestamp, CompareByTimestamp);
    if (lk == Locations.end()) {
        return {};
    }
    Y_ASSERT(lk->Timestamp <= timestamp);
    while (skipZeros && !(*lk) && lk != Locations.begin()) {
        --lk;
    }
    return *lk;
}

TVector<NDrive::TGpsLocation> NDrive::TLocationHistory::GetValues() const {
    TReadGuard guard(Lock);
    return MakeVector(Locations);
}

void NDrive::TLocationHistory::Add(const TGpsLocation& location) {
    TWriteGuard guard(Lock);
    Y_ASSERT(location.Timestamp >= location.Since);
    Y_ASSERT(std::is_sorted(Locations.begin(), Locations.end(), CompareByTimestamp));
    if (Locations.empty()) {
        Locations.push_back(location);
    } else if (Locations.back().Timestamp <= location.Timestamp) {
        if (Equal(Locations.back(), location)) {
            Locations.back().Timestamp = location.Timestamp;
        } else {
            Locations.push_back(location);
        }
    } if (Locations.front().Timestamp >= location.Timestamp) {
        if (Equal(Locations.front(), location)) {
            Locations.front().Since = location.Since;
        } else {
            Locations.push_front(location);
        }
    } else {
        Locations.push_back(location);
        std::sort(Locations.begin(), Locations.end(), CompareByTimestamp);
    }
    Y_ASSERT(std::is_sorted(Locations.begin(), Locations.end(), CompareByTimestamp));
    for (auto&& i : Locations) {
        Y_ASSERT(i.Since <= i.Timestamp);
    }
    while (!Locations.empty() && Locations.back().Since - Locations.front().Timestamp > Depth) {
        Locations.pop_front();
    }
}

void NDrive::TLocationHistory::Serialize(NDrive::NProto::TLocationHistory& proto) const {
    TReadGuard guard(Lock);
    for (auto&& location : Locations) {
        auto lp = proto.AddLocation();
        Y_ENSURE(lp);
        location.Serialize(*lp);
    }
}

bool NDrive::TLocationHistory::Deserialize(const NDrive::NProto::TLocationHistory& proto) {
    TWriteGuard guard(Lock);
    Locations.clear();
    for (auto&& lp : proto.GetLocation()) {
        if (!Locations.emplace_back().Deserialize(lp)) {
            return false;
        }
    }
    return true;
}
