#include "location.h"

#include <drive/telematics/server/location/proto/location.pb.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>

#include <util/string/cast.h>

constexpr double CoordinatePrecision = 0.001;

NDrive::NProto::TLocation NDrive::TLocation::ToProto() const {
    NDrive::NProto::TLocation result;
    result.SetLatitude(Latitude);
    result.SetLongitude(Longitude);
    result.SetPrecision(Precision);
    result.SetCourse(Course);
    result.SetTimestamp(Timestamp.Seconds());
    result.SetSince(Since.Seconds());
    result.SetType(static_cast<ui32>(Type));
    if (Name) {
        result.SetName(Name);
    }
    if (Content) {
        result.SetContent(Content);
    }
    if (GeoHash) {
        result.SetGeoHash(GeoHash);
    }
    return result;
}

bool NDrive::TLocation::FromProto(const NDrive::NProto::TLocation& proto) {
    Latitude = proto.GetLatitude();
    Longitude = proto.GetLongitude();
    Precision = proto.GetPrecision();
    Course = proto.GetCourse();
    Timestamp = TInstant::Seconds(proto.GetTimestamp());
    Since = TInstant::Seconds(proto.GetSince());
    Type = static_cast<EType>(proto.GetType());
    Name = proto.GetName();
    Content = proto.GetContent();
    GeoHash = proto.GetGeoHash();
    return true;
}

TGeoCoord NDrive::TLocation::GetCoord() const {
    return TGeoCoord(Longitude, Latitude);
}

NDrive::TLocation& NDrive::TLocation::MutableBase() {
    if (!Base) {
        Base = MakeHolder<TLocation>().Release();
    }
    return *Base;
}

bool NDrive::TLocation::Cross(const TLocation& other, TLocation* result) const {
    TGeoCoord first(Longitude, Latitude);
    TGeoCoord second(other.Longitude, other.Latitude);
    double distance = first.GetLengthTo(second);
    if (distance > Precision + other.Precision) {
        return false;
    }

    if (result) {
        if (Precision < other.Precision) {
            *result = *this;
        } else {
            *result = other;
        }
    }
    return true;
}

bool NDrive::TLocation::IsRealtime() const {
    return Type == GPSCurrent || Type == Linked;
}

bool NDrive::TLocation::IsZero() const {
    return std::abs(Latitude) < CoordinatePrecision && std::abs(Longitude) < CoordinatePrecision;
}

NDrive::TLocation& NDrive::TLocation::SetBase(const NDrive::TLocation& base) {
    MutableBase() = base;
    return *this;
}

bool NDrive::TLocation::TryFromJson(const NJson::TJsonValue& value) {
    unsigned long long timestamp;
    unsigned long long baseTimestamp;
    unsigned long long since;
    TString type;
    double baseLatitude;
    double baseLongitude;
    double precision;
    double course;
    value["name"].GetString(&Name);
    value["content"].GetString(&Content);
    value["geohash"].GetString(&GeoHash);
    value["base_latitude"].GetDouble(&baseLatitude);
    value["base_longitude"].GetDouble(&baseLongitude);
    if (value["base_timestamp"].GetUInteger(&baseTimestamp)) {
        TLocation base;
        base.Latitude = baseLatitude;
        base.Longitude = baseLongitude;
        base.Timestamp = TInstant::Seconds(baseTimestamp);
        SetBase(base);
    }
    const NJson::TJsonValue& base = value["base"];
    if (base.IsDefined()) {
        if (!MutableBase().TryFromJson(base)) {
            return false;
        }
    }
    if ((value["latitude"].GetDouble(&Latitude) || value["lat"].GetDouble(&Latitude)) &&
        (value["longitude"].GetDouble(&Longitude) || value["lon"].GetDouble(&Longitude)) &&
        value["course"].GetDouble(&course) &&
        value["timestamp"].GetUInteger(&timestamp) &&
        value["type"].GetString(&type) &&
        TryFromString(type, Type)
    ) {
        Course = static_cast<float>(course);
        Timestamp = TInstant::Seconds(timestamp);
    } else {
        return false;
    }
    if (value["since"].GetUInteger(&since)) {
        Since = TInstant::Seconds(since);
    } else {
        Since = Timestamp;
    }
    if (Since > Timestamp) {
        return false;
    }
    if (value["precision"].GetDouble(&precision)) {
        Precision = static_cast<float>(precision);
    }
    return true;
}

NJson::TJsonValue NDrive::TLocation::ToJson() const {
    NJson::TJsonValue result;
    result["latitude"] = Latitude;
    result["longitude"] = Longitude;
    result["course"] = Course;
    result["timestamp"] = Timestamp.Seconds();
    result["type"] = ToString(Type);
    if (Since) {
        result["since"] = Since.Seconds();
    }
    if (Base) {
        result["base"] = Base->ToJson();
    }
    if (Content) {
        result["content"] = Content;
    }
    if (Name) {
        result["name"] = Name;
    }
    if (GeoHash) {
        result["geohash"] = GeoHash;
    }
    if (std::abs(Precision) > 0.001) {
        result["precision"] = Precision;
    }
    return result;
}

TGeoCoord NDrive::TSimpleLocation::GetCoord() const {
    return TGeoCoord(Longitude, Latitude);
}

NDrive::TSimpleLocation::operator bool() const {
    return std::abs(Latitude) > CoordinatePrecision || std::abs(Longitude) > CoordinatePrecision;
}

void NDrive::TGpsLocation::Serialize(NDrive::NProto::TLocation& proto) const {
    proto.SetLatitude(Latitude);
    proto.SetLongitude(Longitude);
    proto.SetCourse(Course);
    proto.SetTimestamp(Timestamp.Seconds());
    proto.SetSince(Since.Seconds());
    proto.SetSpeed(Speed);
    proto.SetType(static_cast<ui32>(Type));
}

bool NDrive::TGpsLocation::Deserialize(const NDrive::NProto::TLocation& proto) {
    Latitude = proto.GetLatitude();
    Longitude = proto.GetLongitude();
    Course = proto.GetCourse();
    Timestamp = TInstant::Seconds(proto.GetTimestamp());
    Since = TInstant::Seconds(proto.GetSince());
    Speed = proto.GetSpeed();
    Type = static_cast<EType>(proto.GetType());
    return true;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TLocation& object) {
    return object.ToJson();
}

template <>
bool NJson::TryFromJson(const TJsonValue& value, NDrive::TLocation& result) {
    return result.TryFromJson(value);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TSimpleLocation& object) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "lat", object.Latitude);
    NJson::InsertField(result, "lon", object.Longitude);
    NJson::InsertField(result, "dir", object.Course);
    NJson::InsertField(result, "since", object.Since);
    NJson::InsertField(result, "timestamp", object.Timestamp);
    NJson::InsertField(result, "type", NJson::Stringify(object.Type));
    return result;
}

template <>
bool NJson::TryFromJson(const TJsonValue& value, NDrive::TSimpleLocation& result) {
    return
        NJson::ParseField(value["lat"], result.Latitude) &&
        NJson::ParseField(value["lon"], result.Longitude) &&
        NJson::ParseField(value["dir"], result.Course) &&
        NJson::ParseField(value["since"], result.Since) &&
        NJson::ParseField(value["timestamp"], result.Timestamp) &&
        NJson::ParseField(value["type"], NJson::Stringify(result.Type));
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TGpsLocation& object) {
    NJson::TJsonValue result = ToJson<NDrive::TSimpleLocation>(object);
    NJson::InsertField(result, "speed", object.Speed);
    return result;
}
