#pragma once

#include <library/cpp/json/json_value.h>

#include <rtline/util/types/timestamp.h>

#include <util/datetime/base.h>

#include <cmath>

class TGeoCoord;

namespace NDrive::NProto {
    class TLocation;
}

namespace NDrive {
    struct TLocation {
    public:
        enum EType {
            Unknown = 0,
            GPSCurrent = 1,
            GPSPrevious = 2,
            LBS = 3,
            Linked = 4,
            LinkedPrevious = 5,
            External = 6,
            Beacon = 7,
            Projection = 8,
            ProjectionPrevious = 9,
        };

    public:
        TCopyPtr<TLocation> Base;

        TString Name;
        TString Content;
        TString GeoHash;
        double Latitude = 0;
        double Longitude = 0;
        float Precision = 0;
        float Course = 0;
        TInstant Timestamp = TInstant::Zero();
        TInstant Since = TInstant::Zero();
        EType Type = Unknown;

    public:
        TLocation() = default;
        TLocation(double latitude, double longitude, double precision, double course, EType type, TInstant timestamp = TInstant::Zero(), TInstant since = TInstant::Zero())
            : Latitude(latitude)
            , Longitude(longitude)
            , Precision(precision)
            , Course(course)
            , Timestamp(timestamp)
            , Since(since ? since : timestamp)
            , Type(type)
        {
        }

        explicit operator bool() const {
            return !IsZero();
        }

        TGeoCoord GetCoord() const;

        TLocation& MutableBase();
        TLocation& SetBase(const NDrive::TLocation& base);

        bool Cross(const TLocation& other, TLocation* result = nullptr) const;
        bool IsRealtime() const;
        bool IsZero() const;

        bool TryFromJson(const NJson::TJsonValue& value);
        NJson::TJsonValue ToJson() const;

        NDrive::NProto::TLocation ToProto() const;
        bool FromProto(const NDrive::NProto::TLocation& proto);
    };
    using TLocations = TVector<TLocation>;

    struct TSignalqLocation {
        TLocation Location;
        double Speed = 0;

        explicit TSignalqLocation(double latitude, double longitude, double precision, double course, TLocation::EType type, double speed, TInstant timestamp = TInstant::Zero(), TInstant since = TInstant::Zero())
        {
            Location = TLocation(
                latitude,
                longitude,
                precision,
                course,
                type,
                timestamp,
                since
            );
            Speed = speed;
        }
    };

    struct TSimpleLocation {
    public:
        using EType = TLocation::EType;

    public:
        float Latitude = 0;
        float Longitude = 0;
        float Course = 0;
        TTimestamp Timestamp;
        TTimestamp Since;
        EType Type = EType::Unknown;

    public:
        inline TSimpleLocation() = default;
        inline TSimpleLocation(const TLocation& location)
            : Latitude(location.Latitude)
            , Longitude(location.Longitude)
            , Course(location.Course)
            , Timestamp(location.Timestamp)
            , Since(location.Since)
            , Type(location.Type)
        {
        }

        TGeoCoord GetCoord() const;

        inline operator TLocation() const {
            return TLocation(
                Latitude,
                Longitude,
                /*precision=*/0.0,
                Course,
                Type,
                Timestamp,
                Since
            );
        }

        explicit operator bool() const;
    };

    struct TGpsLocation: public TSimpleLocation {
    public:
        ui32 Speed = 0;

    public:
        inline TGpsLocation() {
            Type = EType::GPSCurrent;
        }

        void Serialize(NDrive::NProto::TLocation& proto) const;
        bool Deserialize(const NDrive::NProto::TLocation& proto);
    };
    using TGpsLocations = TVector<TGpsLocation>;
}
