#include "locator.h"

#include "beacon_recognizer.h"
#include "cache.h"
#include "names.h"

#include <drive/telematics/server/common/signals.h>
#include <drive/telematics/server/sensors/cache.h>

#include <drive/telematics/api/client.h>
#include <drive/telematics/api/sensor/local.h>
#include <drive/telematics/protocol/vega.h>

#include <drive/library/cpp/geocoder/api/client.h>
#include <drive/library/cpp/lbs/api/client.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/tracks/client.h>

#include <library/cpp/geohash/geohash.h>

#include <rtline/api/graph/router/router.h>

#include <util/generic/adaptor.h>
#include <util/generic/xrange.h>
#include <util/random/random.h>

namespace {
    using TClusterId = size_t;
    using TElementIdx = size_t;
    using TDistanceFunction = std::function<double(const NDrive::TLocation&, const NDrive::TLocation&)>;

    constexpr auto IncorrectClusterId = static_cast<TClusterId>(-1);

    TVector<TClusterId> Clusterize(const TVector<NDrive::TLocation>& locations, double threshold, const TDistanceFunction& distance) {
        TVector<TClusterId> clusters = xrange<TClusterId>(0, locations.size());
        for (size_t i = 0; i < locations.size(); ++i) {
            for (size_t j = i + 1; j < locations.size(); ++j) {
                auto d = distance(locations[i], locations[j]);
                if (d <= threshold) {
                    auto cluster = std::min(clusters[i], clusters[j]);
                    Y_ASSERT(cluster <= i);
                    Y_ASSERT(cluster <= j);
                    clusters[i] = cluster;
                    clusters[j] = cluster;
                }
            }
        }
        // collapse cluster indices
        TVector<TClusterId> remap;
        TClusterId current = 0;
        for (auto&& cluster : clusters) {
            if (remap.size() <= cluster) {
                remap.resize(cluster + 1, IncorrectClusterId);
            }
            if (remap[cluster] == IncorrectClusterId) {
                remap[cluster] = current++;
            }
            cluster = remap[cluster];
        }
        return clusters;
    }

    TVector<TVector<TElementIdx>> Transpose(const TVector<TClusterId>& clusters) {
        TVector<TVector<TElementIdx>> result;
        for (size_t i = 0; i < clusters.size(); ++i) {
            TClusterId cluster = clusters[i];
            Y_ASSERT(cluster <= i);
            if (result.size() <= cluster) {
                result.resize(cluster + 1);
            }
            result[cluster].push_back(i);
        }
        for (auto&& cluster : result) {
            Y_ASSERT(!cluster.empty());
            Y_ASSERT(std::is_sorted(cluster.begin(), cluster.end()));
        }
        return result;
    }

    TClusterId GetLargestCluster(const TVector<TVector<TElementIdx>>& clusters) {
        TClusterId result = 0;
        for (TClusterId i = 0; i < clusters.size(); ++i) {
            if (clusters[result].size() < clusters[i].size()) {
                result = i;
            }
        }
        Y_ASSERT(result < clusters.size());
        return result;
    }
}

NDrive::TLocator::TLocator(const TOptions& options /*= Default<TOptions>()*/, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm /*= nullptr*/)
    : Options(options)
{
    if (Options.LBSToken) {
        LBSClient = MakeHolder<NDrive::TLBSClient>(Options.LBSToken);
    }
    if (Options.SensorHost) {
        NRTLine::TRequestBuilder::TBalancerOptions balancerOptions;
        balancerOptions.BalancerTimeoutTable = Options.BalancerTimeoutTable;
        auto tvmAuth = tvm && Options.SensorSaasTvmId ? MakeMaybe<NDrive::TTvmAuth>(tvm, Options.SensorSaasTvmId) : Nothing();
        SensorClient = MakeAtomicShared<NRTLine::TNehSearchClient>(
                NRTLine::TNehSearchClient::TEndpoint(Options.SensorHost, Options.SensorPort)
                        .SetService(Options.SensorService)
                        .SetBalancerOptions(balancerOptions),
                NSimpleMeta::TConfig::ForRequester(),
                tvmAuth,
                Options.SensorSpMetaSearch
        );
        SensorApi = MakeAtomicShared<NDrive::TSensorApi>(options.SensorAPIName, *SensorClient);
    } else {
        SensorApi = MakeAtomicShared<TLocalSensorApi>();
    }
    if (Options.LinkerHost) {
        NGraph::TRouter::TOptions opts;
        opts.Timeout = Options.LinkLinkerTimeout;
        LinkerClient = MakeAtomicShared<NGraph::TRouter>(Options.LinkerHost, Options.LinkerPort, Options.LinkerService, opts);
    }
    if (Options.TracksHost) {
        TracksClient = MakeAtomicShared<NDrive::TTracksClient>(Options.TracksService, Options.TracksHost, Options.TracksPort);
    }
    if (Options.GeocoderHost) {
        NDrive::TGeocoder::TOptions opts;
        opts.Host = Options.GeocoderHost;
        opts.Port = Options.GeocoderPort ? Options.GeocoderPort : opts.Port;
        opts.Path = Options.GeocoderPath ? Options.GeocoderPath : opts.Path;
        opts.TvmDestination = Options.GeocoderClientId ? Options.GeocoderClientId : opts.TvmDestination;
        auto tvmAuth = tvm ? MakeMaybe<NDrive::TTvmAuth>(tvm, opts.TvmDestination) : Nothing();
        Geocoder = MakeAtomicShared<NDrive::TGeocoder>(opts, std::move(tvmAuth));
    }
}

NDrive::TLocator::~TLocator() {
}

NDrive::TLocator::TAsyncLocations NDrive::TLocator::LocateAll(const TString& imei, const TSensorsCache& sensors, const TLocationCache* cache) const {
    TAsyncLocations result;
    auto gpss = GetGPS(sensors);
    auto now = Now();
    bool stationary = IsStationary(sensors);

    auto regular = Locate(imei, sensors, gpss);
    if (regular.Initialized()) {
        result.push_back(regular);
    }

    auto linked = Link(imei, sensors, gpss);
    if (linked.Initialized()) {
        result.push_back(linked);
    }

    if (auto lbs = LBS(imei, sensors, gpss); lbs.Initialized()) {
        result.push_back(std::move(lbs));
    }

    auto previousGeocodedLocation = cache ? cache->GetLocation(GeocodedLocationName) : TMaybe<TLocation>();
    auto previousGeocodedTimestamp = previousGeocodedLocation ? previousGeocodedLocation->Timestamp : TInstant::Zero();
    if (stationary && now > previousGeocodedTimestamp + Options.GeocoderRefreshInterval) {
        auto geocoded = Geocode(imei, linked.Initialized() ? linked : regular);
        if (geocoded.Initialized()) {
            result.push_back(std::move(geocoded));
        }
    }

    return result;
}

NDrive::TLocator::TAsyncLocation NDrive::TLocator::Locate(const TString& imei, const TSensorsCache& sensors) const {
    auto gpss = GetGPS(sensors);
    return Locate(imei, sensors, gpss);
}

NDrive::TLocator::TAsyncLocation NDrive::TLocator::Locate(const TString& imei, const TSensorsCache& sensors, TLocations gpss) const {
    NDrive::TLocation raw = gpss.size() ? gpss.back() : NDrive::TLocation();
    NDrive::TLocation dirty = GetGPSNonZero(gpss);
    NDrive::TLocation validated = GetGPSValidated(imei, sensors, std::move(gpss));

    if (validated.Type == NDrive::TLocation::GPSCurrent) {
        return NThreading::MakeFuture(validated);
    }

    if (dirty.IsZero()) {
        if (SensorApi) {
            INFO_LOG << imei << ": restoring sensors from Sensor API" << Endl;
            auto asyncMultiSensors = SensorApi->GetSensors(imei, { VEGA_DIR, VEGA_LAT, VEGA_LON });
            if (!asyncMultiSensors.Wait(Options.SensorTimeout)) {
                ERROR_LOG << imei << ": could not wait for sensors request" << Endl;
            }
            if (asyncMultiSensors.HasValue()) {
                const auto& multisensors = asyncMultiSensors.GetValue();
                auto course = NDrive::ISensorApi::FindSensor(multisensors, VEGA_DIR);
                auto latitude = NDrive::ISensorApi::FindSensor(multisensors, VEGA_LAT);
                auto longitude = NDrive::ISensorApi::FindSensor(multisensors, VEGA_LON);
                if (course && latitude && longitude) {
                    dirty = GetGPS(sensors, *latitude, *longitude, *course, 1);
                } else {
                    ERROR_LOG << imei << ": sensors are missing in Sensor API " << !course.Empty() << !latitude.Empty() << !longitude.Empty() << Endl;
                }
            }
            if (asyncMultiSensors.HasException()) {
                ERROR_LOG << imei << ": an exception occurred during sensor request " << NThreading::GetExceptionMessage(asyncMultiSensors) << Endl;
            }
        }
    }
    if (dirty.Precision > Options.PrecisionThreshold) {
        dirty = {};
    }
    if (validated.IsZero()) {
        if (SensorApi) {
            INFO_LOG << imei << ": restoring location from Sensor API" << Endl;
            auto asyncLocation = SensorApi->GetLocation(imei);
            if (!asyncLocation.Wait(Options.SensorTimeout)) {
                ERROR_LOG << imei << ": could not wait for location request" << Endl;
            }
            if (asyncLocation.HasValue()) {
                auto location = asyncLocation.ExtractValue();
                if (location) {
                    validated = *location;
                    INFO_LOG << imei << ": restored location " << validated.ToJson().GetStringRobust() << Endl;
                } else {
                    ERROR_LOG << imei << ": location is missing in Sensor API" << Endl;
                }
            }
            if (asyncLocation.HasException()) {
                ERROR_LOG << imei << ": an exception occurred during location request " << NThreading::GetExceptionMessage(asyncLocation) << Endl;
            }
        }
    }
    if (validated.Precision > Options.PrecisionThreshold) {
        validated = {};
    }

    constexpr ui8 geoHashPrecision = 10;
    if (LBSClient && Options.EnableLocateFromLbs) {
        auto lbs = GetGSM(imei, sensors);
        return lbs.Apply([imei, raw, dirty, validated, this](const NThreading::TFuture<TLocation>& l) -> TLocation {
            const NDrive::TLocation& lbs = l.GetValue();
            NDrive::TLocation fixedLBS = lbs.Precision <= Options.PrecisionThreshold ? lbs : NDrive::TLocation();
            NDrive::TLocation result = Locate(imei, dirty, validated, fixedLBS);
            result.SetBase(raw);
            result.GeoHash = NGeoHash::EncodeToString(result.Latitude, result.Longitude, geoHashPrecision);
            return result;
        });
    } else {
        NDrive::TLocation result = Locate(imei, dirty, validated, NDrive::TLocation());
        result.SetBase(raw);
        result.GeoHash = NGeoHash::EncodeToString(result.Latitude, result.Longitude, geoHashPrecision);
        return NThreading::MakeFuture(result);
    }
}

NDrive::TLocator::TAsyncLocation NDrive::TLocator::Link(const TString& imei, const TSensorsCache& sensors) const {
    auto gpss = GetGPS(sensors);
    return Link(imei, sensors, gpss);
}

NDrive::TLocator::TAsyncLocation NDrive::TLocator::Link(const TString& imei, const TSensorsCache& sensors, const TLocations& gpss) const {
    if (!LinkerClient || !TracksClient) {
        return {};
    }

    if (Options.LinkFilterByRandomFraction > 0) {
        if (Options.LinkFilterByRandomFraction > RandomNumber<double>()) {
            DEBUG_LOG << imei << ": skip linking by random" << Endl;
            return {};
        }
    }
    if (Options.LinkFilterBySpeed) {
        auto speed = sensors.Get(VEGA_SPEED);
        if (speed && speed->ConvertTo<double>() > 0) {
            DEBUG_LOG << imei << ": skip linking by speed" << Endl;
            return {};
        }
    }
    if (Options.LinkFilterByEngine) {
        auto engine = sensors.Get(CAN_ENGINE_IS_ON);
        if (engine && engine->ConvertTo<bool>()) {
            DEBUG_LOG << imei << ": skip linking by engine" << Endl;
            return {};
        }
    }

    NDrive::TLocation dirty = GetGPSNonZero(gpss);
    return Link(imei, dirty, Options.LinkTailLength);
}

NDrive::TLocator::TAsyncLocation NDrive::TLocator::Link(const TString& imei, const TLocation& dirty, double tailLength) const {
    NDrive::TTrackQuery trackQuery;
    trackQuery.IMEI = imei;
    trackQuery.NumDoc = 10;
    if (Options.LinkQueryByTimestamp) {
        trackQuery.Until = dirty.Timestamp;
    }

    auto tracksResponse = TracksClient->GetTracks(trackQuery, Options.LinkTrackTimeout);
    auto tracks = tracksResponse.Apply([this, dirty](const NThreading::TFuture<NDrive::TTracks>& tr) {
        if (!tr.HasValue()) {
            TTelematicsUnistatSignals::Get().LinkerTrackRequestFailed.Signal(1);
        }

        const auto realtimeThreshold = Options.LinkRealtimeThreshold;
        const auto& tracks = tr.GetValue();
        NDrive::TTracks result;
        for (auto&& track : tracks) {
            if (track.Coordinates.empty()) {
                continue;
            }
            if (Options.LinkQueryByTimestamp) {
                Y_ASSERT(track.Coordinates.front().Timestamp < dirty.Timestamp);
            }
            if (track.Coordinates.back().Timestamp + realtimeThreshold < dirty.Timestamp) {
                ythrow yexception() << "lag detected: " << track.Coordinates.back().Timestamp << '/' << dirty.Timestamp;
            } else {
                break;
            }
        }
        for (auto&& track : tracks) {
            if (track.Status == NDrive::ECarStatus::csRide) {
                result.push_back(track);
            }
        }
        for (auto&& track : Reversed(result)) {
            track = NDrive::CropTrack(std::move(track), TInstant::Zero(), dirty.Timestamp);
            break;
        }
        Y_ENSURE(!result.empty(), "no tracks: " << tracks.size());
        return result;
    });

    auto linked = tracks.Apply([this, tailLength](const NThreading::TFuture<NDrive::TTracks>& t) {
        if (!t.HasValue()) {
            TTelematicsUnistatSignals::Get().LinkerTrackParsingFailed.Signal(1);
        }

        const auto linker = LinkerClient;
        CHECK_WITH_LOG(linker);
        const auto& tracks = t.GetValue();
        Y_ENSURE(!tracks.empty(), "no tracks");
        const auto optionalTail = NDrive::GetTail(tracks, tailLength);
        const auto d = Options.LinkDistancePrecision;
        const auto fcWeight = Options.LinkFCWeight;

        Y_ENSURE(optionalTail, "cannot GetTail from " << tracks.size() << " tracks");
        const auto& tail = *optionalTail;

        NGraph::TRouter::TMatchingOptions options;
        options.AskFeatures = true;
        options.DistancePrecision = d;
        options.FCWeight = fcWeight;
        options.Timestamp = tail.Coordinates.size() ? tail.Coordinates.back().Timestamp : TInstant::Zero();
        return linker->GetEdgesAsync(tail.Coordinates, options);
    });

    auto linkedOrProjection = linked.Apply([this, dirty, imei](const NThreading::TFuture<NGraph::TRouter::TMatch>& l) {
        if (!l.HasValue()) {
            TTelematicsUnistatSignals::Get().LinkerMatchingFailed.Signal(1);
            if (Options.LinkEnableProjection) {
                ERROR_LOG << imei << ": falling back to dirty projection " << dirty.ToJson().GetStringRobust() << " due to " << NThreading::GetExceptionMessage(l) << Endl;
                NGraph::TRouter::TTimedGeoCoordinates coordinates;
                coordinates.emplace_back(dirty.GetCoord(), dirty.Timestamp);

                NGraph::TRouter::TMatchingOptions matchingOptions;
                matchingOptions.AskFeatures = true;
                matchingOptions.DistancePrecision = Options.LinkDistancePrecision;
                matchingOptions.FCWeight = static_cast<float>(Options.LinkFCWeight);
                matchingOptions.Timestamp = dirty.Timestamp;
                matchingOptions.FeaturePrecision = 0;

                return LinkerClient->GetEdgesAsync(coordinates, matchingOptions);
            }
        }
        return l;
    });

    auto location = linkedOrProjection.Apply([this, dirty, imei, tailLength](const NThreading::TFuture<NGraph::TRouter::TMatch>& l) {
        if (!l.HasValue()) {
            TTelematicsUnistatSignals::Get().LinkerMatchingOrProjectingFailed.Signal(1);
        }

        const auto& linked = l.GetValue();
        if (linked.Elements.empty() && tailLength < Options.LinkTailLengthLimit) {
            const auto nextTailLength = tailLength + Options.LinkTailLength;
            NOTICE_LOG << imei << ": relink " << dirty.ToJson().GetStringRobust() << " with tail length " << nextTailLength << Endl;
            return Link(imei, dirty, nextTailLength);
        }

        const auto deviationLengthThreshold = Options.LinkDeviationLengthThreshold;
        const auto shift = Options.LinkCoordinateShift;

        Y_ENSURE(!linked.Elements.empty(), "no matched edges");
        auto finish = linked.Elements.rbegin();
        auto i = finish;
        double deviation = 0;
        for (; i != linked.Elements.rend(); ++i) {
            if (i->FC < static_cast<ui32>(ERoadFC::fcMinorRoads)) {
                break;
            }
            deviation += i->Length;
        }
        if (i != linked.Elements.rend() && deviation < deviationLengthThreshold) {
            finish = i;
        }

        const auto& edge = *finish;
        auto position = TGeoCoord();
        auto previous = TGeoCoord();
        if (linked.LinePointsCount > 1) {
            Y_ENSURE(edge.PolyLine.Size() >= 2, "bad projection: " << edge.PolyLine.ToString());
            position = edge.PolyLine.GetCoords().back();
            previous = edge.PolyLine.GetCoords().at(edge.PolyLine.GetCoords().size() - 2);
        } else {
            Y_ENSURE(edge.PolyLine.Size() == 1, "bad projection: " << edge.PolyLine.ToString());
            Y_ENSURE(edge.GeometryShape.size() >= 1, "bad GeometryShape: " << TGeoCoord::SerializeVector(edge.GeometryShape));
            position = edge.PolyLine.GetCoords()[0];
            previous = edge.GeometryShape.front();
        }
        const bool bidirectional = std::abs(edge.Features[NGraph::FI_ONE_WAY_DURATION]) < NGraph::FeaturePrecision;

        double dlon = (position.X - previous.X) * M_PI / 180;
        double lat0 = previous.Y * M_PI / 180;
        double lat1 = position.Y * M_PI / 180;

        double a = sin(dlon) * cos(lat1);
        double b = cos(lat0) * sin(lat1) - sin(lat0) * cos(lat1) * cos(dlon);

        double bearing = atan2(a, b);
        double course = 180 / M_PI * (bearing >= 0 ? bearing : 2 * M_PI + bearing);
        float downcourse = std::clamp<ui32>(std::floor(course), 0, 359);
        if (bidirectional) {
            if (std::abs(downcourse - dirty.Course) > 90) {
                downcourse += 180;
            }
        }
        while (downcourse > 360) {
            downcourse -= 360;
        }

        TGeoCoord original = dirty.GetCoord();
        TGeoPolyLine connection({ position, original });
        TGeoCoord shifted = connection.GetCoordByLength(shift);

        NDrive::TLocation::EType type = NDrive::TLocation::Unknown;
        if (linked.LinePointsCount > 1) {
            type = dirty.IsRealtime() ? NDrive::TLocation::Linked : NDrive::TLocation::LinkedPrevious;
        } else {
            type = dirty.IsRealtime() ? NDrive::TLocation::Projection : NDrive::TLocation::ProjectionPrevious;
        }

        NDrive::TLocation result = {
            shifted.Y,
            shifted.X,
            /*precision=*/std::min(connection.GetLength(), 1000.0),
            downcourse,
            type,
            dirty.Timestamp,
            /*since=*/linked.MetaInfo.Timestamp
        };
        result.SetBase(dirty);
        result.Name = LinkedLocationName;

        TTelematicsUnistatSignals::Get().LinkerSuccess.Signal(1);
        return NThreading::MakeFuture(result);
    });

    location.Subscribe([](const NThreading::TFuture<NDrive::TLocation>& l) {
        if (!l.HasValue()) {
            TTelematicsUnistatSignals::Get().LinkerLocatingFailed.Signal(1);
        }
    });

    return location;
}

NDrive::TLocator::TAsyncLocation NDrive::TLocator::LBS(const TString& imei, const TSensorsCache& sensors) const {
    auto gpss = GetGPS(sensors);
    return LBS(imei, sensors, gpss);
}

NDrive::TLocator::TAsyncLocation NDrive::TLocator::LBS(const TString& imei, const NDrive::TSensorsCache& sensors, const TLocations& gpss) const {
    if (!LBSClient) {
        return {};
    }

    NDrive::TLocation raw = gpss.size() ? gpss.back() : NDrive::TLocation();
    if (!raw.IsZero()) {
        return {};
    }

    auto gsm = GetGSM(imei, sensors);
    auto lbs = gsm.Apply([](const NThreading::TFuture<TLocation>& g) {
        auto gsm = g.GetValue();
        gsm.Name = LBSLocationName;
        return gsm;
    });
    return lbs;
}

NDrive::TLocator::TAsyncLocation NDrive::TLocator::Geocode(const TString& imei, const TAsyncLocation& location) const {
    Y_UNUSED(imei);
    if (!Geocoder) {
        return {};
    }
    if (!location.Initialized()) {
        return {};
    }

    auto decoded = location.Apply([geocoder = Geocoder](const NThreading::TFuture<TLocation>& l) {
        const auto& location = l.GetValue();
        Y_ENSURE(location, "cannot geodecode zero location");
        auto start = Now();
        auto decoded = geocoder->Decode(location.GetCoord());
        TTelematicsUnistatSignals::Get().GeocoderRequests.Signal(1);
        return decoded.Apply([start](const NThreading::TFuture<NDrive::TGeocoder::TResponse>& r) {
            if (r.HasValue()) {
                TInstant finish = Now();
                TDuration duration = finish - start;
                TTelematicsUnistatSignals::Get().GeocoderTimes.Signal(duration.MilliSeconds());
            }
            return r.GetValue();
        });
    });
    auto result = decoded.Apply([location](const NThreading::TFuture<NDrive::TGeocoder::TResponse>& r) {
        NDrive::TLocation result = location.GetValue();
        const auto& response = r.GetValue();
        result.Content = response.Title;
        result.Name = GeocodedLocationName;
        result.Type = TLocation::External;
        return result;
    });
    return result;
}

NDrive::TLocations NDrive::TLocator::GetGPS(const NDrive::TSensorsCache& sensors) const {
    TSet<TInstant> timestamps;
    {
        auto latitudeTimestamps = sensors.GetTimestamps(VEGA_LAT, Options.NumOfReviewedSensorValues);
        timestamps = std::move(latitudeTimestamps);
    }
    {
        auto longitudeTimestamps = sensors.GetTimestamps(VEGA_LON, Options.NumOfReviewedSensorValues);
        timestamps.insert(longitudeTimestamps.begin(), longitudeTimestamps.end());
    }

    NDrive::TLocations result;
    size_t i = 0;
    for (auto&& timestamp : Reversed(timestamps)) {
        auto index = i++;
        if (timestamp > Now() + Options.FutureThreshold) {
            continue;
        }

        auto course = sensors.Get(VEGA_DIR, timestamp);
        auto latitude = sensors.Get(VEGA_LAT, timestamp);
        auto longitude = sensors.Get(VEGA_LON, timestamp);
        if (!course || !latitude || !longitude) {
            continue;
        }
        auto location = GetGPS(sensors, *latitude, *longitude, *course, index);
        result.push_back(location);
    }
    std::sort(result.begin(), result.end(), [](const TLocation& left, const TLocation& right) {
        return std::make_tuple(left.Timestamp, left.Type == TLocation::GPSCurrent ? 1 : 0) < std::make_tuple(right.Timestamp, right.Type == TLocation::GPSCurrent ? 1 : 0);
    });
    return result;
}

NDrive::TLocation NDrive::TLocator::GetGPS(const TSensorsCache& sensors, const TSensor& latitude, const TSensor& longitude, const TSensor& course, size_t index) const {
    auto since = std::max(latitude.Since, longitude.Since);
    auto timestamp = std::max(latitude.Timestamp, longitude.Timestamp);
    Y_ASSERT(since <= timestamp);
    if (timestamp > Now() + Options.FutureThreshold) {
        return {};
    }

    const auto hdopSensor = sensors.Get(VEGA_HDOP, timestamp);
    const auto hdop = hdopSensor ? hdopSensor->ConvertTo<double>() : Options.DefaultHDOP;
    {
        NDrive::TLocation::EType type = index ? NDrive::TLocation::GPSPrevious : NDrive::TLocation::GPSCurrent;
        NDrive::TLocation location(
            std::get<double>(latitude.Value),
            std::get<double>(longitude.Value),
            GetPrecisionFromHDOP(hdop),
            std::get<double>(course.Value),
            type,
            timestamp,
            since
        );
        return location;
    }
}

NDrive::TLocation NDrive::TLocator::GetGPSNonZero(const NDrive::TLocations& locations) const {
    for (auto&& location : Reversed(locations)) {
        if (!location.IsZero()) {
            return location;
        }
    }
    return {};
}

NDrive::TLocation NDrive::TLocator::GetGPSValidated(const TString& imei, const TSensorsCache& sensors, NDrive::TLocations locations) const {
    for (auto i = locations.begin(); i != locations.end();) {
        const auto coordinate = i->GetCoord();
        const auto timestamp = i->Timestamp;

        const auto gpsJammedSensor = sensors.Get(VEGA_GPS_JAMMED, timestamp);
        const bool gpsJammed = gpsJammedSensor ? gpsJammedSensor->ConvertTo<bool>(false) : false;

        const auto gpsSpoofedSensor = sensors.Get(VEGA_GPS_SPOOF_SENSOR, timestamp);
        const bool gpsSpoofed = gpsSpoofedSensor ? gpsSpoofedSensor->ConvertTo<ui64>(0) > 1 : false;

        bool restricted = false;
        for (auto&& area : Options.RestrictedAreas) {
            if (area.IsPointInternal(coordinate)) {
                restricted = true;
                break;
            }
        }

        if (i->IsZero() || gpsJammed || gpsSpoofed || restricted) {
            NOTICE_LOG << imei << ": filter out compromized "
                << gpsJammed << '/'
                << gpsSpoofed << '/'
                << restricted << ' '
                << "location " << i->ToJson().GetStringRobust() << Endl;
            i = locations.erase(i);
        } else {
            ++i;
        }
    }
    if (locations.empty()) {
        return {};
    }
    if (!Options.EnableClusterization) {
        return locations.back();
    }
    if (DynamicSettings) {
        auto disableClusterization = DynamicSettings->Get<bool>("disable_clusterization").GetOrElse(false);
        if (disableClusterization) {
            return locations.back();
        }
    }
    if (DynamicSettings) {
        auto disabledImeis = DynamicSettings->Get<TString>("disable_clusterization_imei").GetOrElse({});
        if (disabledImeis.Contains(imei)) {
            NOTICE_LOG << imei << ": clusterization disabled by 'disable_clusterization_imei' dynamic setting" << Endl;
            return locations.back();
        }
    }

    auto distance = [](const TLocation& first, const TLocation& second) {
        return first.GetCoord().GetLengthTo(second.GetCoord());
    };
    auto speed = [distance, minimalTime = Options.MinimalTime](const TLocation& first, const TLocation& second) {
        double length = distance(first, second);
        double time = std::max<double>(minimalTime, std::abs(first.Timestamp.SecondsFloat() - second.Since.SecondsFloat()));
        return length / time;
    };
    auto clusters = Clusterize(locations, Options.LengthThreshold, distance);
    auto transposed = Transpose(clusters);
    auto largest = GetLargestCluster(transposed);

    TVector<bool> banned(transposed.size(), false);
    size_t resultIdx = transposed[largest].front();
    for (size_t i = resultIdx + 1; i < locations.size(); ++i) {
        CHECK_WITH_LOG(i < clusters.size());
        auto cluster = clusters[i];
        if (banned[cluster]) {
            continue;
        }

        auto d = speed(locations[resultIdx], locations[i]);
        if (d <= Options.SpeedThreshold) {
            resultIdx = i;
        } else {
            banned[cluster] = true;
        }
    }
    return locations[resultIdx];
}

NThreading::TFuture<NDrive::TLocation> NDrive::TLocator::GetGSM(const TString& imei, const TSensorsCache& sensors) const {
    CHECK_WITH_LOG(LBSClient);
    NThreading::TFuture<TLBSClient::TResponse> lbs;
    TInstant start;
    TInstant timestamp;

    const auto stations = sensors.Get(VEGA_EXT_SERVING_CELL_INF);
    if (stations && !stations->IsZero()) {
        timestamp = stations->Timestamp;

        const auto stationsString = stations->ConvertTo<TString>();
        const auto cells = ParseCells(stationsString);
        const auto age = Now() - timestamp;
        if (!cells.empty()) {
            TLBSClient::TQuery query;
            for (auto&& cell : cells) {
                query.Cells.emplace_back(cell, age);
            }
            start = Now();
            lbs = LBSClient->Locate(query);
        } else if (stationsString) {
            WARNING_LOG << imei << ": bad VEGA_EXT_SERVING_CELL_INF string " << stationsString << Endl;
        }
    }

    const auto cellid = sensors.Get(VEGA_CELLID);
    if (cellid && !cellid->IsZero()) {
        timestamp = cellid->Timestamp;

        const auto mcc = sensors.Get(VEGA_MCC, timestamp);
        const auto mnc = sensors.Get(VEGA_MNC, timestamp);
        const auto lac = sensors.Get(VEGA_LAC, timestamp);
        const auto level = sensors.Get(VEGA_GSM_SIGNAL_LEVEL, timestamp);
        const auto age = Now() - timestamp;
        if (mcc && mnc && lac && level) {
            TLBSClient::TCell cell;
            cell.MCC = mcc->ConvertTo<ui64>();
            cell.MNC = mnc->ConvertTo<ui64>();
            cell.LAC = lac->ConvertTo<ui64>();
            cell.CellId = cellid->ConvertTo<ui64>();
            cell.SignalLevel = level->ConvertTo<double>();
            cell.Age = age;

            TLBSClient::TQuery query;
            query.Cells.push_back(cell);
            start = Now();
            lbs = LBSClient->Locate(query);
        } else {
            WARNING_LOG << imei << ": incomplete GSM cell info " << !mcc.Empty() << !mnc.Empty() << !lac.Empty() << !level.Empty() << Endl;
        }
    }

    if (lbs.Initialized()) {
        TTelematicsUnistatSignals::Get().LBSRequests.Signal(1);
        return lbs.Apply([imei, start, timestamp](const NThreading::TFuture<TLBSClient::TResponse>& r) -> TLocation {
            const auto& response = r.GetValue();
            const TInstant finish = Now();
            const TDuration duration = finish - start;
            TTelematicsUnistatSignals::Get().LBSTimes.Signal(duration.MilliSeconds());
            if (auto position = response.Position.Get()) {
                TLocation location = {
                    position->Latitude,
                    position->Longitude,
                    position->Precision,
                    /*course=*/0,
                    NDrive::TLocation::LBS,
                    timestamp
                };
                location.Name = LBSLocationName;
                return location;
            } else {
                if (response.Error) {
                    ERROR_LOG << imei << ": LBS response error: " << response.Error << Endl;
                }
                return {};
            }
        });
    } else {
        WARNING_LOG << imei << ": cannot invoke LBS" << Endl;
        return NThreading::MakeFuture(TLocation());
    }
}

NDrive::TLocation NDrive::TLocator::Locate(const TString& imei, const TLocation& dirty, const TLocation& validated, const TLocation& lbs) const {
    INFO_LOG << imei << ": selecting coordinates"
        << " dirty=" << dirty.ToJson().GetStringRobust()
        << " validated=" << validated.ToJson().GetStringRobust()
        << " lbs=" << lbs.ToJson().GetStringRobust()
        << Endl;
    if (!dirty.IsZero()) {
        if (validated.IsZero() || validated.Cross(dirty) || validated.Type == TLocation::LBS) {
            INFO_LOG << imei << ": selecting dirty coordinate " << dirty.ToJson().GetStringRobust() << Endl;
            return dirty;
        }
    }
    if (!validated.IsZero()) {
        {
            INFO_LOG << imei << ": selecting validated coordinate " << validated.ToJson().GetStringRobust() << Endl;
            return validated;
        }
    }
    if (!lbs.IsZero()) {
        INFO_LOG << imei << ": selecting LBS coordinate " << lbs.ToJson().GetStringRobust() << Endl;
        return lbs;
    }
    INFO_LOG << imei << ": cannot select coordinate " << Endl;
    return {};
}

NDrive::TLocator::TAsyncLocation NDrive::TLocator::Beacon(const TString& imei, const TSensorsCache& sensors) const {
    INFO_LOG << imei << ": creating location from beacon sensors" << Endl;
    if (!BeaconRecognizer) {
        INFO_LOG << imei << ": can't create location; no BeaconRecognizer available" << Endl;
        return {};
    }
    TMaybe<NVega::TBeaconInfo> beaconInfoBestRSSI;
    TInstant now = TInstant::Now();
    TInstant freshnessThreshold = now - TDuration::Seconds(600);
    NVega::TBeaconInfos beaconInfos;

    for (int id = BLE_EXT_BOARD_BEACONS_INFO1; id <= BLE_EXT_BOARD_BEACONS_INFO10; ++id) {
        auto maybeSensor = sensors.Get(id, now);
        if (maybeSensor && maybeSensor->Timestamp >= freshnessThreshold) {
            const auto& buffer = std::get<TBuffer>(maybeSensor->Value);
            TMemoryInput input(buffer.data(), buffer.size());
            beaconInfos.Load(&input);
            for(const auto& beaconInfo: beaconInfos.Elements) {
                if (!beaconInfoBestRSSI || beaconInfoBestRSSI->GetRSSI() < beaconInfo.GetRSSI()) {
                    if (BeaconRecognizer->CanRecognizeLocation(beaconInfo)) {
                        beaconInfoBestRSSI = beaconInfo;
                    }
                }
            }
        }
    }

    if (beaconInfoBestRSSI) {
        INFO_LOG << imei << ": beacon with the best RSSI is " << NDrive::IBeaconRecognizer::ConvertToKey(*beaconInfoBestRSSI) << Endl;
        auto maybeLocationData = BeaconRecognizer->TryRecognizeLocation(*beaconInfoBestRSSI);
        INFO_LOG << imei << ": recognized location data is: " << maybeLocationData->ToJson().GetStringRobust() << Endl;
        TLocation location(maybeLocationData->Latitude, maybeLocationData->Longitude, 0, 0, TLocation::EType::Beacon, now, now);
        location.Name = BeaconsLocationName;
        location.Content = std::move(maybeLocationData->Name);
        return NThreading::MakeFuture(location);
    } else {
        INFO_LOG << imei << ": can't create location; BeaconRecognizer can't recognize beacons" << Endl;
        return {};
    }
}

double NDrive::TLocator::GetPrecisionFromHDOP(double HDOP) const {
    constexpr double coef = 4;
    return coef * HDOP;
}

bool NDrive::TLocator::IsStationary(const TSensorsCache& sensors) const {
    auto engine = sensors.Get(CAN_ENGINE_IS_ON);
    auto speed = sensors.Get(VEGA_SPEED);
    return (!engine || !engine->ConvertTo<bool>(false)) && (!speed || speed->ConvertTo<double>(0) < 0.001);
}

void NDrive::TLocator::SetBeaconRecognizer(THolder<IBeaconRecognizer> beaconRecognizer) {
    BeaconRecognizer = std::move(beaconRecognizer);
}

void NDrive::TLocator::SetDynamicSettings(TAtomicSharedPtr<NDrive::TTelematicsDynamicSettings> dynamicSettings) {
    DynamicSettings = std::move(dynamicSettings);
}
