#pragma once

#include "location.h"

#include <library/cpp/threading/future/fwd.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <rtline/library/geometry/polyline.h>

#include <functional>

namespace NGraph {
    class TRouter;
}

namespace NRTLine {
    class TNehSearchClient;
}

namespace NDrive {
    namespace NVega {
        class TBlackboxRecords;
    }

    struct TSensor;
    class ISensorApi;
    class TSensorsCache;
    class TLocationCache;
    class TLBSClient;
    class TGeocoder;
    class ITrackClient;
    class IBeaconRecognizer;
    class TTelematicsDynamicSettings;

    class TLocator {
    public:
        struct TOptions {
            TString LBSToken;

            TString LinkerHost;
            ui16 LinkerPort = 17000;
            TString LinkerService = "default-linker";

            TString TracksHost;
            ui16 TracksPort = 17000;
            TString TracksService = "drive_graph";

            TString SensorHost;
            ui16 SensorPort = 17000;
            TString SensorService = "drive_cache";
            TDuration SensorTimeout = TDuration::MilliSeconds(100);
            TVector<TDuration> BalancerTimeoutTable = {
                TDuration::MilliSeconds(10),
                TDuration::MilliSeconds(30),
                TDuration::MilliSeconds(50),
                TDuration::MilliSeconds(150)
            };
            TString SensorSpMetaSearch;
            ui32 SensorSaasTvmId = 0;
            TString SensorAPIName = "sensors";

            TString GeocoderHost;
            ui16 GeocoderPort = 80;
            TString GeocoderPath;
            ui32 GeocoderClientId = 0;
            TDuration GeocoderRefreshInterval = TDuration::Seconds(300);

            TVector<TGeoPolyLine> RestrictedAreas;
            TDuration FutureThreshold = TDuration::Days(1);
            double DefaultHDOP = 10;
            double PrecisionThreshold = 2000;
            double LengthThreshold = 1000;
            double SpeedThreshold = 200.0 * 1000 / 3600;
            double MinimalTime = 5;
            bool EnableClusterization = true;
            bool EnableLocateFromLbs = false;

            TDuration LinkLinkerTimeout = TDuration::MilliSeconds(150);
            TDuration LinkTrackTimeout = TDuration::MilliSeconds(500);
            TDuration LinkRealtimeThreshold = TDuration::Minutes(15);
            double LinkCoordinateShift = 3;
            double LinkDistancePrecision = 20;
            double LinkFCWeight = 2;
            double LinkTailLength = 250;
            double LinkTailLengthLimit = 1500;
            double LinkDeviationLengthThreshold = 50;
            double LinkFilterByRandomFraction = 0;
            bool LinkEnableProjection = false;
            bool LinkFilterByEngine = true;
            bool LinkFilterBySpeed = true;
            bool LinkQueryByTimestamp = false;

            ui32 NumOfReviewedSensorValues = 16;
        };

        using TFunction = std::function<void(const TLocation&)>;
        using TAsyncLocation = NThreading::TFuture<TLocation>;
        using TAsyncLocations = TVector<TAsyncLocation>;

    public:
        TLocator(const TOptions& options = Default<TOptions>(), TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm = nullptr);
        ~TLocator();

        TAtomicSharedPtr<NRTLine::TNehSearchClient> GetSensorSearchClient() const {
            return SensorClient;
        }
        TAtomicSharedPtr<ISensorApi> GetSensors() const {
            return SensorApi;
        }

        TAsyncLocation Geocode(const TString& imei, const TAsyncLocation& location) const;

        TAsyncLocations LocateAll(const TString& imei, const TSensorsCache& sensors, const TLocationCache* cache = nullptr) const;
        TAsyncLocation Locate(const TString& imei, const TSensorsCache& sensors) const;
        TAsyncLocation Locate(const TString& imei, const TSensorsCache& sensors, TLocations gpss) const;
        TAsyncLocation Link(const TString& imei, const TSensorsCache& sensors) const;
        TAsyncLocation Link(const TString& imei, const TSensorsCache& sensors, const TLocations& gpss) const;
        TAsyncLocation LBS(const TString& imei, const TSensorsCache& sensors) const;
        TAsyncLocation LBS(const TString& imei, const TSensorsCache& sensors, const TLocations& gpss) const;
        TAsyncLocation Beacon(const TString& imei, const TSensorsCache& sensors) const;

        void SetBeaconRecognizer(THolder<IBeaconRecognizer> beaconRecognizer);
        void SetDynamicSettings(TAtomicSharedPtr<NDrive::TTelematicsDynamicSettings> dynamicSettings);

    private:
        TLocations GetGPS(const TSensorsCache& sensors) const;
        TLocation GetGPS(const TSensorsCache& sensors, const TSensor& latitude, const TSensor& longitude, const TSensor& course, size_t index) const;
        TLocation GetGPSNonZero(const TLocations& locations) const;
        TLocation GetGPSValidated(const TString& imei, const TSensorsCache& sensors, TLocations locations) const;
        TAsyncLocation GetGSM(const TString& imei, const TSensorsCache& sensors) const;

        TAsyncLocation Link(const TString& imei, const TLocation& dirty, double tailLength) const;
        TLocation Locate(const TString& imei, const TLocation& dirty, const TLocation& validated, const TLocation& lbs) const;
        double GetPrecisionFromHDOP(double HDOP) const;
        bool IsStationary(const TSensorsCache& sensors) const;

    private:
        TOptions Options;

        THolder<TLBSClient> LBSClient;
        TAtomicSharedPtr<NRTLine::TNehSearchClient> SensorClient;
        TAtomicSharedPtr<NDrive::TGeocoder> Geocoder;
        TAtomicSharedPtr<ISensorApi> SensorApi;
        TAtomicSharedPtr<NGraph::TRouter> LinkerClient;
        TAtomicSharedPtr<NDrive::ITrackClient> TracksClient;
        TAtomicSharedPtr<NDrive::TTelematicsDynamicSettings> DynamicSettings;
        THolder<IBeaconRecognizer> BeaconRecognizer;
    };
}
