#include "yagr_pusher.h"

#include "location.h"

#include <drive/telematics/server/common/signals.h>
#include <drive/telematics/server/pusher/common.h>

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/string_utils/url/url.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/network/neh.h>

#include <util/string/vector.h>

THolder<NDrive::IPusher> NDrive::TYAGRPusherOptions::BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const {
    return MakeHolder<TYAGRPusher>(*this, tvm);
}

void NDrive::TYAGRPusherOptions::Init(const TYandexConfig::Section& section) {
    const auto& directives = section.GetDirectives();
    Endpoint = directives.Value("Endpoint", Endpoint);
    DestinationClientId = directives.Value("DestinationClientId", DestinationClientId);
}

TSet<NTvmAuth::TTvmId> NDrive::TYAGRPusherOptions::GetDestinationClientIds() const {
  return {DestinationClientId};
}

void NDrive::TYAGRPusherOptions::Print(IOutputStream& os) const {
    os << "Endpoint: " << Endpoint << Endl;
    os << "DestinationClientId: " << DestinationClientId << Endl;
}

NDrive::TYAGRPusher::TYAGRPusher(const NDrive::TYAGRPusherOptions& options, const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm)
    : Client(MakeHolder<NNeh::THttpClient>(options.Endpoint))
    , TvmAuth(options.DestinationClientId ? MakeHolder<NDrive::TTvmAuth>(tvm, options.DestinationClientId) : nullptr)
    , Handler(NUrl::SplitUrlToHostAndPath(options.Endpoint).path)
{
}

NDrive::TYAGRPusher::~TYAGRPusher() = default;

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TYAGRPusher::Push(const TString& imei, const IHandlerDescription& handler, TInstant deadline) {
    Y_UNUSED(deadline);
    Y_UNUSED(handler);
    Y_UNUSED(imei);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TYAGRPusher::Push(const TString& imei, const THeartbeat& heartbeat, TInstant deadline) {
    Y_UNUSED(deadline);
    Y_UNUSED(heartbeat);
    Y_UNUSED(imei);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TYAGRPusher::Push(const TString& imei, const TLocation& location, TInstant deadline) {
    Y_UNUSED(deadline);
    constexpr auto positionSource = "Verified"; // NOTE: No need to convert from location.type

    auto request = CreateRequest();
    {
        NJson::TJsonValue post;
        {
          NJson::TJsonValue contractor;
          contractor["dbid"] = imei;
          contractor["uuid"] = imei; // TODO: get uuid
          post["contractor_id"] = contractor;
        }
        {
          NJson::TJsonValue positions{NJson::EJsonValueType::JSON_ARRAY};
          {
            NJson::TJsonValue position;
            position["source"] = positionSource;
            position["lat"] = location.Latitude;
            position["lon"] = location.Longitude;
            position["unix_timestamp"] = location.Timestamp.MilliSeconds();
            position["direction"] = static_cast<int>(location.Course);
            position["accuracy"] = location.Precision;
            positions.AppendValue(position);
          }

          post["positions"] = positions;
        }
        request.AddHeader("Content-Type", "application/json");
        request.SetPostData(post.GetStringRobust());
    }

    auto asyncResponse = Yensured(Client)->SendAsync(request);
    return ProcessResponse(asyncResponse);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TYAGRPusher::Push(const TString& imei, const TSensor& sensor, TInstant deadline) {
    Y_UNUSED(imei);
    Y_UNUSED(sensor);
    Y_UNUSED(deadline);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NNeh::THttpRequest NDrive::TYAGRPusher::CreateRequest() const {
    NNeh::THttpRequest request;
    request.SetUri(Handler);
    if (TvmAuth) {
        TvmAuth->UpdateRequest(request);
    }
    return request;
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TYAGRPusher::ProcessResponse(const NThreading::TFuture<NNeh::THttpReply>& reply) const {
    return reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        if (r.HasException()) {
            NDrive::TTelematicsUnistatSignals::Get().YAGRPusherException.Signal(1);
            NDrive::TTelematicsUnistatSignals::Get().YAGRPusherFailure.Signal(1);
            return TPushResult{ /* Written= */ false, /* Message= */ NThreading::GetExceptionMessage(r) };
        }
        const auto& response = r.GetValue();
        if (response.IsSuccessReply()) {
            NDrive::TTelematicsUnistatSignals::Get().YAGRPusherSuccess.Signal(1);
            return TPushResult{ /* Written= */ true, /* Message= */ "" };
        } else {
            NDrive::TTelematicsUnistatSignals::Get().YAGRPusherFailure.Signal(1);
            return TPushResult{ /* Written= */ false, /* Message= */ response.Serialize().GetStringRobust()};
        }
    });
}
