#include "client.h"

#include <drive/telematics/server/pusher/common.h>
#include <drive/telematics/server/pusher/pusher.h>

#include <rtline/library/json/adapters.h>

const TString HandlerKeyPrefixString = ToString(NDrive::TPusher::HandlerKeyPrefix);

NThreading::TFuture<TMaybe<NDrive::TTelematicsMetadataClient::THandler>> NDrive::TTelematicsMetadataClient::GetHandler(const TString& id) const {
    auto query = MakeQuery();
    auto replies = Query({ GetHandlerUrl(id) }, std::move(query));
    auto waiter = NThreading::WaitExceptionOrAll(replies);
    return waiter.Apply([this, id, replies = std::move(replies)](const NThreading::TFuture<void>& /*w*/) -> TMaybe<THandler> {
        THandlers handlers;
        for (auto&& r : replies) {
            FillHandlers(r.GetValue(), handlers);
        }
        std::sort(handlers.begin(), handlers.end(), [](const THandler& left, const THandler& right) {
            return left.Timestamp > right.Timestamp;
        });
        for (auto&& handler : handlers) {
            if (handler.Id == id) {
                return std::move(handler);
            }
        }
        return {};
    });
}

NThreading::TFuture<NDrive::TTelematicsMetadataClient::THandlers> NDrive::TTelematicsMetadataClient::GetActiveHandlers(const TString& imei) const {
    auto query = MakeQuery();
    query.AddProperty("Id");
    query.AddProperty("Data");
    query.AddProperty("Finished");
    query.AddProperty("Timestamp");
    auto reply = HeavyQuery(imei, "active_imei", std::move(query));
    return reply.Apply([this](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        THandlers result;
        FillHandlers(r.GetValue(), result);
        return result;
    });
}

void NDrive::TTelematicsMetadataClient::FillHandlers(const NRTLine::TSearchReply& reply, THandlers& handlers) const {
    Y_ENSURE(reply.IsSucceeded(), "handlers fetch has been unsuccessful: " << reply.GetReqId() << ' ' << reply.GetCode());
    reply.ScanDocs([&handlers](const NMetaProtocol::TDocument& document) {
        THandler handler;
        try {
            for (auto&& propertie : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
                const auto& key = propertie.GetKey();
                const auto& value = propertie.GetValue();
                if (handler.Id.empty() && key == "Id") {
                    handler.Id = value;
                    continue;
                }
                if (handler.IMEI.empty() && (key == "IMEI" || key == "_SK_imei" || key == "_SK_active_imei")) {
                    handler.IMEI = value;
                    continue;
                }
                if (handler.Timestamp == TInstant::Zero() && key == "Timestamp") {
                    handler.Timestamp = TInstant::Seconds(FromString<ui64>(value));
                    continue;
                }
                if (!handler.Finished && key == "Finished") {
                    handler.Finished = IsTrue(value);
                    continue;
                }
                if (!handler.Data.IsDefined() && key == "Data") {
                    handler.Data = NJson::ToJson(NJson::JsonString(value));
                    continue;
                }
            }
            handlers.push_back(std::move(handler));
        } catch (const std::exception& e) {
            ERROR_LOG << "cannot parse handler " << handler.Id << " from " << document.DebugString() << Endl;
        }
    });
}

NRTLine::TQuery NDrive::TTelematicsMetadataClient::MakeQuery() const {
    auto query = NRTLine::TQuery();
    query.AddExtraParam("sgkps", HandlerKeyPrefixString);
    return query;
}
