#include "interface.h"

#include <drive/telematics/server/library/connection.h>

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/threading/future/future.h>
#include <library/cpp/threading/future/wait/wait.h>

#include <util/generic/ptr.h>

THolder<NDrive::IPusherOptions> NDrive::IPusherOptions::ConstructPusherOptions(const TYandexConfig::Section& section) {
    const auto& directives = section.GetDirectives();
    const auto type = directives.Value("Type", TString{});
    Y_ENSURE(type);
    THolder<IPusherOptions> options{TFactory::Construct(type)};
    AssertCorrectConfig(options.Get(), "Incorrect PusherOptions type: '%s'", type.data());
    options->Init(section);
    return options;
}

template <typename T>
NThreading::TFuture< NDrive::IPusher::TBulkPushResult>  NDrive::IPusher::BulkPushGeneric(const TString& imei, const TConstArrayRef<T> data, TInstant deadline) {
    if (data.empty()) {
        return NThreading::MakeFuture(TBulkPushResult{ /* Written= */ 0, /* Messages= */ {} });
    }
    NThreading::TFutures<NDrive::IPusher::TPushResult> pushResults;
    pushResults.reserve(data.size());
    for (const auto& item : data) {
        if constexpr (std::is_same_v<T, TTelematicsHandlerPtr>) {
            pushResults.push_back(Push(imei, *item, deadline));
        } else {
            pushResults.push_back(Push(imei, item, deadline));
        }
    }
    auto waiter = NThreading::WaitAll(pushResults);
    return waiter.Apply([pushResults = std::move(pushResults)](const NThreading::TFuture<void>&) mutable {
        TBulkPushResult result;
        result.Messages.reserve(pushResults.size());
        for (auto&& pushResultFuture : pushResults) {
            auto pushResult = pushResultFuture.ExtractValue();
            result.Written += pushResult.Written;
            result.Messages.push_back(std::move(pushResult.Message));
        }
        return result;
    });
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::IPusher::BulkPush(const TString& imei, const TConstArrayRef<TTelematicsHandlerPtr> handlersPtrs, TInstant deadline) {
    return BulkPushGeneric(imei, handlersPtrs, deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::IPusher::BulkPush(const TString& imei, const TConstArrayRef<THeartbeat> heartbeats, TInstant deadline) {
    return BulkPushGeneric(imei, heartbeats, deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::IPusher::BulkPush(const TString& imei, const TConstArrayRef<TLocation> locations, TInstant deadline) {
    return BulkPushGeneric(imei, locations, deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::IPusher::BulkPush(const TString& imei, const TConstArrayRef<TSensor> sensors, TInstant deadline) {
    return BulkPushGeneric(imei, sensors, deadline);
}
