#pragma once

#include <drive/telematics/server/location/heartbeat.h>

#include <drive/telematics/protocol/sensor.h>

#include <drive/library/cpp/auth/tvm.h>

#include <library/cpp/object_factory/object_factory.h>
#include <library/cpp/threading/future/fwd.h>
#include <library/cpp/yconf/conf.h>

#include <util/generic/ptr.h>

namespace NTvmAuth {
    class TTvmClient;
}

namespace NDrive {
    namespace NVega {
        class TBlackboxRecords;
    }

    struct TLocation;
    class ITelematicsHandler;
    using TTelematicsHandlerPtr = TIntrusivePtr<ITelematicsHandler>;

    class IHandlerDescription {
    public:
        using TSequenceId = ui32;

    public:
        virtual ~IHandlerDescription() = default;

        virtual TString GetId() const {
            ythrow yexception() << "unimplemented";
        }
        virtual TInstant GetTimestamp() const {
            ythrow yexception() << "unimplemented";
        }
        virtual bool IsFinished() const {
            ythrow yexception() << "unimplemented";
        }
        virtual NJson::TJsonValue Serialize() const {
            ythrow yexception() << "unimplemented";
        }
    };

    class IPusher {
    public:
        virtual ~IPusher() = default;

        struct TPushResult {
            bool Written = false;
            TString Message;
        };

        struct TBulkPushResult {
            size_t Written = 0;
            TVector<TString> Messages;
        };

        virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const IHandlerDescription& handler, TInstant deadline = TInstant::Zero()) = 0;
        virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const THeartbeat& heartbeat, TInstant deadline = TInstant::Zero()) = 0;
        virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const TLocation& location, TInstant deadline = TInstant::Zero()) = 0;
        virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const TSensor& sensor, TInstant deadline = TInstant::Zero()) = 0;

        virtual NThreading::TFuture<TBulkPushResult> BulkPush(const TString& imei, const TConstArrayRef<TTelematicsHandlerPtr> handlersPtrs, TInstant deadline = TInstant::Zero());
        virtual NThreading::TFuture<TBulkPushResult> BulkPush(const TString& imei, const TConstArrayRef<THeartbeat> heartbeats, TInstant deadline = TInstant::Zero());
        virtual NThreading::TFuture<TBulkPushResult> BulkPush(const TString& imei, const TConstArrayRef<TLocation> locations, TInstant deadline = TInstant::Zero());
        virtual NThreading::TFuture<TBulkPushResult> BulkPush(const TString& imei, const TConstArrayRef<TSensor> sensors, TInstant deadline = TInstant::Zero());

    private:
        template <typename T>
        NThreading::TFuture<TBulkPushResult> BulkPushGeneric(const TString& imei, const TConstArrayRef<T> data, TInstant deadline);
    };

    class IPusherOptions {
    protected:
        using TFactory = NObjectFactory::TObjectFactory<IPusherOptions, TString>;

    public:
        virtual ~IPusherOptions() {
        }
        virtual THolder<IPusher> BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const = 0;
        virtual void Print(IOutputStream& os) const = 0;
        virtual TSet<NTvmAuth::TTvmId> GetDestinationClientIds() const = 0;

    public:
        static THolder<IPusherOptions> ConstructPusherOptions(const TYandexConfig::Section& section);

    private:
        virtual void Init(const TYandexConfig::Section& section) = 0;
    };

    using TPusherPtr = TAtomicSharedPtr<IPusher>;
}
