#include "local.h"

#include <drive/telematics/server/location/location.h>

#include <drive/telematics/protocol/sensor.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/threading/future/future.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/container.h>

TLocalSensorStorage::TPtr TLocalSensorStorage::Instance() {
    struct TLocalSensorStorageInstance {
        TPtr Ptr;
        TLocalSensorStorageInstance()
            : Ptr(MakeAtomicShared<TLocalSensorStorage>())
        {
        }
    };
    return Singleton<TLocalSensorStorageInstance>()->Ptr;
}

void TLocalSensorStorage::Push(const TString& imei, const NDrive::IHandlerDescription& handler) {
    THandler current;
    current.IMEI = imei;
    current.Id = handler.GetId();
    current.Timestamp = handler.GetTimestamp();
    current.Finished = handler.IsFinished();
    current.Data = handler.Serialize();

    TWriteGuard guard(Lock);
    THandler previous = Handlers[imei][current.Id];
    if (current.Timestamp > previous.Timestamp) {
        Handlers[imei][current.Id] = current;
        INFO_LOG << imei << ": accepted handler " << current.Id << " " << current.Timestamp << Endl;
    }
}

void TLocalSensorStorage::Push(const TString& imei, const NDrive::THeartbeat& heartbeat) {
    TWriteGuard guard(Lock);
    Heartbeats[imei][heartbeat.Name] = heartbeat;
    INFO_LOG << imei << ": accepted heartbeat " << heartbeat.Timestamp << Endl;
}

void TLocalSensorStorage::Push(const TString& imei, const NDrive::TLocation& location) {
    TWriteGuard guard(Lock);
    Locations[imei][location.Name] = location;
    INFO_LOG << imei << ": accepted location " << location.ToJson().GetStringRobust() << Endl;
}

void TLocalSensorStorage::Push(const TString& imei, const NDrive::TSensor& sensor) {
    TWriteGuard guard(Lock);
    Sensors[imei][sensor] = sensor;
    INFO_LOG << imei << ": accepted sensor " << sensor.ToJson().GetStringRobust() << Endl;
}

TMaybe<TLocalSensorStorage::THandler> TLocalSensorStorage::GetHandler(const TString& id) const {
    TReadGuard guard(Lock);
    for (auto&&[imei, handlers] : Handlers) {
        if (auto handler = handlers.FindPtr(id)) {
            return *handler;
        }
    }
    return {};
}

TVector<TLocalSensorStorage::THandler> TLocalSensorStorage::GetHandlers(const TString& imei) const {
    TReadGuard guard(Lock);
    if (auto data = Handlers.FindPtr(imei)) {
        return MakeVector(NContainer::Values(*data));
    } else {
        return {};
    }
}

TMaybe<NDrive::THeartbeat> TLocalSensorStorage::GetHeartbeat(const TString& imei, TStringBuf name) const {
    TReadGuard guard(Lock);
    if (auto data = Heartbeats.FindPtr(imei)) {
        if (auto heartbeat = data->FindPtr(name)) {
            auto result = *heartbeat;
            INFO_LOG << imei << ": returned heartbeat " << result.Timestamp << Endl;
            return result;
        }
    }
    return {};
}

TMap<TString, NDrive::THeartbeat> TLocalSensorStorage::GetHeartbeats(TStringBuf name) const {
    TReadGuard guard(Lock);
    TMap<TString, NDrive::THeartbeat> result;
    for (auto&&[imei, data] : Heartbeats) {
        if (auto heartbeat = data.FindPtr(name)) {
            result.emplace(imei, *heartbeat);
        }
    }
    INFO_LOG << "returned " << result.size() << " " << name << " heartbeats" << Endl;
    return result;
}

TMaybe<NDrive::TLocation> TLocalSensorStorage::GetLocation(const TString& imei, TStringBuf name) const {
    TReadGuard guard(Lock);
    if (auto data = Locations.FindPtr(imei)) {
        if (auto location = data->FindPtr(name)) {
            auto result = *location;
            INFO_LOG << imei << ": returned location " << result.ToJson().GetStringRobust() << Endl;
            return result;
        }
    }
    return {};
}

TMap<TString, NDrive::TLocation> TLocalSensorStorage::GetLocations(TStringBuf name) const {
    TReadGuard guard(Lock);
    TMap<TString, NDrive::TLocation> result;
    for (auto&&[imei, data] : Locations) {
        if (auto location = data.FindPtr(name)) {
            result.emplace(imei, *location);
        }
    }
    INFO_LOG << "returned " << result.size() << " " << name << " locations" << Endl;
    return result;
}

TMaybe<NDrive::TSensor> TLocalSensorStorage::GetSensor(const TString& imei, ui16 id, ui16 subid) const {
    TReadGuard guard(Lock);
    NDrive::TSensorId sensorId(id, subid);
    if (auto data = Sensors.FindPtr(imei)) {
        if (auto s = data->FindPtr(sensorId)) {
            auto result = *s;
            INFO_LOG << imei << ": returned sensor " << result.ToJson().GetStringRobust() << Endl;
            return result;
        }
    }
    return {};
}

TMap<TString, NDrive::TSensor> TLocalSensorStorage::GetSensor(NDrive::TSensorId sensor) const {
    TReadGuard guard(Lock);
    TMap<TString, NDrive::TSensor> result;
    for (auto&&[imei, data] : Sensors) {
        if (auto s = data.FindPtr(sensor)) {
            result.emplace(imei, *s);
        }
    }
    INFO_LOG << "returned " << result.size() << " " << sensor << " sensors" << Endl;
    return result;
}

TMaybe<TVector<NDrive::TSensor>> TLocalSensorStorage::GetSensors(const TString& imei) const {
    TReadGuard guard(Lock);
    if (auto data = Sensors.FindPtr(imei)) {
        return MakeVector(NContainer::Values(*data));
    }
    return {};
}

NJson::TJsonValue TLocalSensorStorage::ToJson() const {
    NJson::TJsonValue result;
    result["heartbeats"] = NJson::ToJson(Heartbeats);
    result["locations"] = NJson::ToJson(Locations);
    result["sensors"] = NJson::ToJson(Sensors);
    return result;
}

bool TLocalSensorStorage::TryFromJson(const NJson::TJsonValue& value) {
    if (!NJson::TryFromJson(value["heartbeats"], Heartbeats)) {
        // compatibility
        TMap<TString, NDrive::THeartbeat> heartbeats;
        if (!NJson::TryFromJson(value["heartbeats"], heartbeats)) {
            return false;
        }
        for (auto&&[imei, heartbeat] : heartbeats) {
            Heartbeats[imei][heartbeat.Name] = heartbeat;
        }
    }
    return
        NJson::TryFromJson(value["locations"], Locations) &&
        NJson::TryFromJson(value["sensors"], Sensors);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> TLocalPusher::Push(const TString& imei, const NDrive::IHandlerDescription& handler, TInstant deadline) {
    Y_UNUSED(deadline);
    CHECK_WITH_LOG(LocalSensorStorage);
    LocalSensorStorage->Push(imei, handler);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> TLocalPusher::Push(const TString& imei, const NDrive::THeartbeat& heartbeat, TInstant deadline) {
    Y_UNUSED(deadline);
    CHECK_WITH_LOG(LocalSensorStorage);
    LocalSensorStorage->Push(imei, heartbeat);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> TLocalPusher::Push(const TString& imei, const NDrive::TLocation& location, TInstant deadline) {
    Y_UNUSED(deadline);
    CHECK_WITH_LOG(LocalSensorStorage);
    LocalSensorStorage->Push(imei, location);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> TLocalPusher::Push(const TString& imei, const NDrive::TSensor& sensor, TInstant deadline) {
    Y_UNUSED(deadline);
    CHECK_WITH_LOG(LocalSensorStorage);
    LocalSensorStorage->Push(imei, sensor);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

TAtomicSharedPtr<TLocalSensorStorage> TLocalPusher::GetLocalSensorStorage() {
    return LocalSensorStorage;
};

THolder<NDrive::IPusher> TLocalPusherOptions::BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& /*tvm*/) const {
    return MakeHolder<TLocalPusher>();
}

void TLocalPusherOptions::Init(const TYandexConfig::Section& /* section */) {
}

void TLocalPusherOptions::Print(IOutputStream& /* os */) const {
}

TSet<NTvmAuth::TTvmId> TLocalPusherOptions::GetDestinationClientIds() const {
  return {};
}
