#pragma once

#include "interface.h"

#include <drive/telematics/server/location/location.h>

#include <util/generic/map.h>
#include <util/system/rwlock.h>

class TLocalSensorStorage {
public:
    struct THandler {
        NJson::TJsonValue Data;
        TString IMEI;
        TString Id;
        TInstant Timestamp;
        bool Finished = false;
    };

    using TPtr = TAtomicSharedPtr<TLocalSensorStorage>;
    using TConstPtr = TAtomicSharedPtr<const TLocalSensorStorage>;

public:
    TLocalSensorStorage() = default;

    static TPtr Instance();

    void Push(const TString& imei, const NDrive::IHandlerDescription& handler);
    void Push(const TString& imei, const NDrive::THeartbeat& heartbeat);
    void Push(const TString& imei, const NDrive::TLocation& location);
    void Push(const TString& imei, const NDrive::TSensor& sensor);

    TMaybe<THandler> GetHandler(const TString& id) const;
    TVector<THandler> GetHandlers(const TString& imei) const;
    TMaybe<NDrive::THeartbeat> GetHeartbeat(const TString& imei, TStringBuf name) const;
    TMap<TString, NDrive::THeartbeat> GetHeartbeats(TStringBuf name) const;
    TMaybe<NDrive::TLocation> GetLocation(const TString& imei, TStringBuf name) const;
    TMap<TString, NDrive::TLocation> GetLocations(TStringBuf name) const;
    TMaybe<NDrive::TSensor> GetSensor(const TString& imei, ui16 id, ui16 subid) const;
    TMap<TString, NDrive::TSensor> GetSensor(NDrive::TSensorId sensor) const;
    TMaybe<TVector<NDrive::TSensor>> GetSensors(const TString& imei) const;

    NJson::TJsonValue ToJson() const;
    bool TryFromJson(const NJson::TJsonValue& value);

private:
    TMap<TString, TMap<TString, THandler>> Handlers;
    TMap<TString, TMap<TString, NDrive::THeartbeat>> Heartbeats;
    TMap<TString, TMap<TString, NDrive::TLocation>> Locations;
    TMap<TString, TMap<NDrive::TSensorId, NDrive::TSensor>> Sensors;
    TRWMutex Lock;
};

class TLocalPusher: public NDrive::IPusher {
public:
    TLocalPusher(TLocalSensorStorage::TPtr localSensorStorage)
        : LocalSensorStorage(localSensorStorage)
    {
    }
    TLocalPusher()
        : TLocalPusher(TLocalSensorStorage::Instance())
    {
    }

    virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const NDrive::IHandlerDescription& handler, TInstant deadline = TInstant::Zero()) override;
    virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const NDrive::THeartbeat& heartbeat, TInstant deadline = TInstant::Zero()) override;
    virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const NDrive::TLocation& location, TInstant deadline = TInstant::Zero()) override;
    virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const NDrive::TSensor& sensor, TInstant deadline = TInstant::Zero()) override;

    TLocalSensorStorage::TPtr GetLocalSensorStorage();

private:
    TLocalSensorStorage::TPtr LocalSensorStorage;
};

class TLocalPusherOptions : public NDrive::IPusherOptions {
public:
    virtual THolder<NDrive::IPusher> BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const override;

private:
    virtual void Init(const TYandexConfig::Section& section) override;
    virtual void Print(IOutputStream& os) const override;
    virtual TSet<NTvmAuth::TTvmId> GetDestinationClientIds() const override;

private:
    static inline auto Registrator = TFactory::TRegistrator<TLocalPusherOptions>("Local");
};
