#include "meta.h"

#include <drive/library/cpp/threading/future.h>


THolder<NDrive::IPusher> NDrive::TMetaPusherOptions::BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const {
    return MakeHolder<TMetaPusher>(*this, tvm);
}

void NDrive::TMetaPusherOptions::Init(const TYandexConfig::Section& /* section */) {
}

void NDrive::TMetaPusherOptions::Print(IOutputStream& /* os */) const {
}

TSet<NTvmAuth::TTvmId> NDrive::TMetaPusherOptions::GetDestinationClientIds() const {
  TSet<NTvmAuth::TTvmId> merged;
  for (auto& options: PushersOptions) {
    const auto ids = options->GetDestinationClientIds();
    merged.insert(ids.begin(), ids.end());
  }
  return merged;
}

NDrive::TMetaPusher::TMetaPusher(const NDrive::TMetaPusherOptions& options, const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) {
    for(const auto& pusherOptions : options.GetPushersOptions()) {
        AddPusher(pusherOptions->BuildPusher(tvm));
    }
}

NDrive::TMetaPusher& NDrive::TMetaPusher::AddPusher(THolder<IPusher>&& pusher) {
    Pushers.push_back(std::move(pusher));
    return *this;
}

template <typename TValue>
NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TMetaPusher::PushGeneric(const TString& imei, const TValue& value, TInstant deadline) {
    TVector<NThreading::TFuture<TPushResult>> waiters;
    for (const auto& pusher : Pushers) {
        try {
            waiters.push_back(pusher->Push(imei, value, deadline));
        } catch (const std::exception& e) {
            ERROR_LOG << "TMetaPusher: can't push; " << FormatExc(e) << Endl;
        }
    }
    auto waitF = NThreading::WaitAll(waiters);
    return waitF.Apply([futures = std::move(waiters)](const NThreading::TFuture<void>& /* w */) {
        for (const auto& f : futures) {
            if (f.HasException()) {
                return TPushResult{/* Written= */ false, /* Message= */ NThreading::GetExceptionMessage(f) };
            }
            auto pushResult = f.GetValue();
            if (!pushResult.Written) {
                return TPushResult{/* Written= */ false, /* Message= */ pushResult.Message };
            }
        }
        return TPushResult{/* Written= */ true, /* Message= */ "" };
    });
}

template <typename TValue>
NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::TMetaPusher::BulkPushGeneric(const TString& imei, const TConstArrayRef<TValue> values, TInstant deadline) {
    if (values.empty()) {
        return NThreading::MakeFuture(TBulkPushResult{ /* Written= */ 0, /* Messages= */ {} });
    }
    TVector<NThreading::TFuture<TBulkPushResult>> waiters;
    waiters.reserve(Pushers.size());
    for (const auto& pusher : Pushers) {
        try {
            waiters.push_back(pusher->BulkPush(imei, values, deadline));
        } catch (const std::exception& e) {
            ERROR_LOG << "TMetaPusher: can't bulk push; " << FormatExc(e) << Endl;
        }
    }
    auto waitF = NThreading::WaitAll(waiters);
    return waitF.Apply([futures = std::move(waiters)](const NThreading::TFuture<void>& /* w */) {
        TBulkPushResult result;
        result.Messages.reserve(futures.size());
        for (const auto& f : futures) {
            if (f.HasException()) {
                result.Messages.push_back(NThreading::GetExceptionMessage(f));
                continue;
            }
            auto pushResult = f.GetValue();
            result.Written += pushResult.Written;
            std::move(pushResult.Messages.begin(), pushResult.Messages.end(), std::back_inserter(result.Messages));
        }
        return result;
    });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TMetaPusher::Push(const TString& imei, const IHandlerDescription& handler, TInstant deadline) {
    return PushGeneric(imei, handler, deadline);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TMetaPusher::Push(const TString& imei, const THeartbeat& heartbeat, TInstant deadline) {
    return PushGeneric(imei, heartbeat, deadline);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TMetaPusher::Push(const TString& imei, const TLocation& location, TInstant deadline) {
    return PushGeneric(imei, location, deadline);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TMetaPusher::Push(const TString& imei, const TSensor& sensor, TInstant deadline) {
    return PushGeneric(imei, sensor, deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::TMetaPusher::BulkPush(const TString& imei, const TConstArrayRef<TTelematicsHandlerPtr> handlersPtrs, TInstant deadline) {
    return BulkPushGeneric(imei, handlersPtrs, deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::TMetaPusher::BulkPush(const TString& imei, const TConstArrayRef<THeartbeat> heartbeats, TInstant deadline) {
    return BulkPushGeneric(imei, heartbeats, deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::TMetaPusher::BulkPush(const TString& imei, const TConstArrayRef<TLocation> locations, TInstant deadline) {
    return BulkPushGeneric(imei, locations, deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::TMetaPusher::BulkPush(const TString& imei, const TConstArrayRef<TSensor> sensors, TInstant deadline) {
    return BulkPushGeneric(imei, sensors, deadline);
}
