#include "pusher.h"

#include "common.h"

#include <drive/telematics/server/location/location.h>

#include <drive/telematics/protocol/vega.h>

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/mediator/global_notifications/system_status.h>

#include <rtline/api/indexing_client/neh_client.h>

#include <util/generic/adaptor.h>
#include <util/string/builder.h>
#include <util/string/cast.h>


void NDrive::TPusherOptions::Init(const TYandexConfig::Section& section) {
    const TYandexConfig::Directives& directives = section.GetDirectives();
    Host = directives.Value("Host", Host);
    AssertCorrectConfig(!Host.empty(), "host mustn't be empty");
    Port = directives.Value("Port", Port);
    Token = directives.Value("Token", Token);
    Timeout = TDuration::MilliSeconds(directives.Value("Timeout", Timeout.MilliSeconds()));
    InstantReply = directives.Value("InstantReply", InstantReply);
}

void NDrive::TPusherOptions::Print(IOutputStream& os) const {
    os << "Host: " << Host << Endl;
    os << "Post: " << Port << Endl;
    os << "Token: " << Token << Endl;
    os << "Timeout: " << Timeout.MilliSeconds() << Endl;
    os << "InstantReply: " << ToString(InstantReply) << Endl;
}

THolder<NDrive::IPusher> NDrive::TPusherOptions::BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const {
    Y_UNUSED(tvm);
    return MakeHolder<TPusher>(*this);
}

TSet<NTvmAuth::TTvmId> NDrive::TPusherOptions::GetDestinationClientIds() const {
  return {};
}

NDrive::TPusher::TPusher(const NDrive::TPusherOptions& options)
    : OwnedClient(MakeHolder<NRTLine::TNehIndexingClient>(options.Token, options.Host, options.Port))
    , Client(*OwnedClient)
    , InstantReply(options.InstantReply)
{
}

NDrive::TPusher::TPusher(const NRTLine::TNehIndexingClient& client, const TPusherOptions& options)
    : Client(client)
    , InstantReply(options.InstantReply)
{
}

NDrive::TPusher::~TPusher() {
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TPusher::Push(const TString& imei, const IHandlerDescription& handler, TInstant deadline) {
    auto id = handler.GetId();
    auto timestamp = handler.GetTimestamp();
    bool finished = handler.IsFinished();

    NRTLine::TAction action;
    action.SetPrefix(HandlerKeyPrefix);
    NRTLine::TDocument& document = action.AddDocument();
    document.SetUrl(GetHandlerUrl(id));
    document.SetTimestamp(timestamp.Seconds());
    if (deadline) {
        document.SetDeadline(deadline);
    } else {
        document.SetDeadline(timestamp + HandlerLifetime);
    }
    if (!finished) {
        document.AddSpecialKey("active_imei", imei);
    }
    document.AddSpecialKey("imei", imei);
    document.AddProperty("Id", id);
    document.AddProperty("IMEI", imei);
    document.AddProperty("Finished", finished);
    document.AddProperty("Timestamp", timestamp.Seconds());
    document.AddProperty("Data", handler.Serialize().GetStringRobust());

    NRTLine::TGeoData& geo = document.AddGeoData();
    geo.AddCoordinate({});

    return Push(std::move(action));
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TPusher::Push(const TString& imei, const THeartbeat& heartbeat, TInstant deadline) {
    NRTLine::TAction action;
    NRTLine::TDocument& document = action.AddDocument();
    document.SetUrl(GetHeartbeatUrl(imei, heartbeat.Name));
    document.SetTimestamp(heartbeat.Timestamp.Seconds());
    document.AddSpecialKey("imei", imei);
    document.AddSpecialKey("Sensor", GetHeartbeatKey(heartbeat.Name));
    document.AddProperty("IMEI", imei);
    document.AddProperty("Host", heartbeat.Host);
    document.AddProperty("Timestamp", heartbeat.Timestamp.Seconds());
    document.AddProperty("Created", heartbeat.Created.Seconds());

    if (deadline) {
        document.SetDeadline(deadline);
    }

    NRTLine::TGeoData& geo = document.AddGeoData();
    geo.AddCoordinate({});

    return Push(std::move(action));
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TPusher::Push(const TString& imei, const TLocation& location, TInstant deadline) {
    return Push(CreateAction(imei, location, deadline));
}

NRTLine::TAction NDrive::TPusher::CreateAction(const TString& imei, const TLocation& location, TInstant deadline) {
    {
        NRTLine::TAction action;
        NRTLine::TDocument& document = action.AddDocument();
        document.SetUrl(GetLocationUrl(imei, location.Name));
        document.SetTimestamp(location.Timestamp.Seconds());
        document.AddSpecialKey("imei", imei);
        document.AddSpecialKey("Sensor", GetLocationKey(location.Name));
        document.AddProperty("IMEI", imei);
        document.AddProperty("Latitude", location.Latitude);
        document.AddProperty("Longitude", location.Longitude);
        document.AddProperty("Precision", location.Precision);
        document.AddProperty("Course", location.Course);
        document.AddProperty("Timestamp", location.Timestamp.Seconds());
        document.AddProperty("Since", location.Since.Seconds());
        document.AddProperty("Type", location.Type);

        if (deadline) {
            document.SetDeadline(deadline);
        }
        if (location.Base) {
            document.AddProperty("BaseLatitude", location.Base->Latitude);
            document.AddProperty("BaseLongitude", location.Base->Longitude);
            document.AddProperty("BaseTimestamp", location.Base->Timestamp.Seconds());
        }
        if (location.Content) {
            document.AddProperty("Content", location.Content);
        }
        if (location.Name) {
            document.AddProperty("Name", location.Name);
        }

        NRTLine::TGeoData& geo = document.AddGeoData();
        geo.AddCoordinate({ location.Longitude, location.Latitude });
        return action;
    }
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TPusher::Push(const TString& imei, const TSensor& sensor, TInstant deadline) {
    auto id = sensor.Id;
    auto timestamp = sensor.Timestamp;
    auto v = sensor.GetJsonValue().GetStringRobust();
    NRTLine::TAction action;
    NRTLine::TDocument& document = action.AddDocument();
    document.SetUrl(GetSensorUrl(imei, sensor));
    document.SetTimestamp(timestamp.Seconds());
    document.AddSpecialKey("imei", imei);
    document.AddSpecialKey<ui32>("Sensor", id);
    document.AddProperty("IMEI", imei);
    document.AddProperty<ui32>("Id", id);
    document.AddProperty<ui32>("SubId", sensor.SubId);
    document.AddProperty("Since", sensor.Since.Seconds());
    document.AddProperty("Timestamp", timestamp.Seconds());
    document.AddProperty("Value", v);

    if (deadline) {
        document.SetDeadline(deadline);
    }

    const TString& name = sensor.GetName();
    if (name) {
        document.AddProperty("Name", name);
        document.AddSpecialKey("Sensor", name);
    }

    NRTLine::TGeoData& geo = document.AddGeoData();
    geo.AddCoordinate({});

    return Push(std::move(action));
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TPusher::Remove(const TString& imei, const THeartbeat& heartbeat) {
    return Remove(GetHeartbeatUrl(imei, heartbeat.Name), heartbeat.Timestamp);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TPusher::Remove(const TString& imei, const TLocation& location) {
    return Remove(GetLocationUrl(imei, location.Name), location.Timestamp);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TPusher::Remove(const TString& imei, ui16 id, ui16 subid, TInstant timestamp) {
    return Remove(GetSensorUrl(imei, { id, subid }), timestamp);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TPusher::Push(NRTLine::TAction&& action) {
    NRTLine::TSendParams params;
    params.InstantReply = InstantReply;
    return Client.Send(action, params).Apply([](const NThreading::TFuture<NRTLine::TSendResult>& r) {
        if (r.HasException()) {
            return TPushResult{ /* Written= */ false, /* Message= */ NThreading::GetExceptionMessage(r) };
        }
        const auto& result = r.GetValue();
        if (!result.IsSucceeded()) {
            return TPushResult{ /* Written= */ false, /* Message= */ TStringBuilder() << result.GetCode() << " " << result.GetMessage() };
        }
        return TPushResult{ /* Written= */ true, /* Message= */ "" };
    });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TPusher::Remove(const TString& url, TInstant timestamp) {
    NRTLine::TAction action;
    action.SetProtobufActionType(NRTYServer::TMessage::DELETE_DOCUMENT);
    action.GetDocument().SetUrl(url);
    action.GetDocument().SetTimestamp(timestamp.Seconds());

    return Push(std::move(action));
}
