#pragma once

#include "interface.h"

#include <util/datetime/base.h>

namespace NRTLine {
    class TAction;
    class TNehIndexingClient;
}

namespace NDrive {

    class TPusherOptions : public IPusherOptions {
    public:
        void Init(const TYandexConfig::Section& section) override;
        void Print(IOutputStream& os) const override;
        virtual THolder<IPusher> BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const override;
        virtual TSet<NTvmAuth::TTvmId> GetDestinationClientIds() const override;

    private:
        static inline TFactory::TRegistrator<TPusherOptions> Registrator{"Pusher"};

    public:
        TString Host;
        ui16 Port = 80;
        TString Token;
        TDuration Timeout = TDuration::MilliSeconds(100);
        bool InstantReply = false;
    };

    class TPusher
        : public IPusher
    {
    public:
        static constexpr ui64 HandlerKeyPrefix = 1;

    public:
        static NRTLine::TAction CreateAction(const TString& imei, const TLocation& location, TInstant deadline = TInstant::Zero());

    public:
        using IPusher::Push;

    public:
        TPusher(const TPusherOptions& options);
        TPusher(const NRTLine::TNehIndexingClient& client, const TPusherOptions& options);
        ~TPusher();

        virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const IHandlerDescription& handler, TInstant deadline = TInstant::Zero()) override;
        virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const THeartbeat& heartbeat, TInstant deadline = TInstant::Zero()) override;
        virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const TLocation& location, TInstant deadline = TInstant::Zero()) override;
        virtual NThreading::TFuture<TPushResult> Push(const TString& imei, const TSensor& sensor, TInstant deadline = TInstant::Zero()) override;

        NThreading::TFuture<TPushResult> Remove(const TString& imei, const THeartbeat& heartbeat);
        NThreading::TFuture<TPushResult> Remove(const TString& imei, const TLocation& location);
        NThreading::TFuture<TPushResult> Remove(const TString& imei, ui16 id, ui16 subid, TInstant timestamp);

    private:
        NThreading::TFuture<TPushResult> Push(NRTLine::TAction&& action);
        NThreading::TFuture<TPushResult> Remove(const TString& url, TInstant timestamp);

    private:
        const THolder<NRTLine::TNehIndexingClient> OwnedClient;
        const NRTLine::TNehIndexingClient& Client;

        TDuration HandlerLifetime = TDuration::Days(1);
        TDuration FutureThreshold = TDuration::Days(1);
        bool InstantReply = false;
    };
}
