#include "pusher.h"

#include <drive/telematics/server/common/signals.h>
#include <drive/telematics/server/location/location.h>
#include <drive/telematics/server/pusher/common.h>

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/mediator/global_notifications/system_status.h>

#include <saas/library/persqueue/logger/logger.h>

#include <util/stream/file.h>
#include <util/system/env.h>

void NDrive::TSaasPusherOptions::Init(const TYandexConfig::Section& section) {
    const TYandexConfig::Directives& directives = section.GetDirectives();
    directives.GetValue("ServiceName", ServiceName);
    directives.GetValue("Host", Host);
    directives.GetValue("TvmSource", TvmSourceId);
    directives.GetValue("TvmToken", TvmToken);
    directives.GetValue("TvmFile", TvmFile);
    if (TvmFile && !TvmToken) {
        TvmToken = TIFStream(TvmFile).ReadAll();
    }
    directives.GetValue("TvmDestination", TvmDestinationId);
    directives.GetValue("ThreadsCount", ThreadsCount);
    directives.GetValue("ThreadsBlockingMode", ThreadsBlockingMode);
    directives.GetValue("MaxInFlightCount", MaxInFlightCount);
    directives.GetValue("MaxAttempts", MaxAttempts);
    directives.GetValue("NoWriteQueue", NoWriteQueue);
    ConnectionTimeout = TDuration::Seconds(directives.Value("ConnectionTimeout", ConnectionTimeout.Seconds()));
    SendTimeout = TDuration::Seconds(directives.Value("SendTimeout", SendTimeout.Seconds()));
    if (directives.contains("Codec")) {
        TString codec;
        directives.GetValue("Codec", codec);
        Codec = FromString<NPersQueueCommon::ECodec>(codec);
    }
    TYandexConfig::TSectionsMap sectionsMap = section.GetAllChildren();
    auto iter = sectionsMap.find("SearchMapInputSettings");
    AssertCorrectConfig(iter != sectionsMap.end(), "TSaasPusherOptions: no SearchMapInputSettings");
    SearchMapInputSettings.Init(iter->second);
}

void NDrive::TSaasPusherOptions::Print(IOutputStream& os) const {
    os << "ServiceName: " << ServiceName << Endl;
    os << "Host: " << Host << Endl;
    os << "TvmSource: " << TvmSourceId << Endl;
    os << "TvmDestination: " << TvmDestinationId << Endl;
    os << "ThreadsCount: " << ThreadsCount << Endl;
    os << "ThreadsBlockingMode: " << ThreadsBlockingMode << Endl;
    os << "MaxInFlightCount: " << MaxInFlightCount << Endl;
    os << "MaxAttempts: " << MaxAttempts << Endl;
    os << "ConnectionTimeout: " << ConnectionTimeout.Seconds() << Endl;
    os << "SendTimeout: " << SendTimeout.Seconds() << Endl;
    if (Codec) {
        os << "Codec : " << *Codec << Endl;
    }
    os << "<SearchMapInputSettings>" << Endl;
    SearchMapInputSettings.Print(os);
    os << "</SearchMapInputSettings>" << Endl;
}

TSet<NTvmAuth::TTvmId> NDrive::TSaasPusherOptions::GetDestinationClientIds() const {
  return {TvmDestinationId};
}

THolder<NDrive::IPusher> NDrive::TSaasPusherOptions::BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const {
    Y_UNUSED(tvm);
    return MakeHolder<TSaasPusher>(*this);
}

NDrive::TSaasPusher::TSaasPusher(const TSaasPusherOptions& options)
    : Writer(MakeHolder<NSaas::TPersQueueWriter>())
{
    TIntrusivePtr<NPersQueue::ILogger> logger = MakeIntrusive<NSaas::TPersQueueLogger<NPersQueue::ILogger>>("cerr"); // TODO
    TString topicDirectory = "saas/services/" + options.GetServiceName() + "/" + options.GetSearchMapInputSettings().Ctype + "/topics";

    NSaas::TPersQueueWriterSettings writerSettings;
    writerSettings
        .SetLogger(logger)
        .SetServiceInfo(options.GetSearchMapInputSettings(), options.GetServiceName())
        .SetTvm(options.GetTvmSourceId(), options.GetTvmDestinationId(), options.GetTvmToken())
        .SetPersQueueSettings(options.GetHost(), topicDirectory)
        .SetWriteThreadsCount(options.GetThreadsCount())
        .SetThreadsBlockingMode(options.GetThreadsBlockingMode())
        .SetNoWriteQueue(options.GetNoWriteQueue())
        .SetMaxAttempts(options.GetMaxAttempts())
        .SetMaxInFlightCount(options.GetMaxInFlightCount())
        .SetConnectionTimeout(options.GetConnectionTimeout())
        .SetSendTimeout(options.GetSendTimeout());

    if (options.HasCodec()) {
        writerSettings.SetCodec(options.GetCodecRef());
    }
    try {
        Writer->Init(writerSettings);
    } catch (const std::exception& e) {
        AbortFromCorruptedIndex("cannot create saas pusher: %s", FormatExc(e).data());
    }
}

NDrive::TSaasPusher::~TSaasPusher() {
}

NSaas::TAction NDrive::TSaasPusher::CreateAction(const TString& imei, const TLocation& location, TInstant deadline) {
    NSaas::TAction action;
    NSaas::TDocument& document = action.AddDocument();
    document.SetUrl(GetLocationUrl(imei, location.Name));
    document.SetTimestamp(location.Timestamp.Seconds());
    document.AddSpecialKey("imei", imei);
    document.AddSpecialKey("Sensor", GetLocationKey(location.Name));
    document.AddProperty("IMEI", imei);
    document.AddProperty("Latitude", location.Latitude);
    document.AddProperty("Longitude", location.Longitude);
    document.AddProperty("Precision", location.Precision);
    document.AddProperty("Course", location.Course);
    document.AddProperty("Timestamp", location.Timestamp.Seconds());
    document.AddProperty("Since", location.Since.Seconds());
    document.AddProperty("Type", location.Type);

    if (deadline) {
        document.SetDeadline(deadline);
    }
    if (location.Base) {
        document.AddProperty("BaseLatitude", location.Base->Latitude);
        document.AddProperty("BaseLongitude", location.Base->Longitude);
        document.AddProperty("BaseTimestamp", location.Base->Timestamp.Seconds());
    }
    if (location.Content) {
        document.AddProperty("Content", location.Content);
    }
    if (location.Name) {
        document.AddProperty("Name", location.Name);
    }

    NSaas::TGeoBlock& geo = document.AddGeo();
    geo.AddPointSet("coords", true).AddPoint(
        static_cast<float>(location.Longitude),
        static_cast<float>(location.Latitude)
    );
    return action;
}


NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TSaasPusher::Push(const TString& imei, const IHandlerDescription& handler, TInstant deadline) {
    auto id = handler.GetId();
    auto timestamp = handler.GetTimestamp();
    bool finished = handler.IsFinished();

    NSaas::TAction action;
    action.SetPrefix(HandlerKeyPrefix);
    NSaas::TDocument& document = action.AddDocument();
    document.SetUrl(GetHandlerUrl(id));
    document.SetTimestamp(timestamp.Seconds());
    if (deadline) {
        document.SetDeadline(deadline);
    } else {
        document.SetDeadline(timestamp + HandlerLifetime);
    }
    if (!finished) {
        document.AddSpecialKey("active_imei", imei);
    }
    document.AddSpecialKey("imei", imei);
    document.AddProperty("Id", id);
    document.AddProperty("IMEI", imei);
    document.AddProperty("Finished", finished);
    document.AddProperty("Timestamp", timestamp.Seconds());
    document.AddProperty("Data", handler.Serialize().GetStringRobust());
    return Push(action);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TSaasPusher::Push(const TString& imei, const THeartbeat& heartbeat, TInstant deadline) {
    NSaas::TAction action;
    NSaas::TDocument& document = action.AddDocument();
    document.SetUrl(GetHeartbeatUrl(imei, heartbeat.Name));
    document.SetTimestamp(heartbeat.Timestamp.Seconds());
    document.AddSpecialKey("imei", imei);
    document.AddSpecialKey("Sensor", GetHeartbeatKey(heartbeat.Name));
    document.AddProperty("IMEI", imei);
    document.AddProperty("Host", heartbeat.Host);
    document.AddProperty("Timestamp", heartbeat.Timestamp.Seconds());
    document.AddProperty("Created", heartbeat.Created.Seconds());

    if (deadline) {
        document.SetDeadline(deadline);
    }

    return Push(action);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TSaasPusher::Push(const TString& imei, const TLocation& location, TInstant deadline) {
    return Push(CreateAction(imei, location, deadline));
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TSaasPusher::Push(const TString& imei, const TSensor& sensor, TInstant deadline) {
    auto id = sensor.Id;
    auto timestamp = sensor.Timestamp;
    auto v = sensor.GetJsonValue().GetStringRobust();
    NSaas::TAction action;
    NSaas::TDocument& document = action.AddDocument();
    document.SetUrl(GetSensorUrl(imei, sensor));
    document.SetTimestamp(timestamp.Seconds());
    document.AddSpecialKey("imei", imei);
    document.AddSpecialKey<ui32>("Sensor", id);
    document.AddProperty("IMEI", imei);
    document.AddProperty<ui32>("Id", id);
    document.AddProperty<ui32>("SubId", sensor.SubId);
    document.AddProperty("Since", sensor.Since.Seconds());
    document.AddProperty("Timestamp", timestamp.Seconds());
    document.AddProperty("Value", v);

    if (deadline) {
        document.SetDeadline(deadline);
    }

    const TString& name = sensor.GetName();
    if (name) {
        document.AddProperty("Name", name);
        document.AddSpecialKey("Sensor", name);
    }

    return Push(action);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TSaasPusher::Push(const NSaas::TAction& action) {
    return Writer->Write(action).Apply([](const NSaas::TPersQueueWriter::TWriteResultFuture& r) {
        if (r.HasException()) {
            TTelematicsUnistatSignals::Get().SaasPusherFailure.Signal(1);
            TTelematicsUnistatSignals::Get().SaasPusherException.Signal(1);
            return TPushResult{ /* Written= */ false, /* Message= */ NThreading::GetExceptionMessage(r) };
        }
        const auto& result = r.GetValue();
        if (result.Written) {
            TTelematicsUnistatSignals::Get().SaasPusherSuccess.Signal(1);
        } else {
            TTelematicsUnistatSignals::Get().SaasPusherFailure.Signal(1);
            if (result.UserError) {
                TTelematicsUnistatSignals::Get().SaasPusherUserError.Signal(1);
            } else if (result.ExceededInFlight) {
                TTelematicsUnistatSignals::Get().SaasPusherExceededInFlight.Signal(1);
            } else {
                TTelematicsUnistatSignals::Get().SaasPusherUnknown.Signal(1);
            }
        }
        if (!result.Written) {
            return TPushResult{ /* Written= */ false, /* Message= */ result.ToString() };
        }
        return TPushResult{ /* Written= */ true, /* Message= */ "" };
    });
}
