#include "cache.h"

#include <drive/telematics/server/sensors/proto/cache.pb.h>

#include <drive/telematics/server/library/navtelecom.h>

#include <drive/telematics/protocol/navtelecom.h>
#include <drive/telematics/protocol/vega.h>
#include <drive/telematics/protocol/wialon.h>

#include <rtline/util/algorithm/container.h>

#include <util/generic/adaptor.h>

namespace {
    struct TCompareBySince {
        inline bool operator() (const NDrive::TSensor& left, const NDrive::TSensor& right) const {
            return left.Since < right.Since;
        }
        inline bool operator() (TInstant left, const NDrive::TSensor& right) {
            return left < TInstant(right.Since);
        }
    };
    struct TCompareByTimestamp {
        inline bool operator() (const NDrive::TSensor& left, const NDrive::TSensor& right) const {
            return left.Timestamp < right.Timestamp;
        }
    };

    using TAddCallback = std::function<void(ui32, NDrive::TSensorValue)>;
    using TConvertCallback = std::function<void(ui32 id, const NDrive::NNavTelecom::TParameter::TValue&, const TAddCallback&)>;

    template<typename TValueType>
    void DefaultConvert(ui32 id, const NDrive::NNavTelecom::TParameter::TValue& value, const TAddCallback& add) {
        if (std::holds_alternative<ui64>(value)) {
            add(id, static_cast<TValueType>(std::get<ui64>(value)));
        } else if (std::holds_alternative<i64>(value)) {
            add(id, static_cast<TValueType>(double(std::get<i64>(value))));
        } else if (std::holds_alternative<double>(value)) {
            add(id, static_cast<TValueType>(std::get<double>(value)));
        }
    }
}

ui64 NDrive::TSensorsCache::Index(ui32 id, ui32 subid) const {
    ui64 result = 0;
    result += subid;
    result <<= 32;
    result += id;
    return result;
}

ui64 NDrive::TSensorsCache::Index(TSensorId id) const {
    return Index(id.Id, id.SubId);
}

TMaybe<NDrive::TSensor> NDrive::TSensorsCache::Add(ui32 id, ui32 subid, NDrive::TSensorValue value, TInstant timestamp /*= TInstant::Zero()*/) {
    NDrive::TSensor sensor;
    sensor.Id = id;
    sensor.SubId = subid;
    sensor.Value = std::move(value);
    sensor.Timestamp = timestamp ? timestamp : Now();
    sensor.Since = sensor.Timestamp;
    return Add(std::move(sensor));
}

NDrive::TMultiSensor NDrive::TSensorsCache::Add(const NVega::TBlackboxRecords& records) {
    TMultiSensor sensors;
    for (auto&& record : records.Records) {
        auto recordSensors = NDrive::NVega::ToSensors(record);
        for (auto&& sensor : recordSensors) {
            sensors.push_back(sensor);
        }
    }
    return Add(std::move(sensors));
}

NDrive::TMultiSensor NDrive::TSensorsCache::Add(const NWialon::TShortData& data) {
    return Add(NDrive::NWialon::ToSensors(data));
}

NDrive::TMultiSensor NDrive::TSensorsCache::Add(TInstant timestamp, const NWialon::TAdditionalData& data) {
    return Add(NDrive::NWialon::ToSensors(data, timestamp));
}

NDrive::TMultiSensor NDrive::TSensorsCache::Add(const NDrive::NNavTelecom::TRecord& data) {
    return Add(NDrive::NNavTelecom::ToSensors(data));
}

NDrive::TMultiSensor NDrive::TSensorsCache::Add(const NNavTelecom::TICCIDAnswer& data) {
    return Add(NDrive::NNavTelecom::ToSensors(data));
}

NDrive::TMultiSensor NDrive::TSensorsCache::Add(const NNavTelecom::TAdditional& data) {
    return Add(NDrive::NNavTelecom::ToSensors(data));
}

TMaybe<NDrive::TSensor> NDrive::TSensorsCache::Add(NDrive::TSensor&& sensor) {
    TWriteGuard guard(Lock);
    return AddUnsafe(std::move(sensor));
}

NDrive::TMultiSensor NDrive::TSensorsCache::Add(NDrive::TMultiSensor&& sensors) {
    if (sensors.empty()) {
        return {};
    }

    NDrive::TMultiSensor result;
    result.reserve(sensors.size());
    TWriteGuard guard(Lock);
    for (auto&& sensor : sensors) {
        if (!sensor) {
            continue;
        }
        auto r = AddUnsafe(std::move(sensor));
        if (r) {
            result.push_back(std::move(*r));
        }
    }
    return result;
}

TMaybe<NDrive::TSensor> NDrive::TSensorsCache::Get(ui32 id, ui32 subid /*= 0*/, size_t index /*= 0*/) const {
    TReadGuard guard(Lock);
    auto p = Sensors.find(Index(id, subid));
    if (p == Sensors.end()) {
        return {};
    }

    auto& values = p->second;
    if (values.size() > index) {
        return values[values.size() - index - 1];
    } else {
        return {};
    }
}

TMaybe<NDrive::TSensor> NDrive::TSensorsCache::Get(ui32 id, ui32 subid, TInstant timestamp) const {
    TReadGuard guard(Lock);
    auto p = Sensors.find(Index(id, subid));
    if (p == Sensors.end()) {
        return {};
    }

    auto& values = p->second;
    Y_ASSERT(std::is_sorted(values.begin(), values.end(), TCompareByTimestamp()));
    auto bound = std::upper_bound(values.begin(), values.end(), timestamp, [] (TInstant value, const NDrive::TSensor& sensor) {
        return value < TInstant(sensor.Since);
    });
    if (bound == values.begin()) {
        return {};
    }
    bound--;
    const NDrive::TSensor& result = *bound;
    Y_ASSERT(result.Since <= timestamp);
    return result;
}

TMaybe<NDrive::TSensor> NDrive::TSensorsCache::Get(ui32 id, TInstant timestamp) const {
    return Get(id, 0, timestamp);
}

NDrive::TMultiSensor NDrive::TSensorsCache::GetRange(TSensorId id, TInstant since, TInstant until) const {
    TReadGuard guard(Lock);
    auto p = Sensors.find(Index(id.Id, id.SubId));
    if (p != Sensors.end()) {
        const auto& values = p->second;
        Y_ASSERT(std::is_sorted(values.begin(), values.end(), TCompareByTimestamp()));

        auto start = values.begin();
        for (; start != values.end(); ++start) {
            if (start->Timestamp > since) {
                break;
            }
        }
        auto finish = values.begin();
        for (; finish != values.end(); ++finish) {
            if (finish->Since >= until) {
                break;
            }
        }
        return { start, finish };
    } else {
        return {};
    }
}

TDuration NDrive::TSensorsCache::GetDuration(TSensorId id, TInstant since, TInstant until) const {
    return GetDuration(id, since, until, [](const TSensor& sensor) {
        return !sensor.IsZero();
    });
}

TDuration NDrive::TSensorsCache::GetDuration(TSensorId id, TInstant since, TInstant until, const TIsActiveFunction& func) const {
    Y_ENSURE(since <= until, since << " <= " << until);
    TReadGuard guard(Lock);
    auto p = Sensors.find(Index(id));
    if (p == Sensors.end()) {
        return {};
    }

    const auto& values = p->second;
    if (values.empty()) {
        return {};
    }
    Y_ASSERT(std::is_sorted(values.begin(), values.end(), TCompareByTimestamp()));
    auto i = LowerKey(values, since, TCompareBySince());
    if (i == values.end()) {
        i = values.begin();
    }

    TDuration result;
    TInstant start;
    for (; i != values.end(); ++i) {
        if (i->Since >= until) {
            break;
        }
        bool active = func(*i);
        if (active && start) {
            continue;
        }
        if (active && !start) {
            start = std::max<TInstant>(i->Since, since);
            continue;
        }
        if (!active && start) {
            result += (i->Since - start);
            start = {};
        }
    }
    if (start) {
        result += (until - start);
    }
    return result;
}

TInstant NDrive::TSensorsCache::GetTimestamp(TSensorId id) const {
    TReadGuard guard(Lock);
    auto p = Sensors.find(Index(id));
    if (p == Sensors.end()) {
        return {};
    }

    const auto& values = p->second;
    if (values.empty()) {
        return {};
    }

    Y_ASSERT(std::is_sorted(values.begin(), values.end(), TCompareByTimestamp()));
    return values.back().Timestamp;
}

TSet<TInstant> NDrive::TSensorsCache::GetTimestamps(TSensorId id, size_t depth) const {
    TReadGuard guard(Lock);
    auto p = Sensors.find(Index(id));
    if (p == Sensors.end()) {
        return {};
    }

    const auto& values = p->second;
    Y_ASSERT(std::is_sorted(values.begin(), values.end(), TCompareByTimestamp()));
    size_t index = depth;

    TSet<TInstant> result;
    for (auto&& value : Reversed(values)) {
        if (auto current = index--; current > 0) {
            result.insert(value.Timestamp);
        }
    }
    return result;
}

TVector<NDrive::TSensorId> NDrive::TSensorsCache::Fill(NJson::TJsonValue& report, const TMap<NDrive::TSensorId, NJson::TJsonValue>* meta) const {
    TReadGuard guard(Lock);
    TVector<NDrive::TSensorId> cachedSensors;
    for (auto&& i : Sensors) {
        const auto& sensors = i.second;
        if (sensors.empty()) {
            continue;
        }

        NJson::TJsonValue& s = report.AppendValue(NJson::JSON_MAP);
        NDrive::TSensorId sensorId(sensors.back());
        s["id"] = sensorId.Id;
        s["subid"] = sensorId.SubId;
        if (meta && meta->contains(sensorId)) {
            s["sensor_meta"] = meta->at(sensorId);
        }
        cachedSensors.push_back(sensorId);
        for (auto&& sensor : sensors) {
            NJson::TJsonValue& v = s["values"].AppendValue(NJson::JSON_MAP);
            v["value"] = sensor.GetJsonValue();
            v["timestamp"] = sensor.Timestamp.Seconds();
            v["since"] = sensor.Since.Seconds();
        }
    }
    return cachedSensors;
}

TMaybe<NDrive::TSensor> NDrive::TSensorsCache::AddUnsafe(NDrive::TSensor&& sensor) {
    Y_ASSERT(sensor.Since);
    Y_ASSERT(sensor.Timestamp);
    Y_ASSERT(sensor.Timestamp >= sensor.Since);
    Timestamp = std::max(Timestamp, sensor.Timestamp.Get());
    TMaybe<NDrive::TSensor> result;
    auto& values = Sensors[Index(sensor.Id, sensor.SubId)];
    if (values.empty()) {
        values.push_back(sensor);
        values.push_back(sensor); // TODO: get rid of this line
        result = std::move(sensor);
    } else {
        Y_ASSERT(std::is_sorted(values.begin(), values.end(), TCompareByTimestamp()));

        if (values.back().Timestamp <= sensor.Timestamp) {
            if (values.back().Value == sensor.Value) {
                values.back().Timestamp = sensor.Timestamp;
            } else {
                values.push_back(std::move(sensor));
            }
            result = values.back();
        } else if (values.front().Since >= sensor.Since) {
            if (values.front().Value == sensor.Value) {
                values.front().Since = sensor.Since;
            } else {
                values.push_front(std::move(sensor));
            }
            result = values.front();
        } else {
            // dropping for now
            result = std::move(sensor);
        }
    }
    while (values.size() > ValuesCount) {
        values.pop_front();
    }
    return result;
}

void NDrive::TSensorsCache::Serialize(NDrive::NProto::TSensorsCache& proto) const {
    TReadGuard guard(Lock);
    for (auto&& [id, sensorValues] : Sensors) {
        auto impl = proto.AddSensor();
        Y_ENSURE(impl);
        for (auto&& sensor : sensorValues) {
            auto value = impl->AddValue();
            Y_ENSURE(value);
            *value = sensor.ToProto();
        }
    }
}

bool NDrive::TSensorsCache::Deserialize(const NDrive::NProto::TSensorsCache& proto) {
    TWriteGuard guard(Lock);
    Sensors.clear();
    for (auto&& impl : proto.GetSensor()) {
        for (auto&& value : impl.GetValue()) {
            NDrive::TSensor sensor;
            if (!sensor.FromProto(value)) {
                return false;
            }
            AddUnsafe(std::move(sensor));
        }
    }
    return true;
}
