#pragma once

#include <drive/telematics/protocol/sensor.h>

#include <util/datetime/base.h>
#include <util/generic/map.h>
#include <util/generic/maybe.h>
#include <util/generic/set.h>
#include <util/system/rwlock.h>

#include <deque>

namespace NJson {
    class TJsonValue;
}

namespace NDrive::NProto {
    class TSensorsCache;
}

namespace NDrive {
    namespace NVega {
        class TBlackboxRecords;
    }

    namespace NWialon {
        class TShortData;
        class TAdditionalData;
    }

    namespace NNavTelecom {
        class TRecord;
        class TAdditional;
        class TNavTelecomProtocol;
        class TICCIDAnswer;
    }

    class TSensorsCache {
    public:
        using TIsActiveFunction = std::function<bool(const TSensor&)>;

    public:
        TSensorsCache(ui32 valuesCount)
            : ValuesCount(valuesCount)
        {
        }

        TInstant GetTimestamp() const {
            return Timestamp;
        }

        TMaybe<TSensor> Add(ui32 id, ui32 subid, TSensorValue value, TInstant timestamp = TInstant::Zero());
        TMaybe<TSensor> Add(TSensor&& sensor);
        TMultiSensor Add(const NVega::TBlackboxRecords& records);
        TMultiSensor Add(TMultiSensor&& sensors);
        TMultiSensor Add(const NWialon::TShortData& data);
        TMultiSensor Add(const NNavTelecom::TICCIDAnswer& data);
        TMultiSensor Add(const NNavTelecom::TRecord& data);
        TMultiSensor Add(const NNavTelecom::TAdditional& data);
        TMultiSensor Add(TInstant timestamp, const NWialon::TAdditionalData& data);
        TMaybe<TSensor> Get(ui32 id, ui32 subid = 0, size_t index = 0) const;
        TMaybe<TSensor> Get(ui32 id, ui32 subid, TInstant timestamp) const;
        TMaybe<TSensor> Get(ui32 id, TInstant timestamp) const;
        TMultiSensor GetRange(TSensorId id, TInstant since = TInstant::Zero(), TInstant until = TInstant::Max()) const;
        TDuration GetDuration(TSensorId id, TInstant since, TInstant until) const;
        TDuration GetDuration(TSensorId id, TInstant since, TInstant until, const TIsActiveFunction& func) const;
        TInstant GetTimestamp(TSensorId id) const;
        TSet<TInstant> GetTimestamps(TSensorId id, size_t depth) const;

        TVector<NDrive::TSensorId> Fill(NJson::TJsonValue& report, const TMap<NDrive::TSensorId, NJson::TJsonValue>* meta = nullptr) const;

        void Serialize(NDrive::NProto::TSensorsCache& proto) const;
        bool Deserialize(const NDrive::NProto::TSensorsCache& proto);

    private:
        TMaybe<TSensor> AddUnsafe(TSensor&& sensor);
        ui64 Index(ui32 id, ui32 subid) const;
        ui64 Index(TSensorId id) const;

    private:
        const ui32 ValuesCount;
        TMap<ui64, std::deque<TSensor>> Sensors;
        TInstant Timestamp;

        TRWMutex Lock;
    };
    using TSensorsCachePtr = TAtomicSharedPtr<TSensorsCache>;
}
