#include "cache.h"

#include <drive/telematics/protocol/vega.h>

#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(SensorsCacheSuite) {
    Y_UNIT_TEST(GetByTimestamp) {
        auto timestamp1 = TInstant::Seconds(1532533295);
        auto timestamp15 = timestamp1 + TDuration::Seconds(1);
        auto timestamp2 = timestamp15 + TDuration::Seconds(1);
        ui16 id = 1;
        ui16 subid = 0;

        NDrive::TSensorsCache cache(123);
        auto first = cache.Add(id, subid, TString("fake"), timestamp1);
        auto second = cache.Add(id, subid, TString("fake"), timestamp2);
        UNIT_ASSERT(first);
        UNIT_ASSERT_VALUES_EQUAL(first->Since.Get(), timestamp1);
        UNIT_ASSERT_VALUES_EQUAL(first->Timestamp.Get(), timestamp1);
        UNIT_ASSERT(second);
        UNIT_ASSERT_VALUES_EQUAL(second->Since.Get(), timestamp1);
        UNIT_ASSERT_VALUES_EQUAL(second->Timestamp.Get(), timestamp2);

        auto byIndex = cache.Get(id);
        UNIT_ASSERT(byIndex);
        UNIT_ASSERT(std::holds_alternative<TString>(byIndex->Value));

        auto byTimestamp = cache.Get(id, timestamp15);
        UNIT_ASSERT(byTimestamp);
        UNIT_ASSERT(std::holds_alternative<TString>(byTimestamp->Value));

        auto inTheFuture = cache.Get(id, TInstant::Max());
        UNIT_ASSERT(inTheFuture);

        auto inThePast = cache.Get(id, TInstant::Zero());
        UNIT_ASSERT(!inThePast);
    }

    Y_UNIT_TEST(GetDuration) {
        auto base = TInstant::Seconds(42);
        ui16 id = CAN_ENGINE_IS_ON;
        ui16 subid = 0;

        NDrive::TSensorsCache cache(123);
        UNIT_ASSERT(cache.Add(id, subid, ui64(1), base));
        UNIT_ASSERT(cache.Add(id, subid, ui64(0), base + TDuration::Seconds(5)));
        UNIT_ASSERT(cache.Add(id, subid, ui64(1), base + TDuration::Seconds(42)));
        UNIT_ASSERT(cache.Add(id, subid, ui64(1), base + TDuration::Seconds(49)));
        UNIT_ASSERT(cache.Add(id, subid, ui64(0), base + TDuration::Seconds(100)));
        UNIT_ASSERT(cache.Add(id, subid, ui64(0), base + TDuration::Seconds(142)));
        UNIT_ASSERT(cache.Add(id, subid, ui64(1), base + TDuration::Seconds(200)));

        auto checkpoint0 = base - TDuration::Seconds(10);
        auto checkpoint1 = base;
        auto checkpoint2 = base + TDuration::Seconds(10);
        auto checkpoint3 = base + TDuration::Seconds(105);
        auto checkpoint4 = base + TDuration::Seconds(150);
        auto checkpoint5 = base + TDuration::Seconds(250);
        auto checkpoints = {
            checkpoint0,
            checkpoint1,
            checkpoint2,
            checkpoint3,
            checkpoint4,
            checkpoint5,
        };
        UNIT_ASSERT(std::is_sorted(checkpoints.begin(), checkpoints.end()));
        {
            TDuration previous;
            for (auto&& checkpoint : checkpoints) {
                auto value = cache.GetDuration(id, checkpoint0, checkpoint);
                UNIT_ASSERT(value >= previous);
                previous = value;
            }
        }
        UNIT_ASSERT_VALUES_EQUAL(cache.GetDuration(id, checkpoint1, checkpoint3), TDuration::Seconds(63));
        UNIT_ASSERT_VALUES_EQUAL(cache.GetDuration(id, checkpoint1, checkpoint5), TDuration::Seconds(113));
    }
}
