#include "calculator.h"

#include "cache.h"

#include <drive/telematics/common/firmware.h>
#include <drive/telematics/protocol/settings.h>
#include <drive/telematics/protocol/vega.h>

namespace {
    const TMap<ui16, ui16> CustomSensorRemap = {
        { 2800, CAN_ODOMETER_KM },
        { 2801, CAN_SPEED },
        { 2802, CAN_ENGINE_RPM },
        { 2804, CAN_FUEL_DISTANCE_KM },
        { 2806, CAN_IGNITION },
        { 2812, CAN_ENGINE_TEMP },
        { 2870, CAN_CHECK_ENGINE },
        { 2871, CAN_AIRBAG },
        { 2874, CAN_ESP },
        { 2875, CAN_INFLATION_PRESSURE },
        { 2877, VEGA_CAN_WASHER_LIQUID },
        { 2883, CAN_WARNING },
        { 2885, CAN_CHECK_OIL },
        { 2889, CAN_CHECK_BRAKE_PADS },
        { 2903, CAN_FUEL_LEVEL_P },
        { 2975, VEGA_VIN_SENSOR },
        { 2976, VEGA_ECM_DTC_LIST_SENSOR }
    };

    bool IsCustomNumericSensor(ui16 id) {
        return id >= CUSTOM_CAN_SENSOR_FIRST_ID && id <= CUSTOM_CAN_SENSOR_LAST_ID;
    }
    bool IsCustomStringSensor(ui16 id) {
        return id >= CUSTOM_CAN_STR_SENSOR_FIRST_ID && id <= CUSTOM_CAN_STR_SENSOR_LAST_ID;
    }
    bool IsCustomSensor(ui16 id) {
        return IsCustomNumericSensor(id) || IsCustomStringSensor(id);
    }

    enum class EEngineState {
        On,
        Off
    };

    static const double SatellitesSensorThreshold(5.);
    static const double SpeedSensorThreshold(5.);

    void AddWithEngine(const NDrive::TSensor& sensor, const NDrive::TSensorsCache& cache, NDrive::TMultiSensor& result, EEngineState state) {
        auto engine = cache.Get(CAN_ENGINE_IS_ON, sensor.Timestamp);

        if (!engine) {
            return;
        }

        bool engineValue = engine->ConvertTo<bool>();
        bool engineState = state == EEngineState::On ? engineValue : !engineValue;

        if (engineState) {
            result.push_back(sensor);
            result.back().SubId = CAN_ENGINE_IS_ON;
        }
    }

    void AddSpeed(const NDrive::TSensor& sensor, const NDrive::TSensorsCache& cache, NDrive::TMultiSensor& result) {
        auto satellite = cache.Get(VEGA_SAT_USED);
        if (!satellite || satellite->Timestamp != sensor.Timestamp || satellite->ConvertTo<double>() < SatellitesSensorThreshold) {
            return;
        }
        double speed = sensor.ConvertTo<double>();
        if (speed < SpeedSensorThreshold) {
            return;
        }
        NDrive::TSensor speedByEngine = sensor;
        speedByEngine.SubId = CAN_ENGINE_IS_ON;
        result.push_back(speedByEngine);
    }
}

TVector<NDrive::TSensor> NDrive::TSensorCalculator::Derive(const NDrive::TSensor& sensor, const NDrive::TSensorsCache& cache) const {
    TVector<NDrive::TSensor> result;
    switch (sensor.Id) {
        case CAN_FUEL_LEVEL_P:
        case CAN_FUEL_DISTANCE_KM:
        case CAN_CHECK_OIL:
        case CAN_CHECK_COOLANT:
        case CAN_AIRBAG:
        case CAN_CHECK_ENGINE:
        case CAN_FAULT_LIGHTING:
        case CAN_INFLATION_PRESSURE:
        case CAN_CHECK_BRAKE_PADS:
        case CAN_ESP:
        case VEGA_CAN_WASHER_LIQUID:
        case CAN_WARNING:
            AddWithEngine(sensor, cache, result, EEngineState::On);
            break;
        case VEGA_SPEED:
            AddSpeed(sensor, cache, result);
            break;
        case VEGA_GSENSOR_AXIS_X:
        case VEGA_GSENSOR_AXIS_Y:
        case VEGA_GSENSOR_AXIS_Z:
            AddWithEngine(sensor, cache, result, EEngineState::Off);
            break;
        case VEGA_MCU_FIRMWARE_VERSION: {
            TSensor vegaMcuFirmwareVersionRevisionSensor(NVega::VegaMcuFirmwareVersionRevision);
            vegaMcuFirmwareVersionRevisionSensor.Value = static_cast<ui64>(NVega::ParseFirmwareInfo(std::get<TString>(sensor.Value)).Revision);
            vegaMcuFirmwareVersionRevisionSensor.Timestamp = sensor.Timestamp;
            vegaMcuFirmwareVersionRevisionSensor.Since = sensor.Since;
            result.push_back(vegaMcuFirmwareVersionRevisionSensor);
            break;
        }
        case NDrive::NVega::TSvrRawState::GetId(): {
            const auto& value = std::get<TBuffer>(sensor.Value);
            NDrive::NVega::TSvrRawState state;
            state.Parse({value.data(), value.size()});

            auto add = [&](auto id, auto value) {
                NDrive::TSensor s(id);
                s.Since = sensor.Since;
                s.Timestamp = sensor.Timestamp;
                s.Value = value;
                result.push_back(std::move(s));
            };

            add(NDrive::NVega::TSvrRawState::PollStateId, state.PollState);
            add(NDrive::NVega::TSvrRawState::SerialNumberId, state.SerialNumber);
            add(NDrive::NVega::TSvrRawState::EngineRpmId, state.EngineRpm);
            add(NDrive::NVega::TSvrRawState::VoltageId, state.GetVoltage());
            add(NDrive::NVega::TSvrRawState::MapId, state.GetMap());
            add(NDrive::NVega::TSvrRawState::SypId, state.GetSyp());
            add(NDrive::NVega::TSvrRawState::ReductorTemperatureId, state.GetReductorTemperature());
            add(NDrive::NVega::TSvrRawState::GasTemperatureId, state.GetGasTemperature());
            add(NDrive::NVega::TSvrRawState::PcbTemperatureId, state.GetPbcTemperature());
            add(NDrive::NVega::TSvrRawState::PetrolLoadId, state.PetrolLoad);
            add(NDrive::NVega::TSvrRawState::GasLoadId, state.GasLoad);
            add(NDrive::NVega::TSvrRawState::EngineLoadId, state.EngineLoad);
            add(NDrive::NVega::TSvrRawState::EcuStateId, state.EcuState);
            add(NDrive::NVega::TSvrRawState::TankLevelAverageId, state.TankLevelAverage);
            add(NDrive::NVega::TSvrRawState::DtcCountId, state.DtcCount);
            add(NDrive::NVega::TSvrRawState::EcuResetCounterId, state.EcuResetCounter);
            add(NDrive::NVega::TSvrRawState::EmergencyStartCounterId, state.EmergencyStartCounter);
            add(NDrive::NVega::TSvrRawState::MaxPcbTemperatureId, state.GetMaxPcbTemperature());
            add(NDrive::NVega::TSvrRawState::WorkingOnPetrolId, state.WorkingOnPetrol);
            add(NDrive::NVega::TSvrRawState::WorkingOnGasId, state.WorkingOnGas);
            add(NDrive::NVega::TSvrRawState::DistanceLeftId, state.DistanceLeft);
            break;
        }
        default:
            break;
    }
    if (IsCustomSensor(sensor.Id)) {
        auto remappedIt = CustomSensorRemap.find(sensor.Id);
        if (remappedIt != CustomSensorRemap.end()) {
            auto remapped = sensor;
            remapped.Id = remappedIt->second;
            if (!IsCustomSensor(remapped.Id)) {
                auto derived = Derive(remapped, cache);
                result.insert(result.end(), derived.begin(), derived.end());
            }
            result.push_back(remapped);
            switch (remapped.Id) {
                case CAN_ENGINE_RPM:
                case CAN_IGNITION: {
                    auto complementarySensorId = CAN_ENGINE_RPM ^ CAN_IGNITION ^ remapped.Id;
                    auto complementarySensor = cache.Get(complementarySensorId);
                    TSensor freshSensor;
                    if (!complementarySensor) {
                        freshSensor = remapped;
                    } else if (complementarySensor->Timestamp > remapped.Timestamp) {
                        freshSensor =  *complementarySensor;
                    } else if (complementarySensor->Timestamp < remapped.Timestamp) {
                        freshSensor = remapped;
                    } else {
                        freshSensor = (remapped.ConvertTo<double>() > 0 ? remapped : *complementarySensor);
                    }
                    bool isActive = (freshSensor.ConvertTo<double>() > 0);
                    TSensor engineIsOn = freshSensor;
                    engineIsOn.Id = CAN_ENGINE_IS_ON;
                    engineIsOn.SubId = 0;
                    engineIsOn.Value = static_cast<ui64>(isActive);
                    result.push_back(engineIsOn);
                    break;
                }
                default:
                    break;
            }
        }
    }
    return result;
}
