#include "validation.h"

#include "cache.h"

#include <drive/telematics/protocol/vega.h>

std::pair<NDrive::THeartbeat, NDrive::EDataValidationStatus> NDrive::Validate(NDrive::THeartbeat&& heartbeat, const NDrive::TSensorsCache& cache, const NDrive::TDataValidationOptions& options) {
    Y_UNUSED(cache);
    if (heartbeat.Timestamp > Now() + options.FutureThreshold) {
        return { std::move(heartbeat), NDrive::EDataValidationStatus::Future };
    }
    return { std::move(heartbeat), NDrive::EDataValidationStatus::Ok };
}

std::pair<NDrive::TLocation, NDrive::EDataValidationStatus> NDrive::Validate(NDrive::TLocation&& location, const NDrive::TSensorsCache& cache, const NDrive::TDataValidationOptions& options) {
    Y_UNUSED(cache);
    if (!location) {
        return { std::move(location), NDrive::EDataValidationStatus::ZeroValue };
    }
    if (location.Type == NDrive::TLocation::Unknown) {
        return { std::move(location), NDrive::EDataValidationStatus::UnexpectedValueType };
    }
    if (location.Timestamp > Now() + options.FutureThreshold) {
        return { std::move(location), NDrive::EDataValidationStatus::Future };
    }
    auto status = NDrive::EDataValidationStatus::Ok;
    if (std::abs(location.Latitude) > 90) {
        while (location.Latitude > 90) {
            location.Latitude -= 90;
        }
        while (location.Latitude < -90) {
            location.Latitude += 90;
        }
        status = NDrive::EDataValidationStatus::OverflownValue;
    }
    if (std::abs(location.Longitude) > 180) {
        while (location.Longitude > 180) {
            location.Longitude -= 180;
        }
        while (location.Longitude < -180) {
            location.Longitude += 180;
        }
        status = NDrive::EDataValidationStatus::OverflownValue;
    }
    return { std::move(location), status };
}

std::pair<NDrive::TSensor, NDrive::EDataValidationStatus> NDrive::Validate(NDrive::TSensor&& sensor, const NDrive::TSensorsCache& cache, const NDrive::TDataValidationOptions& options) {
    TSensorValue& value = sensor.Value;
    NVega::TValueTraits traits = NVega::GetValueTraits(sensor.Id);
    if ((traits & NVega::EValueTrait::vtSkipZero) && sensor.IsZero() && options.FilterSkipZeroEnabled) {
        return { std::move(sensor), NDrive::EDataValidationStatus::ZeroValue };
    }
    if ((traits & NVega::EValueTrait::vtSkipZeroIdle) && sensor.IsZero() && options.FilterSkipZeroEnabled) {
        auto engine = cache.Get(CAN_ENGINE_IS_ON, sensor.Timestamp);
        if (!engine || !engine->TryConvertTo<bool>().GetOrElse(false)) {
            return { std::move(sensor), NDrive::EDataValidationStatus::ZeroValue };
        }
    }
    if ((traits & NVega::EValueTrait::vtFilterOdometer) && options.FilterOdometerEnabled) {
        if (std::holds_alternative<double>(value)) {
            auto v = std::get<double>(value);
            if (v > options.FilterOdometerThreshold) {
                return { std::move(sensor), NDrive::EDataValidationStatus::OverflownValue };
            }

            double maxValue = 0;
            auto previous = cache.GetRange(sensor, TInstant::Zero(), sensor.Since);
            auto previousValues = TVector<double>();
            previousValues.reserve(previous.size());
            for (auto&& i : previous) {
                auto optionalValue = i.TryConvertTo<double>();
                if (optionalValue && *optionalValue > 0) {
                    maxValue = std::max(maxValue, *optionalValue);
                    previousValues.push_back(*optionalValue);
                }
            }
            if (!previousValues.empty() && maxValue * options.FilterOdometerDeltaThreshold >= options.FilterOdometerMinimalStep) {
                std::sort(previousValues.begin(), previousValues.end());
                auto median = previousValues[(previousValues.size() + 1) / 2 - 1];
                auto delta = std::abs((v - median) / std::min(v, median));
                if (delta > options.FilterOdometerDeltaThreshold) {
                    return { std::move(sensor), NDrive::EDataValidationStatus::OutlierValue };
                }
            }
        }
    }
    if (sensor.Timestamp > Now() + options.FutureThreshold) {
        return { std::move(sensor), NDrive::EDataValidationStatus::Future };
    }
    if (traits & NVega::EValueTrait::vtPercentValue) {
        if (std::holds_alternative<ui64>(value)) {
            auto v = std::get<ui64>(value);
            ui64 max = 100;
            if (v > max) {
                if ((traits & NVega::EValueTrait::vtSkipInvalidPercent) && (v == NDrive::NVega::InvalidPercent)) {
                    return { std::move(sensor), NDrive::EDataValidationStatus::Invalid };
                }
                value = max;
            }
        } else if (std::holds_alternative<double>(value)) {
            auto v = std::get<double>(value);
            double max = 100;
            double min = 0;
            if (v > max || v < min) {
                if ((traits & NVega::EValueTrait::vtSkipInvalidPercent) && (v == NDrive::NVega::InvalidPercent)) {
                    return { std::move(sensor), NDrive::EDataValidationStatus::Invalid };
                }
                value = ClampVal(v, min, max);
            }
        } else {
            return { std::move(sensor), NDrive::EDataValidationStatus::UnexpectedValueType };
        }
    }
    return { std::move(sensor), NDrive::EDataValidationStatus::Ok };
}
