#include "cache.h"
#include "validation.h"

#include <drive/telematics/protocol/vega.h>

#include <library/cpp/testing/unittest/registar.h>

namespace {
    void PushAndValidateMileage(NDrive::TSensorsCache& cache, double value, TInstant timestamp, NDrive::EDataValidationStatus expected = NDrive::EDataValidationStatus::Ok) {
        NDrive::TSensor sensor(CAN_ODOMETER_KM);
        sensor.Value = value;
        sensor.Timestamp = timestamp;
        sensor.Since = timestamp;
        NDrive::TSensor copy = sensor;

        cache.Add(std::move(sensor));
        auto&&[result, status] = NDrive::Validate(std::move(copy), cache);
        UNIT_ASSERT_VALUES_EQUAL(status, expected);
    }
}

Y_UNIT_TEST_SUITE(SensorsValidationSuite) {
    Y_UNIT_TEST(Mileage) {
        TInstant timestamp = TInstant::Seconds(1230000);
        NDrive::TSensorsCache cache(123);
        cache.Add(CAN_ODOMETER_KM, 0, double(10042), timestamp);

        NDrive::TSensor sensor(CAN_ODOMETER_KM);
        sensor.Value = double(100000);
        {
            auto copy = sensor;
            copy.Timestamp = timestamp + TDuration::Seconds(20);
            copy.Since = copy.Timestamp;
            auto copy2 = copy;
            cache.Add(std::move(copy));

            auto&& [result, status] = NDrive::Validate(std::move(copy2), cache);
            UNIT_ASSERT(status != NDrive::EDataValidationStatus::Ok);
        }
        {
            auto copy = sensor;
            copy.Timestamp = timestamp + TDuration::Seconds(40);
            copy.Since = copy.Timestamp;
            auto copy2 = copy;
            cache.Add(std::move(copy));

            auto&& [result, status] = NDrive::Validate(std::move(copy2), cache);
            UNIT_ASSERT(status != NDrive::EDataValidationStatus::Ok);
        }
    }

    Y_UNIT_TEST(FuelLevel) {
        TInstant timestamp = TInstant::Seconds(1);
        NDrive::TSensorsCache cache(10);
        cache.Add(CAN_FUEL_LEVEL_P, 0, ui8(70), timestamp);

        {
            NDrive::TSensor sensor(CAN_FUEL_LEVEL_P);
            sensor.Value = ui64(80);
            auto&& [result, status] = NDrive::Validate(std::move(sensor), cache);
            UNIT_ASSERT_VALUES_EQUAL(status, NDrive::EDataValidationStatus::Ok);
        }

        {
            NDrive::TSensor sensor(CAN_FUEL_LEVEL_P);
            sensor.Value = ui64(NDrive::NVega::InvalidPercent);
            auto&& [result, status] = NDrive::Validate(std::move(sensor), cache);
            UNIT_ASSERT_VALUES_EQUAL(status, NDrive::EDataValidationStatus::Invalid);
        }

        {
            NDrive::TSensor sensor(CAN_FUEL_LEVEL_P);
            sensor.Value = double(NDrive::NVega::InvalidPercent);
            auto&& [result, status] = NDrive::Validate(std::move(sensor), cache);
            UNIT_ASSERT_VALUES_EQUAL(status, NDrive::EDataValidationStatus::Invalid);
        }

        {
            auto v = ui8(104);
            NDrive::TSensor sensor(CAN_FUEL_LEVEL_P);
            sensor.Value = v;
            auto&& [result, status] = NDrive::Validate(std::move(sensor), cache);
            UNIT_ASSERT_VALUES_EQUAL(status, NDrive::EDataValidationStatus::Ok);
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(sensor.Value), 100);
        }

        {
            auto v = double(104.5);
            NDrive::TSensor sensor(CAN_FUEL_LEVEL_P);
            sensor.Value = v;
            auto&& [result, status] = NDrive::Validate(std::move(sensor), cache);
            UNIT_ASSERT_VALUES_EQUAL(status, NDrive::EDataValidationStatus::Ok);
            UNIT_ASSERT_VALUES_EQUAL(std::get<double>(sensor.Value), 100);
        }
    }

    Y_UNIT_TEST(DRIVESUP_5828) {
        NDrive::TSensorsCache cache(8);
        PushAndValidateMileage(cache, 47107, TInstant::Seconds(1571186158));
        PushAndValidateMileage(cache, 47107, TInstant::Seconds(1571186804));
        PushAndValidateMileage(cache, 65534, TInstant::Seconds(1571186824), NDrive::EDataValidationStatus::OutlierValue);
        PushAndValidateMileage(cache, 47107, TInstant::Seconds(1571186845));
        PushAndValidateMileage(cache, 47107, TInstant::Seconds(1571187189));
        PushAndValidateMileage(cache, 47108, TInstant::Seconds(1571187209));
        PushAndValidateMileage(cache, 47108, TInstant::Seconds(1571187271));
        PushAndValidateMileage(cache, 47109, TInstant::Seconds(1571187291));
        PushAndValidateMileage(cache, 47109, TInstant::Seconds(1571187514));
        PushAndValidateMileage(cache, 47110, TInstant::Seconds(1571187535));
        PushAndValidateMileage(cache, 47110, TInstant::Seconds(1571187778));
        PushAndValidateMileage(cache, 65534, TInstant::Seconds(1571187801), NDrive::EDataValidationStatus::OutlierValue);
        PushAndValidateMileage(cache, 47110, TInstant::Seconds(1571187819));
        PushAndValidateMileage(cache, 47110, TInstant::Seconds(1571188989));
        PushAndValidateMileage(cache, 65534, TInstant::Seconds(1571189009), NDrive::EDataValidationStatus::OutlierValue);
    }
}
